<?php
/**
 * Plugin Name: Selektable
 * Plugin URI: https://selektable.com
 * Description: Integrate the Selektable widget for virtual try-on and room visualization on your WordPress site.
 * Version: 1.5.0
 * Author: Selektable
 * Author URI: https://selektable.com/about
 * Text Domain: selektable
 * Domain Path: /languages
 * Requires at least: 6.9
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('SELEKTABLE_VERSION', '1.5.0');
define('SELEKTABLE_DB_VERSION', '1.1.0');
define('SELEKTABLE_PLUGIN_FILE', __FILE__);
define('SELEKTABLE_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SELEKTABLE_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Check if WooCommerce is active
 */
function selektable_is_woocommerce_active() {
    return class_exists('WooCommerce');
}

/**
 * Declare HPOS compatibility (only when WooCommerce is active)
 */
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

/**
 * Initialize the plugin
 */
function selektable_init() {
    // Run migration on plugins_loaded (for upgrades without activation)
    selektable_maybe_migrate();

    // Core classes - always loaded
    require_once SELEKTABLE_PLUGIN_DIR . 'includes/class-selektable-admin.php';
    require_once SELEKTABLE_PLUGIN_DIR . 'includes/class-selektable-frontend.php';

    new Selektable_Admin();
    new Selektable_Frontend();

    // WooCommerce-specific classes - only when WC is active
    if (selektable_is_woocommerce_active()) {
        require_once SELEKTABLE_PLUGIN_DIR . 'includes/class-selektable-product.php';
        require_once SELEKTABLE_PLUGIN_DIR . 'includes/class-selektable-cart.php';
        require_once SELEKTABLE_PLUGIN_DIR . 'includes/class-selektable-order.php';

        new Selektable_Product();
        new Selektable_Cart();
        new Selektable_Order();
    }
}
add_action('plugins_loaded', 'selektable_init');

/**
 * Activation hook
 */
function selektable_activate() {
    // Set default options if they don't exist
    if (get_option('selektable_app_url') === false) {
        add_option('selektable_app_url', 'https://app.selektable.com');
    }
    if (get_option('selektable_store_id') === false) {
        add_option('selektable_store_id', '');
    }
    if (get_option('selektable_integrations') === false) {
        add_option('selektable_integrations', []);
    }

    // Run migration if needed
    selektable_maybe_migrate();
}
register_activation_hook(__FILE__, 'selektable_activate');

/**
 * Check and run migration from old settings format
 */
function selektable_maybe_migrate() {
    $db_version = get_option('selektable_db_version', '1.0.0');

    // Migration from 1.0.0 to 1.1.0 (old flat settings to integrations array)
    if (version_compare($db_version, '1.1.0', '<')) {
        selektable_migrate_to_integrations();
        update_option('selektable_db_version', '1.1.0');
    }
}

/**
 * Migrate old flat settings to the new integrations format
 */
function selektable_migrate_to_integrations() {
    // Check if migration is needed (old widget_id exists)
    $old_widget_id = get_option('selektable_widget_id', '');

    if (empty($old_widget_id)) {
        return; // Nothing to migrate
    }

    // Check if integrations already exist (avoid duplicate migration)
    $integrations = get_option('selektable_integrations', []);
    if (!empty($integrations)) {
        return;
    }

    // Read old settings
    $old_placement = get_option('selektable_button_placement', 'after_add_to_cart');
    $old_button_text = get_option('selektable_button_text', 'Try it on');
    $old_button_class = get_option('selektable_button_class', 'button selektable-try-on');
    $old_enable_all = get_option('selektable_enable_all', 'no');
    $old_categories = get_option('selektable_enabled_categories', []);
    $old_tags = get_option('selektable_enabled_tags', []);

    // Determine activation mode
    $activation_mode = 'all';
    $categories = [];
    $tags = [];

    if ($old_enable_all !== 'yes') {
        if (!empty($old_categories)) {
            $activation_mode = 'categories';
            $categories = array_map('intval', (array) $old_categories);
        } elseif (!empty($old_tags)) {
            $activation_mode = 'tags';
            $tags = array_map('intval', (array) $old_tags);
        }
    }

    // Handle shortcode_only placement
    if ($old_placement === 'shortcode_only') {
        // Create a shortcode integration
        $integrations[] = [
            'id' => 'int_' . wp_generate_password(8, false),
            'type' => 'shortcode',
            'widget_id' => $old_widget_id,
            'button_text' => $old_button_text,
            'button_class' => $old_button_class,
            'created_at' => time(),
            'updated_at' => time(),
        ];
    } else {
        // Create a WC Product Page integration
        $integrations[] = [
            'id' => 'int_' . wp_generate_password(8, false),
            'type' => 'wc_product_page',
            'widget_id' => $old_widget_id,
            'button_text' => $old_button_text,
            'button_class' => $old_button_class,
            'placement' => $old_placement,
            'activation_mode' => $activation_mode,
            'categories' => $categories,
            'tags' => $tags,
            'created_at' => time(),
            'updated_at' => time(),
        ];
    }

    // Save the new integrations
    update_option('selektable_integrations', $integrations);

    // Note: We keep old options for rollback safety - do not delete them
}

/**
 * Add settings link to plugins page
 */
function selektable_plugin_action_links($links) {
    $settings_link = sprintf(
        '<a href="%s">%s</a>',
        admin_url('options-general.php?page=selektable'),
        __('Settings', 'selektable')
    );
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'selektable_plugin_action_links');

/**
 * Register the shortcode
 *
 * Usage: [selektable_button widget_id="wgt_xxx" text="Try it on" class="my-class"]
 * If no widget_id provided, uses the first shortcode-type integration.
 *
 * Works with or without WooCommerce. When on a WC product page, product data
 * is passed to the widget. Otherwise, the button opens the widget without product context.
 */
function selektable_button_shortcode($atts = []) {
    $atts = shortcode_atts([
        'widget_id' => '',
        'text' => '',
        'class' => '',
    ], $atts, 'selektable_button');

    // Get integrations
    $integrations = get_option('selektable_integrations', []);

    // Find the integration to use
    $integration = null;
    $widget_id = $atts['widget_id'];

    if (!empty($widget_id)) {
        // Find integration by widget_id
        foreach ($integrations as $int) {
            if ($int['widget_id'] === $widget_id) {
                $integration = $int;
                break;
            }
        }
    } else {
        // Use first shortcode-type integration
        foreach ($integrations as $int) {
            if ($int['type'] === 'shortcode') {
                $integration = $int;
                $widget_id = $int['widget_id'];
                break;
            }
        }
    }

    // If still no widget_id, can't render
    if (empty($widget_id)) {
        return '';
    }

    // Determine button text and class
    $button_text = $atts['text'];
    $button_class = $atts['class'];

    if (empty($button_text) && $integration) {
        $button_text = $integration['button_text'] ?? 'Try it on';
    }
    if (empty($button_class) && $integration) {
        $button_class = $integration['button_class'] ?? 'button selektable-try-on';
    }

    // Fallback defaults
    if (empty($button_text)) {
        $button_text = 'Try it on';
    }
    if (empty($button_class)) {
        $button_class = 'button selektable-try-on';
    }

    $frontend = new Selektable_Frontend();

    // If WooCommerce is active and we're on a product page, pass product data
    if (selektable_is_woocommerce_active()) {
        global $product;
        if ($product && is_a($product, 'WC_Product')) {
            ob_start();
            $frontend->render_button_with_widget_id($product, $widget_id, $button_text, $button_class);
            return ob_get_clean();
        }
    }

    // Non-WooCommerce context: render a simple button without product data
    ob_start();
    $frontend->render_simple_button($widget_id, $button_text, $button_class);
    return ob_get_clean();
}
add_shortcode('selektable_button', 'selektable_button_shortcode');

/**
 * Helper function to get all integrations
 */
function selektable_get_integrations() {
    return get_option('selektable_integrations', []);
}

/**
 * Helper function to get an integration by ID
 */
function selektable_get_integration($integration_id) {
    $integrations = selektable_get_integrations();
    foreach ($integrations as $integration) {
        if ($integration['id'] === $integration_id) {
            return $integration;
        }
    }
    return null;
}
