<?php
/**
 * Selektable Product Meta Box
 *
 * Adds per-product Selektable configuration options.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Selektable_Product {

    public function __construct() {
        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('woocommerce_process_product_meta', [$this, 'save_meta']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
    }

    /**
     * Add the meta box to product edit page
     */
    public function add_meta_box() {
        add_meta_box(
            'selektable_product_options',
            __('Selektable Widget', 'selektable'),
            [$this, 'render_meta_box'],
            'product',
            'side',
            'default'
        );
    }

    /**
     * Render the meta box content
     */
    public function render_meta_box($post) {
        wp_nonce_field('selektable_product_meta', 'selektable_product_nonce');

        $status = get_post_meta($post->ID, '_selektable_status', true) ?: 'default';
        $override_image_id = get_post_meta($post->ID, '_selektable_override_image_id', true);
        $override_image_url = $override_image_id ? wp_get_attachment_image_url($override_image_id, 'thumbnail') : '';

        ?>
        <div class="selektable-product-options">
            <p>
                <label for="selektable_status"><strong><?php esc_html_e('Widget Status', 'selektable'); ?></strong></label>
            </p>
            <p>
                <select id="selektable_status" name="selektable_status" class="widefat">
                    <option value="default" <?php selected($status, 'default'); ?>>
                        <?php esc_html_e('Use global settings', 'selektable'); ?>
                    </option>
                    <option value="enabled" <?php selected($status, 'enabled'); ?>>
                        <?php esc_html_e('Enable for this product', 'selektable'); ?>
                    </option>
                    <option value="disabled" <?php selected($status, 'disabled'); ?>>
                        <?php esc_html_e('Disable for this product', 'selektable'); ?>
                    </option>
                </select>
            </p>
            <p class="description">
                <?php esc_html_e('Override global activation rules for this product.', 'selektable'); ?>
            </p>

            <hr style="margin: 15px 0;">

            <p>
                <label for="selektable_override_image"><strong><?php esc_html_e('Override Generation Image', 'selektable'); ?></strong></label>
            </p>
            <p class="description" style="margin-bottom: 10px;">
                <?php esc_html_e('Set a different image to use for AI generation instead of the product featured image.', 'selektable'); ?>
            </p>

            <div class="selektable-image-preview" style="margin-bottom: 10px;">
                <?php if ($override_image_url): ?>
                    <img src="<?php echo esc_url($override_image_url); ?>" style="max-width: 100%; height: auto; display: block;">
                <?php endif; ?>
            </div>

            <input type="hidden" id="selektable_override_image_id" name="selektable_override_image_id" value="<?php echo esc_attr($override_image_id); ?>">

            <p>
                <button type="button" class="button selektable-select-image" style="<?php echo $override_image_id ? 'display:none;' : ''; ?>">
                    <?php esc_html_e('Select Image', 'selektable'); ?>
                </button>
                <button type="button" class="button selektable-remove-image" style="<?php echo !$override_image_id ? 'display:none;' : ''; ?>">
                    <?php esc_html_e('Remove Image', 'selektable'); ?>
                </button>
            </p>
        </div>
        <?php
    }

    /**
     * Save meta box data
     */
    public function save_meta($post_id) {
        // Check nonce
        if (!isset($_POST['selektable_product_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['selektable_product_nonce'])), 'selektable_product_meta')) {
            return;
        }

        // Check autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Save status
        if (isset($_POST['selektable_status'])) {
            $status = sanitize_text_field(wp_unslash($_POST['selektable_status']));
            if (in_array($status, ['default', 'enabled', 'disabled'], true)) {
                update_post_meta($post_id, '_selektable_status', $status);
            }
        }

        // Save override image ID
        if (isset($_POST['selektable_override_image_id'])) {
            $image_id = absint($_POST['selektable_override_image_id']);
            if ($image_id) {
                update_post_meta($post_id, '_selektable_override_image_id', $image_id);
            } else {
                delete_post_meta($post_id, '_selektable_override_image_id');
            }
        }
    }

    /**
     * Enqueue scripts for product edit page
     */
    public function enqueue_scripts($hook) {
        global $post;

        if ($hook !== 'post.php' && $hook !== 'post-new.php') {
            return;
        }

        if (!$post || $post->post_type !== 'product') {
            return;
        }

        wp_enqueue_style(
            'selektable-admin',
            SELEKTABLE_PLUGIN_URL . 'assets/css/admin.css',
            [],
            SELEKTABLE_VERSION
        );

        wp_enqueue_script(
            'selektable-product',
            SELEKTABLE_PLUGIN_URL . 'assets/js/product.js',
            ['jquery'],
            SELEKTABLE_VERSION,
            true
        );

        // Get product images for the picker
        $product = wc_get_product($post->ID);
        $product_images = [];

        if ($product) {
            // Featured image
            $featured_id = $product->get_image_id();
            if ($featured_id) {
                $product_images[] = [
                    'id' => (int) $featured_id,
                    'url' => wp_get_attachment_image_url($featured_id, 'thumbnail'),
                    'label' => __('Featured Image', 'selektable'),
                ];
            }

            // Gallery images
            $gallery_ids = $product->get_gallery_image_ids();
            foreach ($gallery_ids as $index => $gallery_id) {
                $product_images[] = [
                    'id' => (int) $gallery_id,
                    'url' => wp_get_attachment_image_url($gallery_id, 'thumbnail'),
                    /* translators: %d: Gallery image number */
                    'label' => sprintf(__('Gallery Image %d', 'selektable'), $index + 1),
                ];
            }
        }

        wp_localize_script('selektable-product', 'selektableProduct', [
            'images' => $product_images,
            'i18n' => [
                'selectImage' => __('Select Override Image', 'selektable'),
                'noImages' => __('No product images available. Please add a featured image or gallery images first.', 'selektable'),
                'cancel' => __('Cancel', 'selektable'),
            ],
        ]);
    }
}
