<?php
/**
 * Selektable Order Integration
 *
 * Handles order tracking and attribution for Selektable analytics.
 * Stores visitor/session IDs as order meta for conversion tracking.
 *
 * Supports Classic Checkout, Block Checkout (Store API), and
 * orders created via admin or REST API.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Selektable_Order {

    /**
     * Meta keys for order tracking
     */
    const META_VISITOR_ID = '_selektable_visitor_id';
    const META_SESSION_ID = '_selektable_session_id';

    public function __construct() {
        // Classic Checkout: fires during WC_Checkout::create_order() before save
        add_action('woocommerce_checkout_create_order', [$this, 'add_tracking_to_order'], 10, 2);

        // Block Checkout (Store API, WC 8.3+): fires after order is processed
        add_action('woocommerce_store_api_checkout_order_processed', [$this, 'add_tracking_to_order_block_checkout'], 10, 1);

        // Fallback for orders created via other methods (admin, API, etc.)
        add_action('woocommerce_new_order', [$this, 'maybe_add_tracking_to_new_order'], 10, 2);
    }

    /**
     * Find tracking data from available sources.
     *
     * Checks in order of reliability:
     * 1. Cart item data (most robust — survives session changes)
     * 2. WC Session (original approach)
     *
     * @return array{visitor_id: string|null, session_id: string|null}
     */
    private function find_tracking_data() {
        // Source 1: Cart item data (set during add-to-cart AJAX handler)
        if (function_exists('WC') && WC()->cart) {
            foreach (WC()->cart->get_cart() as $cart_item) {
                if (!empty($cart_item['selektable_visitor_id'])) {
                    return [
                        'visitor_id' => sanitize_text_field($cart_item['selektable_visitor_id']),
                        'session_id' => !empty($cart_item['selektable_session_id'])
                            ? sanitize_text_field($cart_item['selektable_session_id'])
                            : null,
                    ];
                }
            }
        }

        // Source 2: WC Session
        if (function_exists('WC') && WC()->session) {
            $visitor_id = WC()->session->get('selektable_visitor_id');
            if ($visitor_id) {
                return [
                    'visitor_id' => sanitize_text_field($visitor_id),
                    'session_id' => WC()->session->get('selektable_session_id')
                        ? sanitize_text_field(WC()->session->get('selektable_session_id'))
                        : null,
                ];
            }
        }

        return ['visitor_id' => null, 'session_id' => null];
    }

    /**
     * Apply tracking data to an order object (in-memory only, does not save).
     *
     * @param WC_Order $order
     * @param array{visitor_id: string|null, session_id: string|null} $tracking
     * @return bool Whether tracking data was applied
     */
    private function apply_tracking_to_order($order, $tracking) {
        if (empty($tracking['visitor_id'])) {
            return false;
        }

        $order->update_meta_data(self::META_VISITOR_ID, $tracking['visitor_id']);

        if (!empty($tracking['session_id'])) {
            $order->update_meta_data(self::META_SESSION_ID, $tracking['session_id']);
        }

        return true;
    }

    /**
     * Classic Checkout: add tracking data during order creation.
     *
     * Hook: woocommerce_checkout_create_order (fires before $order->save())
     *
     * @param WC_Order $order The order object
     * @param array $data Checkout form data
     */
    public function add_tracking_to_order($order, $data) {
        $tracking = $this->find_tracking_data();
        $this->apply_tracking_to_order($order, $tracking);
        // No $order->save() needed — WC saves after this hook returns
    }

    /**
     * Block Checkout: add tracking data after Store API processes the order.
     *
     * Hook: woocommerce_store_api_checkout_order_processed (fires after save)
     *
     * @param WC_Order $order
     */
    public function add_tracking_to_order_block_checkout($order) {
        // Skip if already set (e.g., by woocommerce_new_order during draft creation)
        if ($order->get_meta(self::META_VISITOR_ID)) {
            return;
        }

        $tracking = $this->find_tracking_data();
        if ($this->apply_tracking_to_order($order, $tracking)) {
            $order->save();
        }
    }

    /**
     * Fallback: add tracking to orders created via other methods (admin, API).
     *
     * Hook: woocommerce_new_order (fires after save, inside data store create())
     *
     * @param int $order_id Order ID
     * @param WC_Order $order Order object (WC 7.1+)
     */
    public function maybe_add_tracking_to_new_order($order_id, $order = null) {
        if (!$order) {
            $order = wc_get_order($order_id);
        }

        if (!$order) {
            return;
        }

        // Skip if already has tracking data (set by Classic Checkout hook)
        if ($order->get_meta(self::META_VISITOR_ID)) {
            return;
        }

        $tracking = $this->find_tracking_data();
        if ($this->apply_tracking_to_order($order, $tracking)) {
            $order->save();
        }
    }

    /**
     * Get tracking data from an order.
     *
     * @param WC_Order|int $order Order object or ID
     * @return array{visitor_id: string|null, session_id: string|null}
     */
    public static function get_tracking_data($order) {
        if (is_numeric($order)) {
            $order = wc_get_order($order);
        }

        if (!$order) {
            return [
                'visitor_id' => null,
                'session_id' => null,
            ];
        }

        return [
            'visitor_id' => $order->get_meta(self::META_VISITOR_ID) ?: null,
            'session_id' => $order->get_meta(self::META_SESSION_ID) ?: null,
        ];
    }
}
