<?php
/**
 * Selektable Frontend
 *
 * Handles widget button rendering and embed script loading on the frontend.
 * Supports multiple integrations with dynamic hook registration.
 * Works with or without WooCommerce.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Selektable_Frontend {

    private $app_url;
    private $integrations = [];
    private $registered_placements = [];
    private $is_dev = false;

    public function __construct() {
        $this->app_url = get_option('selektable_app_url', 'https://app.selektable.com');
        $this->is_dev = $this->is_local_environment();
        $this->integrations = get_option('selektable_integrations', []);

        if (empty($this->integrations)) {
            return;
        }

        // Register WC product page button placement hooks (only when WooCommerce is active)
        if (selektable_is_woocommerce_active()) {
            $this->register_placement_hooks();
        }

        // Enqueue scripts (embed.js always, WC-specific scripts conditionally)
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);

        // Add data-store-id attribute to the embed script tag
        add_filter('script_loader_tag', [$this, 'add_store_id_to_embed_script'], 10, 3);
    }

    /**
     * Check if site is running on localhost or .local domain (dev environment)
     */
    private function is_local_environment() {
        $site_url = get_site_url();
        $host = wp_parse_url($site_url, PHP_URL_HOST);

        if (!$host) {
            return false;
        }

        if ($host === 'localhost' || $host === '127.0.0.1') {
            return true;
        }

        if (str_ends_with($host, '.local')) {
            return true;
        }

        // Also check if app_url points to localhost (dev Selektable app)
        $app_host = wp_parse_url($this->app_url, PHP_URL_HOST);
        if ($app_host === 'localhost' || $app_host === '127.0.0.1') {
            return true;
        }

        return false;
    }

    /**
     * Register hooks for each unique placement (WooCommerce only)
     */
    private function register_placement_hooks() {
        $placements = [];

        // Group integrations by placement
        foreach ($this->integrations as $integration) {
            if ($integration['type'] !== 'wc_product_page') {
                continue;
            }

            $placement = $integration['placement'] ?? 'after_add_to_cart';
            if (!isset($placements[$placement])) {
                $placements[$placement] = [];
            }
            $placements[$placement][] = $integration;
        }

        // Register hooks for each placement
        foreach ($placements as $placement => $integrations) {
            $this->registered_placements[$placement] = $integrations;

            switch ($placement) {
                case 'after_add_to_cart':
                    add_action('woocommerce_after_add_to_cart_button', [$this, 'render_buttons_after_add_to_cart']);
                    break;
                case 'before_add_to_cart':
                    add_action('woocommerce_before_add_to_cart_button', [$this, 'render_buttons_before_add_to_cart']);
                    break;
                case 'after_summary':
                    add_action('woocommerce_after_single_product_summary', [$this, 'render_buttons_after_summary'], 15);
                    break;
            }
        }
    }

    /**
     * Render buttons for after_add_to_cart placement
     */
    public function render_buttons_after_add_to_cart() {
        $this->render_buttons_for_placement('after_add_to_cart');
    }

    /**
     * Render buttons for before_add_to_cart placement
     */
    public function render_buttons_before_add_to_cart() {
        $this->render_buttons_for_placement('before_add_to_cart');
    }

    /**
     * Render buttons for after_summary placement
     */
    public function render_buttons_after_summary() {
        $this->render_buttons_for_placement('after_summary');
    }

    /**
     * Render all matching buttons for a specific placement
     */
    private function render_buttons_for_placement($placement) {
        global $product;

        if (!$product || !is_a($product, 'WC_Product')) {
            return;
        }

        if (!isset($this->registered_placements[$placement])) {
            return;
        }

        foreach ($this->registered_placements[$placement] as $integration) {
            if ($this->integration_matches_product($integration, $product)) {
                /**
                 * Filter whether the widget should be displayed.
                 *
                 * @param bool       $should_display Whether to display the widget. Default true.
                 * @param array      $integration    The integration configuration array.
                 * @param WC_Product $product        The current product object.
                 */
                $should_display = apply_filters('selektable_should_display_widget', true, $integration, $product);

                if (!$should_display) {
                    continue;
                }

                $widget_id = $integration['widget_id'] ?? '';
                $button_text = $integration['button_text'] ?? 'Try it on';
                $button_class = $integration['button_class'] ?? 'button selektable-try-on';
                $button_styles = [
                    'button_bg' => $integration['button_bg'] ?? '',
                    'button_color' => $integration['button_color'] ?? '',
                    'button_radius' => $integration['button_radius'] ?? '',
                    'button_border' => $integration['button_border'] ?? '',
                ];

                $this->render_button_with_widget_id($product, $widget_id, $button_text, $button_class, $button_styles);
            }
        }
    }

    /**
     * Check if an integration matches a product based on activation rules
     */
    private function integration_matches_product($integration, $product) {
        $product_id = $product->get_id();

        // 1. Check product-level override (highest priority)
        $status = get_post_meta($product_id, '_selektable_status', true);
        if ($status === 'enabled') {
            return true;
        }
        if ($status === 'disabled') {
            return false;
        }

        // 2. Check activation mode
        $mode = $integration['activation_mode'] ?? 'all';

        if ($mode === 'all') {
            return true;
        }

        if ($mode === 'categories') {
            $enabled_cats = $integration['categories'] ?? [];
            if (!empty($enabled_cats) && is_array($enabled_cats)) {
                $product_cats = wp_get_post_terms($product_id, 'product_cat', ['fields' => 'ids']);
                if (!is_wp_error($product_cats) && array_intersect($product_cats, $enabled_cats)) {
                    return true;
                }
            }
            return false;
        }

        if ($mode === 'tags') {
            $enabled_tags = $integration['tags'] ?? [];
            if (!empty($enabled_tags) && is_array($enabled_tags)) {
                $product_tags = wp_get_post_terms($product_id, 'product_tag', ['fields' => 'ids']);
                if (!is_wp_error($product_tags) && array_intersect($product_tags, $enabled_tags)) {
                    return true;
                }
            }
            return false;
        }

        return false;
    }

    /**
     * Check if any integration is enabled for a product (legacy method for compatibility)
     */
    public function is_enabled_for_product($product) {
        if (!$product || !is_a($product, 'WC_Product')) {
            return false;
        }

        foreach ($this->integrations as $integration) {
            if ($integration['type'] === 'wc_product_page') {
                if ($this->integration_matches_product($integration, $product)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Get product data for the widget (WooCommerce only)
     */
    public function get_product_data($product) {
        $product_id = $product->get_id();
        $images = [];

        // Check for override image
        $override_image_id = get_post_meta($product_id, '_selektable_override_image_id', true);
        $override_image_url = $override_image_id ? wp_get_attachment_url($override_image_id) : null;

        // Get featured image
        $featured_image_id = $product->get_image_id();
        if ($featured_image_id) {
            $images[] = [
                'id' => (int) $featured_image_id,
                'src' => wp_get_attachment_url($featured_image_id),
                'alt' => get_post_meta($featured_image_id, '_wp_attachment_image_alt', true) ?: '',
                'name' => get_the_title($featured_image_id),
            ];
        }

        // Get gallery images
        $gallery_ids = $product->get_gallery_image_ids();
        foreach ($gallery_ids as $gallery_id) {
            $images[] = [
                'id' => (int) $gallery_id,
                'src' => wp_get_attachment_url($gallery_id),
                'alt' => get_post_meta($gallery_id, '_wp_attachment_image_alt', true) ?: '',
                'name' => get_the_title($gallery_id),
            ];
        }

        $data = [
            'productId' => $product_id,
            'productTitle' => $product->get_name(),
            'productUrl' => get_permalink($product_id),
            'productImages' => $images,
            'type' => $product->get_type(),
            'is_purchasable' => $product->is_purchasable(),
            'is_in_stock' => $product->is_in_stock(),
        ];

        // Add variation data for variable products
        if ($product->is_type('variable')) {
            $data['attributes'] = $this->get_variation_attributes($product);
            $data['variations'] = $this->get_product_variations($product);
        }

        // Add physical dimensions if available
        $width = $product->get_width();
        $height = $product->get_height();
        $length = $product->get_length();

        if ($width || $height || $length) {
            $wc_unit = get_option('woocommerce_dimension_unit', 'cm');
            // Map WooCommerce units to SDK-supported units
            $unit_map = ['cm' => 'cm', 'mm' => 'mm', 'in' => 'in', 'm' => 'cm', 'yd' => 'in'];
            $unit = $unit_map[$wc_unit] ?? 'cm';

            $dimensions = ['unit' => $unit];
            if ($width) {
                $dimensions['width'] = (float) ($wc_unit === 'm' ? $width * 100 : ($wc_unit === 'yd' ? $width * 36 : $width));
            }
            if ($height) {
                $dimensions['height'] = (float) ($wc_unit === 'm' ? $height * 100 : ($wc_unit === 'yd' ? $height * 36 : $height));
            }
            if ($length) {
                $dimensions['depth'] = (float) ($wc_unit === 'm' ? $length * 100 : ($wc_unit === 'yd' ? $length * 36 : $length));
            }

            $data['dimensions'] = $dimensions;
        }

        // Add override image if set
        if ($override_image_url) {
            $data['productImage'] = $override_image_url;
        }

        return apply_filters('selektable_product_data', $data, $product);
    }

    /**
     * Get variation attributes for a variable product
     */
    private function get_variation_attributes($product) {
        $attributes = [];

        foreach ($product->get_variation_attributes() as $taxonomy => $terms) {
            $attr_data = [
                'id' => wc_attribute_taxonomy_id_by_name($taxonomy),
                'name' => wp_strip_all_tags(wc_attribute_label($taxonomy)),
                'taxonomy' => $taxonomy,
                'has_variations' => true,
                'terms' => []
            ];

            foreach ($terms as $slug) {
                $term = get_term_by('slug', $slug, $taxonomy);
                if ($term) {
                    $attr_data['terms'][] = [
                        'id' => $term->term_id,
                        'name' => wp_strip_all_tags($term->name),
                        'slug' => $term->slug
                    ];
                } else {
                    // Custom attribute (not taxonomy-based)
                    $attr_data['terms'][] = [
                        'id' => 0,
                        'name' => $slug,
                        'slug' => sanitize_title($slug)
                    ];
                }
            }

            $attributes[] = $attr_data;
        }

        return $attributes;
    }

    /**
     * Get available variations for a variable product
     */
    private function get_product_variations($product) {
        $variations = [];

        foreach ($product->get_available_variations() as $variation) {
            $var_data = [
                'id' => $variation['variation_id'],
                'attributes' => [],
                'is_in_stock' => $variation['is_in_stock']
            ];

            foreach ($variation['attributes'] as $attr_key => $value) {
                // attr_key is like "attribute_pa_color" → extract "pa_color"
                $taxonomy = str_replace('attribute_', '', $attr_key);
                $var_data['attributes'][] = [
                    'name' => $taxonomy,
                    'value' => $value ?: null  // Empty means "any"
                ];
            }

            $variations[] = $var_data;
        }

        return $variations;
    }

    /**
     * Render a button with a specific widget ID and WooCommerce product data
     * Used by shortcode and internal rendering when WooCommerce is active
     */
    public function render_button_with_widget_id($product, $widget_id, $button_text = '', $button_class = '', $button_styles = []) {
        if (empty($widget_id)) {
            return;
        }

        $product_data = $this->get_product_data($product);

        $button_text = $button_text ?: 'Try it on';
        $button_text = apply_filters('selektable_button_text', $button_text, $product);

        $button_class = $button_class ?: 'button selektable-try-on';
        $button_class = apply_filters('selektable_button_class', $button_class, $product);

        // Generate unique ID for this button instance
        $button_id = 'selektable-btn-' . uniqid();

        // Build CSS variable declarations for inline style
        $css_vars = [];
        if (!empty($button_styles['button_bg'])) {
            $css_vars[] = '--selektable-bg: ' . $button_styles['button_bg'];
        }
        if (!empty($button_styles['button_color'])) {
            $css_vars[] = '--selektable-color: ' . $button_styles['button_color'];
        }
        if (!empty($button_styles['button_radius']) || $button_styles['button_radius'] === 0 || $button_styles['button_radius'] === '0') {
            $css_vars[] = '--selektable-radius: ' . intval($button_styles['button_radius']) . 'px';
        }
        if (!empty($button_styles['button_border'])) {
            $css_vars[] = '--selektable-border: ' . $button_styles['button_border'];
        }

        $style_value = $css_vars ? esc_attr(implode('; ', $css_vars)) : '';

        printf(
            '<button type="button" id="%s" class="%s"%s data-selektable-open="%s" data-product="%s">%s</button>',
            esc_attr($button_id),
            esc_attr($button_class),
            $style_value ? ' style="' . $style_value . '"' : '',  // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- $style_value is already escaped via esc_attr() above.
            esc_attr($widget_id),
            esc_attr(wp_json_encode($product_data)),
            esc_html($button_text)
        );
    }

    /**
     * Render a simple button without product data (non-WooCommerce context)
     * Used by the shortcode when WooCommerce is not active or outside product pages
     */
    public function render_simple_button($widget_id, $button_text = '', $button_class = '') {
        if (empty($widget_id)) {
            return;
        }

        $button_text = $button_text ?: 'Try it on';
        $button_text = apply_filters('selektable_button_text', $button_text, null);

        $button_class = $button_class ?: 'button selektable-try-on';
        $button_class = apply_filters('selektable_button_class', $button_class, null);

        printf(
            '<button type="button" class="%s" data-selektable-open="%s">%s</button>',
            esc_attr($button_class),
            esc_attr($widget_id),
            esc_html($button_text)
        );
    }

    /**
     * Legacy render_button method for backwards compatibility
     */
    public function render_button($product, $custom_text = '', $custom_class = '') {
        // Find first matching WC Product Page integration
        foreach ($this->integrations as $integration) {
            if ($integration['type'] === 'wc_product_page') {
                $widget_id = $integration['widget_id'] ?? '';
                $button_text = $custom_text ?: ($integration['button_text'] ?? 'Try it on');
                $button_class = $custom_class ?: ($integration['button_class'] ?? 'button selektable-try-on');

                $this->render_button_with_widget_id($product, $widget_id, $button_text, $button_class);
                return;
            }
        }
    }

    /**
     * Add data-store-id attribute to the Selektable embed script tag
     */
    public function add_store_id_to_embed_script($tag, $handle, $src) {
        if ($handle !== 'selektable-embed') {
            return $tag;
        }

        $store_id = get_option('selektable_store_id', '');
        if (!empty($store_id)) {
            $tag = str_replace('<script ', '<script data-store-id="' . esc_attr($store_id) . '" ', $tag);
        }

        return $tag;
    }

    /**
     * Enqueue frontend scripts
     */
    public function enqueue_scripts() {
        // Use timestamp for cache busting in dev, version in production
        $embed_version = $this->is_dev ? time() : SELEKTABLE_VERSION;

        // Enqueue frontend stylesheet for custom button styles
        wp_enqueue_style(
            'selektable-frontend',
            SELEKTABLE_PLUGIN_URL . 'assets/css/frontend.css',
            [],
            SELEKTABLE_VERSION
        );

        // Enqueue the embed script on every page (works with or without WooCommerce)
        wp_enqueue_script(
            'selektable-embed',
            $this->app_url . '/widgets/embed.js',
            [],
            $embed_version,
            true
        );

        // WooCommerce-specific: product page scripts
        if (!selektable_is_woocommerce_active() || !is_product()) {
            return;
        }

        // Get product object
        global $post;
        $product = wc_get_product($post);

        if (!$product) {
            return;
        }

        // Check if any integration is enabled for this product
        $has_matching_integration = false;
        $widget_ids = [];

        foreach ($this->integrations as $integration) {
            if ($integration['type'] === 'wc_product_page') {
                if ($this->integration_matches_product($integration, $product)) {
                    $has_matching_integration = true;
                    $widget_ids[] = $integration['widget_id'];
                }
            }
        }

        if (!$has_matching_integration) {
            return;
        }

        // Enqueue our frontend handler
        wp_enqueue_script(
            'selektable-frontend',
            SELEKTABLE_PLUGIN_URL . 'assets/js/frontend.js',
            ['jquery', 'selektable-embed'],
            $embed_version,
            true
        );

        // Prepare user data for identity
        $user_data = null;
        if (is_user_logged_in()) {
            $current_user = wp_get_current_user();
            $user_data = [
                'userId' => (string) $current_user->ID,
                'email' => $current_user->user_email,
                'displayName' => $current_user->display_name,
            ];
        }

        // Localize script data
        wp_localize_script('selektable-frontend', 'selektableConfig', [
            'widgetIds' => array_unique($widget_ids),
            'cartUrl' => wc_get_cart_url(),
            'checkoutUrl' => wc_get_checkout_url(),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('selektable_cart_nonce'),
            'user' => $user_data,
        ]);
    }
}
