<?php
/**
 * Selektable Cart Integration
 *
 * Handles AJAX cart operations for the Selektable widget.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Selektable_Cart {

    public function __construct() {
        // AJAX handlers for logged-in and guest users
        add_action('wp_ajax_selektable_add_to_cart', [$this, 'add_to_cart']);
        add_action('wp_ajax_nopriv_selektable_add_to_cart', [$this, 'add_to_cart']);

        // Store identity on generation complete (before add-to-cart)
        add_action('wp_ajax_selektable_store_identity', [$this, 'store_identity']);
        add_action('wp_ajax_nopriv_selektable_store_identity', [$this, 'store_identity']);
    }

    /**
     * Store Selektable identity in WC session.
     *
     * Called on widget:generation-complete so visitor/session IDs are available
     * for order attribution even if the user adds to cart via the normal
     * WooCommerce button instead of the widget's add-to-cart callback.
     */
    public function store_identity() {
        if (!check_ajax_referer('selektable_cart_nonce', 'nonce', false)) {
            wp_send_json(['success' => false], 403);
        }

        $visitor_id = isset($_POST['selektable_visitor_id']) ? sanitize_text_field(wp_unslash($_POST['selektable_visitor_id'])) : '';
        $session_id = isset($_POST['selektable_session_id']) ? sanitize_text_field(wp_unslash($_POST['selektable_session_id'])) : '';

        if ($visitor_id && WC()->session) {
            WC()->session->set('selektable_visitor_id', $visitor_id);
            WC()->session->set('selektable_session_id', $session_id);
        }

        wp_send_json(['success' => true]);
    }

    /**
     * Handle add to cart AJAX request
     *
     * The widget sends product_id which may be:
     * - A simple product ID
     * - A variation ID (for variable products after user selects options)
     * - A parent variable product ID (legacy flow with variation attributes)
     */
    public function add_to_cart() {
        // Verify nonce
        if (!check_ajax_referer('selektable_cart_nonce', 'nonce', false)) {
            wp_send_json([
                'success' => false,
                'error' => 'INVALID_NONCE',
            ], 403);
        }

        // Get request data
        $product_id = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $quantity = isset($_POST['quantity']) ? absint($_POST['quantity']) : 1;

        // Parse variation JSON from widget
        $variation_raw = isset($_POST['variation']) ? sanitize_text_field(wp_unslash($_POST['variation'])) : '';
        $variation_data = is_string($variation_raw) ? json_decode($variation_raw, true) : [];
        if (!is_array($variation_data)) {
            $variation_data = [];
        }

        // Store Selektable identity in WC session for order attribution
        $visitor_id = isset($_POST['selektable_visitor_id']) ? sanitize_text_field(wp_unslash($_POST['selektable_visitor_id'])) : '';
        $session_id = isset($_POST['selektable_session_id']) ? sanitize_text_field(wp_unslash($_POST['selektable_session_id'])) : '';

        if ($visitor_id && WC()->session) {
            WC()->session->set('selektable_visitor_id', $visitor_id);
            WC()->session->set('selektable_session_id', $session_id);
        }

        if (!$product_id) {
            wp_send_json([
                'success' => false,
                'error' => 'INVALID_PRODUCT',
            ]);
        }

        // Get the product
        $product = wc_get_product($product_id);
        if (!$product) {
            wp_send_json([
                'success' => false,
                'error' => 'PRODUCT_NOT_FOUND',
            ]);
        }

        $variation_id = 0;
        $variation = [];

        // Check if product_id is actually a variation (new flow from widget)
        if ($product->is_type('variation')) {
            $variation_id = $product_id;
            $product_id = $product->get_parent_id();

            // Build variation attributes from the variation product
            foreach ($product->get_variation_attributes() as $key => $value) {
                $variation['attribute_' . $key] = $value;
            }
        } elseif ($product->is_type('variable')) {
            // Legacy flow: parent product with variation attributes
            if (!empty($variation_data)) {
                // Widget sends: [{attribute: "pa_color", value: "red"}, ...]
                foreach ($variation_data as $attr) {
                    if (isset($attr['attribute']) && isset($attr['value'])) {
                        $key = 'attribute_' . sanitize_title($attr['attribute']);
                        $variation[$key] = sanitize_text_field($attr['value']);
                    }
                }

                // Find matching variation
                $data_store = WC_Data_Store::load('product');
                $variation_id = $data_store->find_matching_product_variation($product, $variation);
            }

            if (!$variation_id) {
                wp_send_json([
                    'success' => false,
                    'error' => 'VARIATION_REQUIRED',
                ]);
            }
        }

        // Add to cart
        try {
            $cart_item_key = WC()->cart->add_to_cart(
                $product_id,
                $quantity,
                $variation_id,
                $variation
            );

            if ($cart_item_key) {
                // Store tracking data in cart item for robust order attribution.
                // Cart item data survives session regeneration and works with
                // both Classic and Block Checkout.
                if ($visitor_id) {
                    WC()->cart->cart_contents[$cart_item_key]['selektable_visitor_id'] = $visitor_id;
                    WC()->cart->cart_contents[$cart_item_key]['selektable_session_id'] = $session_id;
                }

                // Return success response
                // Note: We don't call WC_AJAX::get_refreshed_fragments() here because
                // it outputs and terminates. The frontend will trigger a fragment refresh
                // if needed after receiving this success response.
                wp_send_json([
                    'success' => true,
                    'cartUrl' => wc_get_cart_url(),
                    'cartCount' => WC()->cart->get_cart_contents_count(),
                    'cartItemKey' => $cart_item_key,
                ]);
            } else {
                // Get any notices that might explain why it failed
                $notices = wc_get_notices('error');
                $error_message = !empty($notices) ? wp_strip_all_tags($notices[0]['notice']) : 'ADD_TO_CART_FAILED';
                wc_clear_notices();

                wp_send_json([
                    'success' => false,
                    'error' => $error_message,
                ]);
            }
        } catch (Exception $e) {
            wp_send_json([
                'success' => false,
                'error' => $e->getMessage(),
            ]);
        }
    }
}
