<?php
/**
 * Selektable Admin Settings
 *
 * Provides a standalone settings page under Settings > Selektable
 * with integrations table and modal-based editing.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Selektable_Admin {

    private $hook_suffix;

    public function __construct() {
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'handle_settings_save']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);

        // AJAX handlers
        add_action('wp_ajax_selektable_get_integrations', [$this, 'ajax_get_integrations']);
        add_action('wp_ajax_selektable_save_integration', [$this, 'ajax_save_integration']);
        add_action('wp_ajax_selektable_delete_integration', [$this, 'ajax_delete_integration']);
    }

    /**
     * Add the Selektable settings page under Settings menu
     */
    public function add_settings_page() {
        $this->hook_suffix = add_options_page(
            __('Selektable', 'selektable'),
            __('Selektable', 'selektable'),
            'manage_options',
            'selektable',
            [$this, 'render_settings_page']
        );
    }

    /**
     * Check if site is running on localhost or .local domain
     */
    private function is_local_environment() {
        $site_url = get_site_url();
        $host = wp_parse_url($site_url, PHP_URL_HOST);

        if (!$host) {
            return false;
        }

        if ($host === 'localhost' || $host === '127.0.0.1') {
            return true;
        }

        if (str_ends_with($host, '.local')) {
            return true;
        }

        if (str_ends_with($host, '.chrisjayden.com')) {
            return true;
        }

        return false;
    }

    /**
     * Handle developer settings save
     */
    public function handle_settings_save() {
        if (!isset($_POST['selektable_save_settings_nonce'])) {
            return;
        }

        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['selektable_save_settings_nonce'])), 'selektable_save_settings')) {
            return;
        }

        if (!current_user_can('manage_options')) {
            return;
        }

        if (isset($_POST['selektable_store_id'])) {
            update_option('selektable_store_id', sanitize_text_field(wp_unslash($_POST['selektable_store_id'])));
        }

        if (isset($_POST['selektable_app_url']) && $this->is_local_environment()) {
            update_option('selektable_app_url', sanitize_text_field(wp_unslash($_POST['selektable_app_url'])));
        }

        add_settings_error('selektable', 'settings_updated', __('Settings saved.', 'selektable'), 'updated');
    }

    /**
     * Render the settings page
     */
    public function render_settings_page() {
        $integrations = get_option('selektable_integrations', []);
        $wc_active = selektable_is_woocommerce_active();
        $categories = $wc_active ? $this->get_product_categories() : [];
        $tags = $wc_active ? $this->get_product_tags() : [];
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <?php settings_errors('selektable'); ?>

            <div class="selektable-settings-wrap">
                <?php if ($this->is_local_environment()): ?>
                <!-- Developer Settings -->
                <div class="selektable-developer-settings">
                    <h2><?php esc_html_e('Developer Settings', 'selektable'); ?></h2>
                    <form method="post" action="">
                        <?php wp_nonce_field('selektable_save_settings', 'selektable_save_settings_nonce'); ?>
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="selektable_app_url"><?php esc_html_e('App URL', 'selektable'); ?></label>
                                </th>
                                <td>
                                    <input type="text"
                                           name="selektable_app_url"
                                           id="selektable_app_url"
                                           value="<?php echo esc_attr(get_option('selektable_app_url', 'https://app.selektable.com')); ?>"
                                           class="regular-text"
                                    />
                                    <p class="description"><?php esc_html_e('The Selektable app URL. Only change this for local development.', 'selektable'); ?></p>
                                </td>
                            </tr>
                        </table>
                        <?php submit_button(__('Save Developer Settings', 'selektable')); ?>
                    </form>
                </div>
                <?php endif; ?>

                <!-- General Settings -->
                <div class="selektable-general-settings">
                    <h2><?php esc_html_e('General Settings', 'selektable'); ?></h2>
                    <form method="post" action="">
                        <?php wp_nonce_field('selektable_save_settings', 'selektable_save_settings_nonce'); ?>
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="selektable_store_id"><?php esc_html_e('Store ID', 'selektable'); ?> <span class="required">*</span></label>
                                </th>
                                <td>
                                    <input type="text"
                                           name="selektable_store_id"
                                           id="selektable_store_id"
                                           value="<?php echo esc_attr(get_option('selektable_store_id', '')); ?>"
                                           class="regular-text"
                                           placeholder="store_xxx"
                                           required
                                    />
                                    <p class="description"><?php esc_html_e('Your Selektable Store ID. Find this in your Selektable dashboard under your store settings.', 'selektable'); ?></p>
                                    <?php if (empty(get_option('selektable_store_id', ''))): ?>
                                    <p class="description" style="color: #d63638;"><?php esc_html_e('Store ID is required for the widget to function.', 'selektable'); ?></p>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        </table>
                        <?php submit_button(__('Save Settings', 'selektable')); ?>
                    </form>
                </div>

                <!-- Integrations Section -->
                <div class="selektable-integrations-section">
                    <div class="selektable-integrations-header">
                        <h2><?php esc_html_e('Integrations', 'selektable'); ?></h2>
                        <button type="button" class="button button-primary" id="selektable-add-integration">
                            <?php esc_html_e('Add New', 'selektable'); ?>
                        </button>
                    </div>

                    <table class="wp-list-table widefat fixed striped" id="selektable-integrations-table">
                        <thead>
                            <tr>
                                <th class="column-type"><?php esc_html_e('Type', 'selektable'); ?></th>
                                <th class="column-widget-id"><?php esc_html_e('Widget ID', 'selektable'); ?></th>
                                <th class="column-summary"><?php esc_html_e('Summary', 'selektable'); ?></th>
                                <th class="column-actions"><?php esc_html_e('Actions', 'selektable'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($integrations)): ?>
                            <tr class="no-integrations">
                                <td colspan="4"><?php esc_html_e('No integrations configured. Click "Add New" to create one.', 'selektable'); ?></td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($integrations as $integration): ?>
                                <tr data-id="<?php echo esc_attr($integration['id']); ?>">
                                    <td class="column-type">
                                        <?php echo $integration['type'] === 'wc_product_page'
                                            ? esc_html__('WC Product Page', 'selektable')
                                            : esc_html__('Shortcode', 'selektable'); ?>
                                    </td>
                                    <td class="column-widget-id">
                                        <code><?php echo esc_html($integration['widget_id']); ?></code>
                                    </td>
                                    <td class="column-summary">
                                        <?php echo esc_html($this->get_integration_summary($integration, $categories, $tags)); ?>
                                    </td>
                                    <td class="column-actions">
                                        <button type="button" class="button selektable-edit-integration" data-id="<?php echo esc_attr($integration['id']); ?>">
                                            <?php esc_html_e('Edit', 'selektable'); ?>
                                        </button>
                                        <button type="button" class="button selektable-delete-integration" data-id="<?php echo esc_attr($integration['id']); ?>">
                                            <?php esc_html_e('Delete', 'selektable'); ?>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>

                    <?php if (!$wc_active): ?>
                    <p class="description" style="margin-top: 10px;">
                        <?php esc_html_e('Install WooCommerce to unlock automatic product page integrations.', 'selektable'); ?>
                    </p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Integration Modal -->
            <div id="selektable-modal" class="selektable-modal" style="display: none;">
                <div class="selektable-modal-backdrop"></div>
                <div class="selektable-modal-content">
                    <div class="selektable-modal-header">
                        <h2 id="selektable-modal-title"><?php esc_html_e('Add Integration', 'selektable'); ?></h2>
                        <button type="button" class="selektable-modal-close">&times;</button>
                    </div>
                    <div class="selektable-modal-body">
                        <!-- Step 1: Type Selection -->
                        <div id="selektable-step-1" class="selektable-step">
                            <p class="selektable-step-intro"><?php esc_html_e('Select the type of integration you want to create:', 'selektable'); ?></p>
                            <div class="selektable-type-cards">
                                <?php if ($wc_active): ?>
                                <div class="selektable-type-card" data-type="wc_product_page">
                                    <div class="selektable-type-card-icon">
                                        <span class="dashicons dashicons-cart"></span>
                                    </div>
                                    <h3><?php esc_html_e('WC Product Page', 'selektable'); ?></h3>
                                    <p><?php esc_html_e('Automatically display the button on WooCommerce product pages based on categories, tags, or all products.', 'selektable'); ?></p>
                                </div>
                                <?php endif; ?>
                                <div class="selektable-type-card" data-type="shortcode">
                                    <div class="selektable-type-card-icon">
                                        <span class="dashicons dashicons-shortcode"></span>
                                    </div>
                                    <h3><?php esc_html_e('Shortcode', 'selektable'); ?></h3>
                                    <p><?php esc_html_e('Use a shortcode to place the button anywhere in your content or templates.', 'selektable'); ?></p>
                                </div>
                            </div>
                        </div>

                        <!-- Step 2: Configuration Form -->
                        <div id="selektable-step-2" class="selektable-step" style="display: none;">
                            <div id="selektable-integration-form">
                                <input type="hidden" id="selektable-int-id" value="">
                                <input type="hidden" id="selektable-int-type" value="">

                                <!-- Common Fields -->
                                <div class="selektable-form-section">
                                    <h3><?php esc_html_e('Widget Settings', 'selektable'); ?></h3>

                                    <div class="selektable-form-row">
                                        <label for="selektable-widget-id"><?php esc_html_e('Widget ID', 'selektable'); ?> <span class="required">*</span></label>
                                        <input type="text" id="selektable-widget-id" placeholder="wgt_abc123">
                                        <p class="description"><?php esc_html_e('Your Selektable widget ID. Find this in your Selektable dashboard.', 'selektable'); ?></p>
                                    </div>

                                    <div class="selektable-form-row">
                                        <label for="selektable-button-text"><?php esc_html_e('Button Text', 'selektable'); ?></label>
                                        <input type="text" id="selektable-button-text" value="Try it on">
                                    </div>

                                    <div class="selektable-form-row">
                                        <label for="selektable-button-class"><?php esc_html_e('Button CSS Class', 'selektable'); ?></label>
                                        <input type="text" id="selektable-button-class" value="button selektable-try-on">
                                    </div>

                                    <div class="selektable-form-row">
                                        <label for="selektable-button-bg"><?php esc_html_e('Button Background Color', 'selektable'); ?></label>
                                        <input type="text" id="selektable-button-bg" placeholder="#ffffff">
                                    </div>

                                    <div class="selektable-form-row">
                                        <label for="selektable-button-color"><?php esc_html_e('Button Text Color', 'selektable'); ?></label>
                                        <input type="text" id="selektable-button-color" placeholder="#000000">
                                    </div>

                                    <div class="selektable-form-row">
                                        <label for="selektable-button-radius"><?php esc_html_e('Button Border Radius', 'selektable'); ?></label>
                                        <input type="number" id="selektable-button-radius" placeholder="4" min="0"> px
                                    </div>

                                    <div class="selektable-form-row">
                                        <label for="selektable-button-border"><?php esc_html_e('Button Border Color', 'selektable'); ?></label>
                                        <input type="text" id="selektable-button-border" placeholder="#cccccc">
                                    </div>
                                </div>

                                <!-- WC Product Page Fields -->
                                <?php if ($wc_active): ?>
                                <div class="selektable-form-section selektable-wc-fields" style="display: none;">
                                    <h3><?php esc_html_e('Product Page Settings', 'selektable'); ?></h3>

                                    <div class="selektable-form-row">
                                        <label for="selektable-placement"><?php esc_html_e('Button Placement', 'selektable'); ?></label>
                                        <select id="selektable-placement">
                                            <option value="after_add_to_cart"><?php esc_html_e('After Add to Cart button', 'selektable'); ?></option>
                                            <option value="before_add_to_cart"><?php esc_html_e('Before Add to Cart button', 'selektable'); ?></option>
                                            <option value="after_summary"><?php esc_html_e('After product summary', 'selektable'); ?></option>
                                        </select>
                                    </div>

                                    <div class="selektable-form-row">
                                        <label><?php esc_html_e('Activation Mode', 'selektable'); ?></label>
                                        <div class="selektable-radio-group">
                                            <label>
                                                <input type="radio" name="selektable_activation_mode" value="all" checked>
                                                <?php esc_html_e('All products', 'selektable'); ?>
                                            </label>
                                            <label>
                                                <input type="radio" name="selektable_activation_mode" value="categories">
                                                <?php esc_html_e('Specific categories', 'selektable'); ?>
                                            </label>
                                            <label>
                                                <input type="radio" name="selektable_activation_mode" value="tags">
                                                <?php esc_html_e('Specific tags', 'selektable'); ?>
                                            </label>
                                        </div>
                                    </div>

                                    <div class="selektable-form-row selektable-categories-row" style="display: none;">
                                        <label for="selektable-categories"><?php esc_html_e('Categories', 'selektable'); ?></label>
                                        <select id="selektable-categories" multiple class="selektable-enhanced-select" data-placeholder="<?php esc_attr_e('Select categories...', 'selektable'); ?>">
                                            <?php foreach ($categories as $id => $name): ?>
                                            <option value="<?php echo esc_attr($id); ?>"><?php echo esc_html($name); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>

                                    <div class="selektable-form-row selektable-tags-row" style="display: none;">
                                        <label for="selektable-tags"><?php esc_html_e('Tags', 'selektable'); ?></label>
                                        <select id="selektable-tags" multiple class="selektable-enhanced-select" data-placeholder="<?php esc_attr_e('Select tags...', 'selektable'); ?>">
                                            <?php foreach ($tags as $id => $name): ?>
                                            <option value="<?php echo esc_attr($id); ?>"><?php echo esc_html($name); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <!-- Shortcode Fields -->
                                <div class="selektable-form-section selektable-shortcode-fields" style="display: none;">
                                    <h3><?php esc_html_e('Shortcode', 'selektable'); ?></h3>

                                    <div class="selektable-form-row">
                                        <label><?php esc_html_e('Generated Shortcode', 'selektable'); ?></label>
                                        <div class="selektable-shortcode-display">
                                            <code id="selektable-generated-shortcode">[selektable_button]</code>
                                            <button type="button" class="button selektable-copy-shortcode">
                                                <span class="dashicons dashicons-clipboard"></span>
                                                <?php esc_html_e('Copy', 'selektable'); ?>
                                            </button>
                                        </div>
                                        <p class="description"><?php esc_html_e('Use this shortcode in your pages, posts, or templates.', 'selektable'); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="selektable-modal-footer">
                        <button type="button" class="button selektable-modal-back" style="display: none;">
                            <?php esc_html_e('Back', 'selektable'); ?>
                        </button>
                        <button type="button" class="button button-primary selektable-modal-save" style="display: none;">
                            <?php esc_html_e('Save Integration', 'selektable'); ?>
                        </button>
                    </div>
                </div>
            </div>

        </div>
        <?php
    }

    /**
     * Get a human-readable summary for an integration
     */
    private function get_integration_summary($integration, $categories, $tags) {
        if ($integration['type'] === 'shortcode') {
            return '[selektable_button widget_id="' . $integration['widget_id'] . '"]';
        }

        // WC Product Page
        $mode = $integration['activation_mode'] ?? 'all';

        if ($mode === 'all') {
            return __('All products', 'selektable');
        }

        if ($mode === 'categories' && !empty($integration['categories'])) {
            $names = [];
            foreach ($integration['categories'] as $cat_id) {
                if (isset($categories[$cat_id])) {
                    $names[] = $categories[$cat_id];
                }
            }
            return implode(', ', $names) . ' ' . __('categories', 'selektable');
        }

        if ($mode === 'tags' && !empty($integration['tags'])) {
            $names = [];
            foreach ($integration['tags'] as $tag_id) {
                if (isset($tags[$tag_id])) {
                    $names[] = $tags[$tag_id];
                }
            }
            return implode(', ', $names) . ' ' . __('tags', 'selektable');
        }

        return __('All products', 'selektable');
    }

    /**
     * Get product categories for multiselect (WooCommerce only)
     */
    private function get_product_categories() {
        $categories = [];
        $terms = get_terms([
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
        ]);

        if (!is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[$term->term_id] = $term->name;
            }
        }

        return $categories;
    }

    /**
     * Get product tags for multiselect (WooCommerce only)
     */
    private function get_product_tags() {
        $tags = [];
        $terms = get_terms([
            'taxonomy' => 'product_tag',
            'hide_empty' => false,
        ]);

        if (!is_wp_error($terms)) {
            foreach ($terms as $term) {
                $tags[$term->term_id] = $term->name;
            }
        }

        return $tags;
    }

    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our settings page
        if ($hook !== $this->hook_suffix) {
            return;
        }

        wp_enqueue_style(
            'selektable-admin',
            SELEKTABLE_PLUGIN_URL . 'assets/css/admin.css',
            [],
            SELEKTABLE_VERSION
        );

        $js_deps = ['jquery'];

        // Include select2 from WooCommerce if available
        if (selektable_is_woocommerce_active() && wp_script_is('select2', 'registered')) {
            $js_deps[] = 'select2';
        }

        wp_enqueue_script(
            'selektable-admin',
            SELEKTABLE_PLUGIN_URL . 'assets/js/admin.js',
            $js_deps,
            SELEKTABLE_VERSION,
            true
        );

        $wc_active    = selektable_is_woocommerce_active();
        $integrations = get_option('selektable_integrations', []);
        $categories   = $wc_active ? $this->get_product_categories() : [];
        $tags         = $wc_active ? $this->get_product_tags() : [];

        wp_localize_script('selektable-admin', 'selektableAdmin', [
            'ajaxUrl'      => admin_url('admin-ajax.php'),
            'nonce'        => wp_create_nonce('selektable_admin_nonce'),
            'integrations' => $integrations,
            'categories'   => $categories,
            'tags'         => $tags,
            'wcActive'     => $wc_active,
            'i18n'         => [
                'addIntegration'  => __('Add Integration', 'selektable'),
                'editIntegration' => __('Edit Integration', 'selektable'),
                'confirmDelete'   => __('Are you sure you want to delete this integration?', 'selektable'),
                'saving'          => __('Saving...', 'selektable'),
                'saved'           => __('Saved!', 'selektable'),
                'error'           => __('An error occurred. Please try again.', 'selektable'),
                'widgetIdRequired' => __('Widget ID is required.', 'selektable'),
                'copied'          => __('Copied!', 'selektable'),
                'wcProductPage'   => __('WC Product Page', 'selektable'),
                'shortcode'       => __('Shortcode', 'selektable'),
                'noIntegrations'  => __('No integrations configured. Click "Add New" to create one.', 'selektable'),
            ],
        ]);
    }

    /**
     * AJAX: Get all integrations
     */
    public function ajax_get_integrations() {
        check_ajax_referer('selektable_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'selektable')]);
        }

        $integrations = get_option('selektable_integrations', []);
        wp_send_json_success(['integrations' => $integrations]);
    }

    /**
     * AJAX: Save integration (create or update)
     */
    public function ajax_save_integration() {
        check_ajax_referer('selektable_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'selektable')]);
        }

        $id = isset($_POST['id']) ? sanitize_text_field(wp_unslash($_POST['id'])) : '';
        $type = isset($_POST['type']) ? sanitize_text_field(wp_unslash($_POST['type'])) : '';
        $widget_id = isset($_POST['widget_id']) ? sanitize_text_field(wp_unslash($_POST['widget_id'])) : '';

        // Validation - allow wc_product_page only when WooCommerce is active
        $valid_types = ['shortcode'];
        if (selektable_is_woocommerce_active()) {
            $valid_types[] = 'wc_product_page';
        }

        if (empty($type) || !in_array($type, $valid_types, true)) {
            wp_send_json_error(['message' => __('Invalid integration type.', 'selektable')]);
        }

        if (empty($widget_id)) {
            wp_send_json_error(['message' => __('Widget ID is required.', 'selektable')]);
        }

        $integrations = get_option('selektable_integrations', []);
        $now = time();

        // Build integration data
        $integration = [
            'type' => $type,
            'widget_id' => $widget_id,
            'button_text' => isset($_POST['button_text']) ? sanitize_text_field(wp_unslash($_POST['button_text'])) : 'Try it on',
            'button_class' => isset($_POST['button_class']) ? sanitize_text_field(wp_unslash($_POST['button_class'])) : 'button selektable-try-on',
            'button_bg' => isset($_POST['button_bg']) ? sanitize_hex_color(wp_unslash($_POST['button_bg'])) : '',
            'button_color' => isset($_POST['button_color']) ? sanitize_hex_color(wp_unslash($_POST['button_color'])) : '',
            'button_radius' => isset($_POST['button_radius']) ? absint($_POST['button_radius']) : '',
            'button_border' => isset($_POST['button_border']) ? sanitize_hex_color(wp_unslash($_POST['button_border'])) : '',
            'updated_at' => $now,
        ];

        // WC Product Page specific fields
        if ($type === 'wc_product_page') {
            $integration['placement'] = isset($_POST['placement']) ? sanitize_text_field(wp_unslash($_POST['placement'])) : 'after_add_to_cart';
            $integration['activation_mode'] = isset($_POST['activation_mode']) ? sanitize_text_field(wp_unslash($_POST['activation_mode'])) : 'all';

            // Categories
            $categories = isset($_POST['categories']) ? array_map('intval', wp_unslash((array) $_POST['categories'])) : [];
            $integration['categories'] = $categories;

            // Tags
            $tags = isset($_POST['tags']) ? array_map('intval', wp_unslash((array) $_POST['tags'])) : [];
            $integration['tags'] = $tags;
        }

        // Update existing or create new
        if (!empty($id)) {
            // Find and update existing
            $found = false;
            foreach ($integrations as $key => $existing) {
                if ($existing['id'] === $id) {
                    $integration['id'] = $id;
                    $integration['created_at'] = $existing['created_at'] ?? $now;
                    $integrations[$key] = $integration;
                    $found = true;
                    break;
                }
            }

            if (!$found) {
                wp_send_json_error(['message' => __('Integration not found.', 'selektable')]);
            }
        } else {
            // Create new
            $integration['id'] = 'int_' . wp_generate_password(8, false);
            $integration['created_at'] = $now;
            $integrations[] = $integration;
        }

        update_option('selektable_integrations', $integrations);

        wp_send_json_success([
            'integration' => $integration,
            'message' => __('Integration saved successfully.', 'selektable'),
        ]);
    }

    /**
     * AJAX: Delete integration
     */
    public function ajax_delete_integration() {
        check_ajax_referer('selektable_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'selektable')]);
        }

        $id = isset($_POST['id']) ? sanitize_text_field(wp_unslash($_POST['id'])) : '';

        if (empty($id)) {
            wp_send_json_error(['message' => __('Integration ID is required.', 'selektable')]);
        }

        $integrations = get_option('selektable_integrations', []);
        $found = false;

        foreach ($integrations as $key => $integration) {
            if ($integration['id'] === $id) {
                unset($integrations[$key]);
                $found = true;
                break;
            }
        }

        if (!$found) {
            wp_send_json_error(['message' => __('Integration not found.', 'selektable')]);
        }

        // Re-index array
        $integrations = array_values($integrations);
        update_option('selektable_integrations', $integrations);

        wp_send_json_success(['message' => __('Integration deleted successfully.', 'selektable')]);
    }
}
