/**
 * Selektable Product Page JavaScript
 *
 * Handles the custom image picker for override images on product edit pages.
 */

(function ($) {
    'use strict';

    var SelektableProduct = {
        modal: null,

        init: function () {
            this.createModal();
            this.bindEvents();
        },

        createModal: function () {
            var $modal = $('<div>', { id: 'selektable-image-picker-modal', 'class': 'selektable-image-picker-modal', style: 'display: none;' });
            $modal.append($('<div>', { 'class': 'selektable-image-picker-backdrop' }));

            var $content = $('<div>', { 'class': 'selektable-image-picker-content' });

            var $header = $('<div>', { 'class': 'selektable-image-picker-header' });
            $header.append($('<h2>', { text: selektableProduct.i18n.selectImage }));
            $header.append($('<button>', { type: 'button', 'class': 'selektable-image-picker-close', html: '&times;' }));
            $content.append($header);

            $content.append($('<div>', { 'class': 'selektable-image-picker-body' }));

            var $footer = $('<div>', { 'class': 'selektable-image-picker-footer' });
            $footer.append($('<button>', { type: 'button', 'class': 'button selektable-image-picker-cancel', text: selektableProduct.i18n.cancel }));
            $content.append($footer);

            $modal.append($content);
            $('body').append($modal);
            this.modal = $modal;
        },

        bindEvents: function () {
            var self = this;

            // Open picker
            $(document).on('click', '.selektable-select-image', function (e) {
                e.preventDefault();
                self.openPicker();
            });

            // Close picker
            $(document).on('click', '.selektable-image-picker-close, .selektable-image-picker-backdrop, .selektable-image-picker-cancel', function (e) {
                e.preventDefault();
                self.closePicker();
            });

            // Select image
            $(document).on('click', '.selektable-image-picker-item', function (e) {
                e.preventDefault();
                var imageId = $(this).data('id');
                var imageUrl = $(this).data('url');
                self.selectImage(imageId, imageUrl);
            });

            // Remove image
            $(document).on('click', '.selektable-remove-image', function (e) {
                e.preventDefault();
                self.removeImage();
            });

            // Escape key closes modal
            $(document).on('keydown', function (e) {
                if (e.key === 'Escape' && self.modal.is(':visible')) {
                    self.closePicker();
                }
            });
        },

        openPicker: function () {
            var $body = this.modal.find('.selektable-image-picker-body');
            $body.empty();

            if (!selektableProduct.images || selektableProduct.images.length === 0) {
                $body.append($('<p>', { 'class': 'selektable-no-images', text: selektableProduct.i18n.noImages }));
            } else {
                var $grid = $('<div>', { 'class': 'selektable-image-picker-grid' });
                selektableProduct.images.forEach(function (image) {
                    var $item = $('<div>', { 'class': 'selektable-image-picker-item', 'data-id': image.id, 'data-url': image.url });
                    $item.append($('<img>', { src: image.url, alt: '' }));
                    $item.append($('<span>', { 'class': 'selektable-image-picker-label', text: image.label }));
                    $grid.append($item);
                });
                $body.append($grid);
            }

            this.modal.show();
            $('body').addClass('selektable-picker-open');
        },

        closePicker: function () {
            this.modal.hide();
            $('body').removeClass('selektable-picker-open');
        },

        selectImage: function (imageId, imageUrl) {
            var $container = $('.selektable-product-options');
            var $preview = $container.find('.selektable-image-preview');
            var $input = $container.find('#selektable_override_image_id');
            var $selectBtn = $container.find('.selektable-select-image');
            var $removeBtn = $container.find('.selektable-remove-image');

            $input.val(imageId);
            $preview.empty().append($('<img>', { src: imageUrl, style: 'max-width: 100%; height: auto; display: block;' }));

            $selectBtn.hide();
            $removeBtn.show();

            this.closePicker();
        },

        removeImage: function () {
            var $container = $('.selektable-product-options');
            var $preview = $container.find('.selektable-image-preview');
            var $input = $container.find('#selektable_override_image_id');
            var $selectBtn = $container.find('.selektable-select-image');
            var $removeBtn = $container.find('.selektable-remove-image');

            $input.val('');
            $preview.html('');

            $removeBtn.hide();
            $selectBtn.show();
        }
    };

    $(document).ready(function () {
        if ($('.selektable-product-options').length) {
            SelektableProduct.init();
        }
    });

})(jQuery);
