/**
 * Selektable Frontend JavaScript
 *
 * Handles widget button clicks, cart integration, and user identity.
 */

(function () {
    'use strict';

    // Wait for Selektable to be available
    function waitForSelektable(callback, maxAttempts) {
        var attempts = 0;
        maxAttempts = maxAttempts || 50;

        var check = function () {
            attempts++;
            if (typeof window.Selektable !== 'undefined') {
                callback();
            } else if (attempts < maxAttempts) {
                setTimeout(check, 100);
            } else {
                console.warn('[Selektable] Widget script not loaded');
            }
        };

        check();
    }

    // Initialize when ready
    waitForSelektable(function () {
        init();
    });

    function init() {
        // Identify user if logged in
        if (selektableConfig.user) {
            Selektable.identify(selektableConfig.user.userId, {
                email: selektableConfig.user.email,
                displayName: selektableConfig.user.displayName
            });
        }

        // Track current variation images (for variable products)
        var currentVariationImage = null;

        // Listen for WooCommerce variation changes
        if (typeof jQuery !== 'undefined') {
            jQuery(document).on('found_variation', function (event, variation) {
                if (variation && variation.image && variation.image.full_src) {
                    currentVariationImage = {
                        id: variation.image_id || 0,
                        src: variation.image.full_src,
                        alt: variation.image.alt || '',
                        name: variation.image.title || ''
                    };
                }
            });

            jQuery(document).on('reset_data', function () {
                currentVariationImage = null;
            });
        }

        // Handle button clicks
        document.addEventListener('click', function (e) {
            var button = e.target.closest('[data-selektable-open]');
            if (!button) return;

            e.preventDefault();

            var widgetId = button.getAttribute('data-selektable-open');
            var productDataStr = button.getAttribute('data-product');

            if (!widgetId) {
                console.warn('[Selektable] No widget ID specified');
                return;
            }

            var productData = {};
            if (productDataStr) {
                try {
                    productData = JSON.parse(productDataStr);
                } catch (err) {
                    console.warn('[Selektable] Invalid product data JSON');
                }
            }

            // If we have a variation image, update product images
            if (currentVariationImage) {
                // Set as primary generation image
                productData.productImage = currentVariationImage.src;
                // Also prepend to images array
                if (productData.productImages && Array.isArray(productData.productImages)) {
                    productData.productImages = [currentVariationImage].concat(
                        productData.productImages.filter(function (img) {
                            return img.id !== currentVariationImage.id;
                        })
                    );
                }
            }

            // Add cart callbacks (widget config controls whether UI is shown)
            productData.onAddToCart = handleAddToCart;
            productData.onViewCart = handleViewCart;
            productData.onCheckout = handleCheckout;
            productData.onGenerationComplete = handleGenerationComplete;

            // Open the widget
            Selektable.open(widgetId, productData);
        });
    }

    /**
     * Handle generation complete from widget.
     *
     * Stores visitor/session IDs in the WooCommerce session immediately
     * so they're available for order attribution regardless of how the
     * user adds to cart (widget callback or normal WC button).
     */
    function handleGenerationComplete() {
        var identity = typeof Selektable !== 'undefined' ? Selektable.getIdentity() : null;
        if (!identity || !identity.visitorId) {
            return;
        }

        var data = new FormData();
        data.append('action', 'selektable_store_identity');
        data.append('nonce', selektableConfig.nonce);
        data.append('selektable_visitor_id', identity.visitorId || '');
        data.append('selektable_session_id', identity.sessionId || '');

        fetch(selektableConfig.ajaxUrl, {
            method: 'POST',
            body: data,
            credentials: 'same-origin'
        }).catch(function () {
            // Silent failure — identity storage is best-effort
        });
    }

    /**
     * Handle add to cart from widget
     *
     * The widget sends:
     * - product.id: For variable products, this is the variation_id (e.g., 456)
     * - variation: Array of {attribute, value} for validation
     */
    function handleAddToCart(product, quantity, variation) {
        return new Promise(function (resolve) {
            var data = new FormData();
            data.append('action', 'selektable_add_to_cart');
            data.append('nonce', selektableConfig.nonce);
            data.append('product_id', product.id);
            data.append('quantity', quantity);

            // Send variation attributes for WooCommerce validation
            if (variation && variation.length > 0) {
                data.append('variation', JSON.stringify(variation));
            }

            // Send Selektable identity for order attribution
            var identity = typeof Selektable !== 'undefined' ? Selektable.getIdentity() : null;
            if (identity) {
                data.append('selektable_visitor_id', identity.visitorId || '');
                data.append('selektable_session_id', identity.sessionId || '');
            }

            fetch(selektableConfig.ajaxUrl, {
                method: 'POST',
                body: data,
                credentials: 'same-origin'
            })
                .then(function (response) {
                    return response.json();
                })
                .then(function (result) {
                    if (result.success) {
                        // Trigger WooCommerce cart fragments refresh
                        if (typeof jQuery !== 'undefined') {
                            jQuery(document.body).trigger('wc_fragment_refresh');
                        }

                        resolve({
                            success: true,
                            cartUrl: result.cartUrl,
                            cartCount: result.cartCount
                        });
                    } else {
                        resolve({
                            success: false,
                            error: result.error || 'ADD_TO_CART_FAILED'
                        });
                    }
                })
                .catch(function (err) {
                    console.error('[Selektable] Cart error:', err);
                    resolve({
                        success: false,
                        error: 'NETWORK_ERROR'
                    });
                });
        });
    }

    /**
     * Handle view cart from widget
     */
    function handleViewCart() {
        window.location.href = selektableConfig.cartUrl;
    }

    /**
     * Handle checkout from widget
     */
    function handleCheckout() {
        window.location.href = selektableConfig.checkoutUrl;
    }

})();
