/**
 * Selektable Admin JavaScript
 *
 * Handles integration management modal and AJAX operations.
 */

(function ($) {
    'use strict';

    var SelektableAdmin = {
        modal: null,
        currentStep: 1,
        editingIntegration: null,

        init: function () {
            this.modal = $('#selektable-modal');
            if (!this.modal.length) {
                console.error('Selektable: Modal element not found');
                return;
            }
            this.bindEvents();
        },

        bindEvents: function () {
            var self = this;

            // Add new integration
            $(document).on('click', '#selektable-add-integration', function (e) {
                e.preventDefault();
                self.openModal();
            });

            // Edit integration
            $(document).on('click', '.selektable-edit-integration', function (e) {
                e.preventDefault();
                var id = $(this).data('id');
                self.openModal(id);
            });

            // Delete integration
            $(document).on('click', '.selektable-delete-integration', function (e) {
                e.preventDefault();
                var id = $(this).data('id');
                self.deleteIntegration(id);
            });

            // Close modal
            $(document).on('click', '.selektable-modal-close, .selektable-modal-backdrop', function (e) {
                e.preventDefault();
                self.closeModal();
            });

            // Type card selection
            $(document).on('click', '.selektable-type-card', function (e) {
                e.preventDefault();
                var type = $(this).data('type');
                self.selectType(type);
            });

            // Back button
            $(document).on('click', '.selektable-modal-back', function (e) {
                e.preventDefault();
                self.goToStep(1);
            });

            // Save button
            $(document).on('click', '.selektable-modal-save', function (e) {
                e.preventDefault();
                self.saveIntegration();
            });

            // Activation mode change
            $(document).on('change', 'input[name="selektable_activation_mode"]', function () {
                self.toggleActivationFields();
            });

            // Widget ID change (update shortcode)
            $(document).on('input', '#selektable-widget-id', function () {
                self.updateShortcodePreview();
            });

            // Copy shortcode
            $(document).on('click', '.selektable-copy-shortcode', function (e) {
                e.preventDefault();
                self.copyShortcode();
            });

            // Escape key closes modal
            $(document).on('keydown', function (e) {
                if (e.key === 'Escape' && self.modal.is(':visible')) {
                    self.closeModal();
                }
            });
        },

        openModal: function (integrationId) {
            this.editingIntegration = null;
            this.resetForm();

            if (integrationId) {
                // Find the integration
                var integration = this.findIntegration(integrationId);
                if (integration) {
                    this.editingIntegration = integration;
                    this.populateForm(integration);
                    $('#selektable-modal-title').text(selektableAdmin.i18n.editIntegration);
                    // Skip to step 2 when editing
                    this.goToStep(2);
                }
            } else {
                $('#selektable-modal-title').text(selektableAdmin.i18n.addIntegration);
                this.goToStep(1);
            }

            this.modal.show();
            $('body').addClass('selektable-modal-open');

            // Initialize Select2 on enhanced selects
            this.initSelect2();
        },

        initSelect2: function () {
            var self = this;
            // Use setTimeout to ensure modal is visible before initializing
            setTimeout(function () {
                if ($.fn.select2) {
                    $('#selektable-categories, #selektable-tags').each(function () {
                        var $select = $(this);
                        // Destroy existing instance if any
                        if ($select.hasClass('select2-hidden-accessible')) {
                            $select.select2('destroy');
                        }
                        $select.select2({
                            width: '100%',
                            dropdownParent: self.modal
                        });
                    });
                }
            }, 100);
        },

        closeModal: function () {
            // Destroy Select2 instances
            if ($.fn.select2) {
                $('#selektable-categories, #selektable-tags').each(function () {
                    var $select = $(this);
                    if ($select.hasClass('select2-hidden-accessible')) {
                        $select.select2('destroy');
                    }
                });
            }

            this.modal.hide();
            $('body').removeClass('selektable-modal-open');
            this.editingIntegration = null;
        },

        findIntegration: function (id) {
            for (var i = 0; i < selektableAdmin.integrations.length; i++) {
                if (selektableAdmin.integrations[i].id === id) {
                    return selektableAdmin.integrations[i];
                }
            }
            return null;
        },

        resetForm: function () {
            // Reset all form fields manually (not using form.reset() since modal is not a form to avoid nested forms issue)
            $('#selektable-int-id').val('');
            $('#selektable-int-type').val('');
            $('#selektable-widget-id').val('');
            $('#selektable-button-text').val('Try it on');
            $('#selektable-button-class').val('button selektable-try-on');
            $('#selektable-button-bg').val('');
            $('#selektable-button-color').val('');
            $('#selektable-button-radius').val('');
            $('#selektable-button-border').val('');
            $('#selektable-placement').val('after_add_to_cart');
            $('input[name="selektable_activation_mode"][value="all"]').prop('checked', true);
            $('#selektable-categories').val([]);
            $('#selektable-tags').val([]);
            $('.selektable-type-card').removeClass('selected');
            $('.selektable-wc-fields').hide();
            $('.selektable-shortcode-fields').hide();
            this.toggleActivationFields();
        },

        populateForm: function (integration) {
            $('#selektable-int-id').val(integration.id);
            $('#selektable-int-type').val(integration.type);
            $('#selektable-widget-id').val(integration.widget_id || '');
            $('#selektable-button-text').val(integration.button_text || 'Try it on');
            $('#selektable-button-class').val(integration.button_class || 'button selektable-try-on');
            $('#selektable-button-bg').val(integration.button_bg || '');
            $('#selektable-button-color').val(integration.button_color || '');
            $('#selektable-button-radius').val(integration.button_radius || '');
            $('#selektable-button-border').val(integration.button_border || '');

            // Type-specific fields
            if (integration.type === 'wc_product_page') {
                $('#selektable-placement').val(integration.placement || 'after_add_to_cart');
                $('input[name="selektable_activation_mode"][value="' + (integration.activation_mode || 'all') + '"]').prop('checked', true);
                $('#selektable-categories').val(integration.categories || []);
                $('#selektable-tags').val(integration.tags || []);
            }

            // Show correct type sections
            this.showTypeFields(integration.type);
            this.toggleActivationFields();
            this.updateShortcodePreview();
        },

        selectType: function (type) {
            $('.selektable-type-card').removeClass('selected');
            $('.selektable-type-card[data-type="' + type + '"]').addClass('selected');
            $('#selektable-int-type').val(type);
            this.showTypeFields(type);
            this.goToStep(2);
        },

        showTypeFields: function (type) {
            if (type === 'wc_product_page') {
                $('.selektable-wc-fields').show();
                $('.selektable-shortcode-fields').hide();
            } else if (type === 'shortcode') {
                $('.selektable-wc-fields').hide();
                $('.selektable-shortcode-fields').show();
                this.updateShortcodePreview();
            }
        },

        goToStep: function (step) {
            this.currentStep = step;

            if (step === 1) {
                $('#selektable-step-1').show();
                $('#selektable-step-2').hide();
                $('.selektable-modal-back').hide();
                $('.selektable-modal-save').hide();
            } else {
                $('#selektable-step-1').hide();
                $('#selektable-step-2').show();
                // Show back button only for new integrations
                if (!this.editingIntegration) {
                    $('.selektable-modal-back').show();
                } else {
                    $('.selektable-modal-back').hide();
                }
                $('.selektable-modal-save').show();
            }
        },

        toggleActivationFields: function () {
            var mode = $('input[name="selektable_activation_mode"]:checked').val();

            if (mode === 'categories') {
                $('.selektable-categories-row').show();
                $('.selektable-tags-row').hide();
            } else if (mode === 'tags') {
                $('.selektable-categories-row').hide();
                $('.selektable-tags-row').show();
            } else {
                $('.selektable-categories-row').hide();
                $('.selektable-tags-row').hide();
            }
        },

        updateShortcodePreview: function () {
            var widgetId = $('#selektable-widget-id').val();
            var shortcode = '[selektable_button';
            if (widgetId) {
                shortcode += ' widget_id="' + widgetId + '"';
            }
            shortcode += ']';
            $('#selektable-generated-shortcode').text(shortcode);
        },

        copyShortcode: function () {
            var shortcode = $('#selektable-generated-shortcode').text();
            navigator.clipboard.writeText(shortcode).then(function () {
                var $btn = $('.selektable-copy-shortcode');
                var originalHtml = $btn.html();
                $btn.html('<span class="dashicons dashicons-yes"></span> ' + selektableAdmin.i18n.copied);
                setTimeout(function () {
                    $btn.html(originalHtml);
                }, 2000);
            });
        },

        saveIntegration: function () {
            var self = this;
            var widgetId = $('#selektable-widget-id').val();

            // Validation
            if (!widgetId) {
                alert(selektableAdmin.i18n.widgetIdRequired);
                $('#selektable-widget-id').focus();
                return;
            }

            var $saveBtn = $('.selektable-modal-save');
            var originalText = $saveBtn.text();
            $saveBtn.prop('disabled', true).text(selektableAdmin.i18n.saving);

            // Collect form data
            var data = {
                action: 'selektable_save_integration',
                nonce: selektableAdmin.nonce,
                id: $('#selektable-int-id').val(),
                type: $('#selektable-int-type').val(),
                widget_id: widgetId,
                button_text: $('#selektable-button-text').val(),
                button_class: $('#selektable-button-class').val(),
                button_bg: $('#selektable-button-bg').val(),
                button_color: $('#selektable-button-color').val(),
                button_radius: $('#selektable-button-radius').val(),
                button_border: $('#selektable-button-border').val()
            };

            // Add WC Product Page specific fields
            if (data.type === 'wc_product_page') {
                data.placement = $('#selektable-placement').val();
                data.activation_mode = $('input[name="selektable_activation_mode"]:checked').val();
                data.categories = $('#selektable-categories').val() || [];
                data.tags = $('#selektable-tags').val() || [];
            }

            $.post(selektableAdmin.ajaxUrl, data, function (response) {
                if (response.success) {
                    // Update local data
                    self.updateLocalIntegrations(response.data.integration);
                    self.refreshTable();
                    self.closeModal();
                } else {
                    alert(response.data.message || selektableAdmin.i18n.error);
                }
            }).fail(function () {
                alert(selektableAdmin.i18n.error);
            }).always(function () {
                $saveBtn.prop('disabled', false).text(originalText);
            });
        },

        deleteIntegration: function (id) {
            var self = this;

            if (!confirm(selektableAdmin.i18n.confirmDelete)) {
                return;
            }

            $.post(selektableAdmin.ajaxUrl, {
                action: 'selektable_delete_integration',
                nonce: selektableAdmin.nonce,
                id: id
            }, function (response) {
                if (response.success) {
                    self.removeLocalIntegration(id);
                    self.refreshTable();
                } else {
                    alert(response.data.message || selektableAdmin.i18n.error);
                }
            }).fail(function () {
                alert(selektableAdmin.i18n.error);
            });
        },

        updateLocalIntegrations: function (integration) {
            var found = false;
            for (var i = 0; i < selektableAdmin.integrations.length; i++) {
                if (selektableAdmin.integrations[i].id === integration.id) {
                    selektableAdmin.integrations[i] = integration;
                    found = true;
                    break;
                }
            }
            if (!found) {
                selektableAdmin.integrations.push(integration);
            }
        },

        removeLocalIntegration: function (id) {
            selektableAdmin.integrations = selektableAdmin.integrations.filter(function (int) {
                return int.id !== id;
            });
        },

        refreshTable: function () {
            var $tbody = $('#selektable-integrations-table tbody');
            $tbody.empty();

            if (selektableAdmin.integrations.length === 0) {
                $tbody.append(
                    '<tr class="no-integrations"><td colspan="4">' +
                    selektableAdmin.i18n.noIntegrations +
                    '</td></tr>'
                );
                return;
            }

            var self = this;
            selektableAdmin.integrations.forEach(function (integration) {
                var typeLabel = integration.type === 'wc_product_page'
                    ? selektableAdmin.i18n.wcProductPage
                    : selektableAdmin.i18n.shortcode;

                var summary = self.getIntegrationSummary(integration);

                var $row = $(
                    '<tr data-id="' + integration.id + '">' +
                    '<td class="column-type">' + typeLabel + '</td>' +
                    '<td class="column-widget-id"><code>' + self.escapeHtml(integration.widget_id) + '</code></td>' +
                    '<td class="column-summary">' + self.escapeHtml(summary) + '</td>' +
                    '<td class="column-actions">' +
                    '<button type="button" class="button selektable-edit-integration" data-id="' + integration.id + '">Edit</button> ' +
                    '<button type="button" class="button selektable-delete-integration" data-id="' + integration.id + '">Delete</button>' +
                    '</td>' +
                    '</tr>'
                );

                $tbody.append($row);
            });
        },

        getIntegrationSummary: function (integration) {
            if (integration.type === 'shortcode') {
                return '[selektable_button widget_id="' + integration.widget_id + '"]';
            }

            var mode = integration.activation_mode || 'all';

            if (mode === 'all') {
                return 'All products';
            }

            if (mode === 'categories' && integration.categories && integration.categories.length > 0) {
                var names = [];
                integration.categories.forEach(function (catId) {
                    if (selektableAdmin.categories[catId]) {
                        names.push(selektableAdmin.categories[catId]);
                    }
                });
                return names.join(', ') + ' categories';
            }

            if (mode === 'tags' && integration.tags && integration.tags.length > 0) {
                var tagNames = [];
                integration.tags.forEach(function (tagId) {
                    if (selektableAdmin.tags[tagId]) {
                        tagNames.push(selektableAdmin.tags[tagId]);
                    }
                });
                return tagNames.join(', ') + ' tags';
            }

            return 'All products';
        },

        escapeHtml: function (text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    $(document).ready(function () {
        // Only initialize if we're on the Selektable settings page
        if ($('#selektable-integrations-table').length) {
            SelektableAdmin.init();
        }
    });

})(jQuery);
