<?php
/**
 * Plugin Name: Select Enhancer for Elementor Forms with Select2
 * Plugin URI: https://wordpress.org/plugins/select-enhancer-for-elementor-forms-with-select2/
 * Description: Adds Select2 functionality to Elementor Pro Form select fields with improved UI, multi-select support, search, and tagging capabilities.
 * Version: 1.0
 * Author: Kamrul Islam
 * Author URI: https://codewithkamrul.com
 * Text Domain: select-enhancer-for-elementor-forms-with-select2
 * Requires Plugins: elementor
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Main Plugin Class
 */
class SELESUFO_Multi_Select_Form_Select2 {

    /**
     * Plugin version
     *
     * @var string
     */
    const VERSION = '1.0';

    /**
     * Instance of this class
     *
     * @var object
     */
    private static $instance = null;

    /**
     * Get instance
     *
     * @return object
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action( 'plugins_loaded', array( $this, 'selesufo_init' ) );
    }

    /**
     * Initialize plugin
     *
     * @return void
     */
    public function selesufo_init() {
        // Check if Elementor is installed and activated
        if ( ! defined( 'ELEMENTOR_VERSION' ) || ! defined('ELEMENTOR_URL') ) {
            add_action( 'admin_notices', array( $this, 'selesufo_admin_notice_missing_elementor' ) );
            return;
        }

        
        // Enqueue frontend scripts and styles
        add_action( 'wp_enqueue_scripts', array( $this, 'selesufo_enqueue_frontend_assets' ) );

        // Enqueue preview scripts and styles
        add_action( 'elementor/editor/after_enqueue_scripts', array( $this, 'selesufo_enqueue_preview_assets' ), 99 );

        // Add custom controls to form fields
        add_action( 'elementor/element/form/section_form_fields/before_section_end', array( $this, 'selesufo_add_select2_controls' ), 10, 2 );

        // Modify select field rendering
        add_filter( 'elementor_pro/forms/render/item/select', array( $this, 'selesufo_modify_select_field_rendering' ), 99, 3 );
    }

    /**
     * Admin notice for missing Elementor
     *
     * @return void
     */
    public function selesufo_admin_notice_missing_elementor() {
        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor */
            esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'select-enhancer-for-elementor-forms-with-select2' ),
            '<strong>' . esc_html__( 'Elementor Multi Select Form With Select2', 'select-enhancer-for-elementor-forms-with-select2' ) . '</strong>',
            '<strong>' . esc_html__( 'Elementor', 'select-enhancer-for-elementor-forms-with-select2' ) . '</strong>'
        );

        printf( '<div class="notice notice-warning is-dismissible"><p>%s</p></div>', wp_kses_post( $message ) );
    }

    /**
     * Enqueue frontend assets
     *
     * @return void
     */
    public function selesufo_enqueue_frontend_assets() {


        if ( !wp_style_is( 'e-select2', 'enqueued' ) ) {

            wp_enqueue_style(
                'e-select2',
                ELEMENTOR_URL . 'assets/lib/e-select2/css/e-select2.min.css',
                array(),
                self::VERSION
            );
        }


        if ( !wp_script_is( 'e-select2', 'enqueued' ) ) {
            wp_enqueue_script(
                'e-select2',
                ELEMENTOR_URL . 'assets/lib/e-select2/js/e-select2.full.min.js',
                array( 'jquery' ),
                self::VERSION,
                true
            );
        }

        wp_register_style(
            'selesufo_elementor-msf',
            plugins_url('/assets/css/elementor-msf.css', __FILE__),
            array(),
            self::VERSION
        );
        
        wp_register_script(
            'selesufo_elementor-msf',
            plugins_url('/assets/js/elementor-msf.js', __FILE__),
            array( 'jquery', 'e-select2' ),
            self::VERSION,
            true
        ); 

        wp_enqueue_style( 'selesufo_elementor-msf' );
        wp_enqueue_script('selesufo_elementor-msf');
        
    }

    /**
     * Enqueue preview assets
     *
     * @return void
     */
    public function selesufo_enqueue_preview_assets() {

        wp_register_script(
            'selesufo_elementor-editor-msf',
            plugin_dir_url(__FILE__) . 'assets/js/elementor-msf.js',
            array( 'jquery', 'e-select2', 'elementor-editor' ),
            self::VERSION,
            true
        ); 
        
        wp_register_style(
            'selesufo_elementor-editor-msf',
            plugins_url('/assets/css/elementor-msf.css', __FILE__),
            array(),
            self::VERSION
        );

        wp_enqueue_style( 'selesufo_elementor-editor-msf' );
        wp_enqueue_style( 'e-select2' );

        wp_enqueue_script( 'selesufo_elementor-editor-msf' );
        wp_enqueue_script( 'e-select2' );
    }


    /**
     * Add Select2 controls to form fields
     *
     * @param object $element Elementor element object.
     * @param array  $args Additional arguments.
     * @return void
     */
    public function selesufo_add_select2_controls( $element, $args ) {
        // Get the form_fields repeater control
        $form_fields = $element->get_controls( 'form_fields' );

        if ( ! $form_fields || ! isset( $form_fields['fields'] ) ) {
            return;
        }

        $form_fields['fields']['enable_select2'] = array(
            'name'          => 'enable_select2',
            'label'         => esc_html__( 'Enable Select2', 'select-enhancer-for-elementor-forms-with-select2' ),
            'type'          => \Elementor\Controls_Manager::SWITCHER,
            'label_on'      => esc_html__( 'Yes', 'select-enhancer-for-elementor-forms-with-select2' ),
            'label_off'     => esc_html__( 'No', 'select-enhancer-for-elementor-forms-with-select2' ),
            'return_value'  => 'yes',
            'default'       => '',
            'condition'     => array(
                'field_type' => 'select',
            ),
            'tab'           => 'content',
            'inner_tab'     => 'form_fields_content_tab',
            'tabs_wrapper'  => 'form_fields_tabs',
        );

        $form_fields['fields']['multiple_select_placeholder'] = array(
            'name'          => 'multiple_select_placeholder',
            'label'         => esc_html__( 'Multiple Select Placeholder', 'select-enhancer-for-elementor-forms-with-select2' ),
            'type'          => \Elementor\Controls_Manager::TEXT,
            'default'       => esc_html__( 'Select', 'select-enhancer-for-elementor-forms-with-select2' ),
            'condition'     => array(
                'field_type'    => 'select',
                'enable_select2' => 'yes',
            ),
            'tab'           => 'content',
            'inner_tab'     => 'form_fields_content_tab',
            'tabs_wrapper'  => 'form_fields_tabs',
        );

        // Update the control
        $element->update_control( 'form_fields', $form_fields );
    }

    /**
     * Modify select field rendering
     *
     * @param array  $item Field item data.
     * @param int    $item_index Field index.
     * @param object $form Form widget instance.
     * @return array Modified field item data.
     */
    public function selesufo_modify_select_field_rendering( $item, $item_index, $form ) {
        if ( ! isset( $item['enable_select2'] ) || 'yes' !== $item['enable_select2'] ) {
            return $item;
        }

        $item['allow_multiple'] = true;

        $css_classes = isset( $item['css_classes'] ) ? $item['css_classes'] : '';
        $item['css_classes'] = trim( $css_classes . ' select-2-allowed' );

        // Add placeholder attribute if set
        if ( isset( $item['multiple_select_placeholder'] ) && ! empty( $item['multiple_select_placeholder'] ) ) {
            $placeholder = sanitize_text_field( $item['multiple_select_placeholder'] );
            $form->add_render_attribute(
                'select' . $item_index,
                'data-multiple-placeholder',
                esc_attr( $placeholder )
            );

            $form->add_render_attribute(
                'select' . $item_index,
                'data-select2-indexed',
                esc_attr( wp_rand(1, 10000) )
            );            
        }

        return $item;
    }
}

// Initialize the plugin
SELESUFO_Multi_Select_Form_Select2::get_instance();
