<?php
/*
Plugin Name: SEInc Backup
Plugin URI: https://seobaked.com/
Description: Backup WordPress website to a selected local folder.
Version: 1.0.2
Author: SEO Baked
Author URI: https://seobaked.com
Text Domain: seinc-backup
License: GPLv2 or later
Requires at least: 4.9
Requires PHP: 7.4

This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details. You should have received a copy of the GNU General Public License along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
    
// Add admin menu
add_action('admin_menu', 'seinc_backup_menu');

function seinc_backup_menu() {
    add_menu_page('SEInc Backup', 'SEInc Backup', 'manage_options', 'seinc-backup', 'seinc_backup_admin_page');
}

// Add this function at the beginning of your file
function seinc_backup_get_default_dir() {
    $upload_dir = wp_upload_dir();
    $backup_dir = trailingslashit($upload_dir['basedir']) . 'seinc-backups/';
    
    // Initialize WP_Filesystem
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    WP_Filesystem();
    global $wp_filesystem;
    
    // Create the backup directory if it doesn't exist
    if (!$wp_filesystem->exists($backup_dir)) {
        wp_mkdir_p($backup_dir);
    }
    
    // Create an index.php file to prevent directory listing
    $index_file = trailingslashit($backup_dir) . 'index.php';
    if (!$wp_filesystem->exists($index_file)) {
        $wp_filesystem->put_contents($index_file, '<?php // Silence is golden', FS_CHMOD_FILE);
    }
    
    return $backup_dir;
}

// Update the get_backup_path function
function seinc_backup_get_path() {
    $backup_path = get_option('seinc_backup_folder_path', '');
    if (empty($backup_path)) {
        $backup_path = seinc_backup_get_default_dir();
    }
    return trailingslashit($backup_path);
}

// Include restore functionality
require_once(plugin_dir_path(__FILE__) . 'functions/restore.php');

// Include scheduled backup functionality
require_once(plugin_dir_path(__FILE__) . 'functions/scheduled-backup.php');

// Include backup logs functionality
require_once(plugin_dir_path(__FILE__) . 'functions/backup-logs.php');

// Register activation and deactivation hooks
register_activation_hook(__FILE__, 'seinc_backup_plugin_activate');
register_deactivation_hook(__FILE__, 'seinc_backup_plugin_deactivate');

/**
 * Plugin activation handler
 */
function seinc_backup_plugin_activate() {
    global $seinc_scheduled_backup;
    if ($seinc_scheduled_backup) {
        $seinc_scheduled_backup->update_backup_schedule();
    }
}

/**
 * Plugin deactivation handler
 */
function seinc_backup_plugin_deactivate() {
    wp_clear_scheduled_hook('seinc_backup_scheduled_event');
    
    // Log the deactivation
    global $seinc_backup_logs;
    if ($seinc_backup_logs) {
        $seinc_backup_logs->add_log(
            'plugin_deactivated',
            '',
            '',
            'info',
            'Plugin deactivated - scheduled backups cleared'
        );
    }
}

// Admin page
function seinc_backup_admin_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'seinc-backup'));
    }

    // Get current tab
    $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'general';
    
    // Define tabs
    $tabs = array(
        'general' => esc_html__('General', 'seinc-backup'),
        'schedule' => esc_html__('Schedule', 'seinc-backup'),
        'logs' => esc_html__('Logs', 'seinc-backup')
    );
    
    // Allow other parts of the plugin to add tabs
    $tabs = apply_filters('seinc_backup_admin_tabs', $tabs);
    
    $current_path = seinc_backup_get_path();
    ?>
    <style>
        #directory_browser {
            border-radius: 4px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        #directory_list {
            list-style: none;
            padding: 0;
            margin: 10px 0;
            border: 1px solid #e5e5e5;
            background: #f9f9f9;
        }
        #directory_list li {
            padding: 8px 12px;
            border-bottom: 1px solid #e5e5e5;
            margin: 0;
        }
        #directory_list li:last-child {
            border-bottom: none;
        }
        #directory_list .directory-link {
            text-decoration: none;
            color: #2271b1;
            display: block;
        }
        #directory_list .directory-link:hover {
            color: #135e96;
            background: #f0f0f1;
        }
        #directory_list .not-writable {
            color: #999;
            cursor: not-allowed;
        }
        #current_path {
            font-weight: bold;
            margin: 0 0 10px 0;
            padding: 5px;
            background: #f0f0f1;
            border-radius: 3px;
        }
        .directory-browser-buttons {
            margin-top: 10px;
            padding-top: 10px;
            border-top: 1px solid #e5e5e5;
        }
        #backup_path {
            background-color: #f9f9f9;
        }
    </style>
    <div class="wrap">
        <h1><?php echo esc_html__('SEInc Backup', 'seinc-backup'); ?></h1>

        <h2 class="nav-tab-wrapper">
            <?php
            foreach ($tabs as $tab => $name) {
                $class = ($tab === $current_tab) ? ' nav-tab-active' : '';
                echo '<a class="nav-tab' . esc_attr($class) . '" href="?page=seinc-backup&tab=' . esc_attr($tab) . '">' . esc_html($name) . '</a>';
            }
            ?>
        </h2>
        
        <div class="tab-content">
            <?php
            if ($current_tab === 'schedule') {
                global $seinc_scheduled_backup;
                $seinc_scheduled_backup->render_schedule_page();
            } elseif ($current_tab === 'logs') {
                global $seinc_backup_logs;
                $seinc_backup_logs->render_logs_page();
            } else {
                ?>
                <!-- Backup Settings Form -->
                <form method="post" id="backup-form">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php echo esc_html__('Backup Directory', 'seinc-backup'); ?></th>
                            <td>
                                <input type="text" id="backup_path" name="backup_path" value="<?php echo esc_attr($current_path); ?>" class="regular-text" readonly />
                                <button type="button" id="browse_button" class="button"><?php echo esc_html__('Browse', 'seinc-backup'); ?></button>
                                
                                <!-- Directory Browser -->
                                <div id="directory_browser" style="display: none; margin-top: 10px; padding: 10px; border: 1px solid #ccd0d4; background: #fff;">
                                    <p id="current_path"></p>
                                    <ul id="directory_list" style="max-height: 300px; overflow-y: auto; margin: 10px 0;"></ul>
                                    <div class="directory-browser-buttons">
                                        <button type="button" id="select_directory" class="button button-primary"><?php echo esc_html__('Select Directory', 'seinc-backup'); ?></button>
                                        <button type="button" id="cancel_browse" class="button"><?php echo esc_html__('Cancel', 'seinc-backup'); ?></button>
                                    </div>
                                </div>
                            </td>
                        </tr>
                    </table>
                    
                    <?php wp_nonce_field('seinc_backup_start', 'backup_nonce'); ?>
                    <?php wp_nonce_field('seinc_backup_save_path', 'save_path_nonce'); ?>
                    <p class="submit">
                        <input type="submit" name="save_path" class="button button-secondary" value="<?php echo esc_attr__('Save Path', 'seinc-backup'); ?>" />
                        <input type="submit" name="create_backup" class="button button-primary" value="<?php echo esc_attr__('Create Backup', 'seinc-backup'); ?>" />
                    </p>
                </form>

                <h2><?php echo esc_html__('Existing Backups', 'seinc-backup'); ?></h2>
                <?php
                // Show restore success message if applicable
                if (isset($_GET['restored']) && $_GET['restored'] === '1') {
                    echo '<div class="notice notice-success is-dismissible"><p>' . 
                        esc_html__('Backup restored successfully!', 'seinc-backup') . 
                        '</p></div>';
                }
                ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php echo esc_html__('Backup File', 'seinc-backup'); ?></th>
                            <th><?php echo esc_html__('Size', 'seinc-backup'); ?></th>
                            <th><?php echo esc_html__('Date', 'seinc-backup'); ?></th>
                            <th><?php echo esc_html__('Actions', 'seinc-backup'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $backups = seinc_backup_list($current_path);
                        foreach ($backups as $backup) {
                            echo '<tr>';
                            echo '<td>' . esc_html($backup['name']) . '</td>';
                            echo '<td>' . esc_html($backup['size']) . '</td>';
                            echo '<td>' . esc_html($backup['date']) . '</td>';
                            echo '<td>';
                            echo '<a href="' . esc_url(wp_nonce_url(admin_url('admin-post.php?action=seinc_backup_download&file=' . urlencode($backup['name'])), 'seinc_backup_download_' . $backup['name'])) . '" class="button">' . esc_html__('Download', 'seinc-backup') . '</a> ';
                            echo '<a href="#" class="button delete-backup" data-file="' . esc_attr($backup['name']) . '">' . esc_html__('Delete', 'seinc-backup') . '</a> ';
                            
                            // Update restore form with proper nonce fields
                            echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline;">';
                            echo '<input type="hidden" name="action" value="seinc_backup_restore">';
                            echo '<input type="hidden" name="backup_file" value="' . esc_attr($backup['name']) . '">';
                            wp_nonce_field('seinc_backup_restore', 'restore_nonce');
                            echo '<input type="hidden" name="verify_nonce" value="' . esc_attr(wp_create_nonce('seinc_backup_verify')) . '">';
                            echo '<input type="submit" class="button restore-backup" value="' . esc_attr__('Restore', 'seinc-backup') . '" onclick="return confirm(\'' . esc_js(__('Are you sure you want to restore this backup? This will overwrite your current WordPress installation.', 'seinc-backup')) . '\');">';
                            echo '</form>';
                            echo '</td>';
                            echo '</tr>';
                        }
                        if (empty($backups)) {
                            echo '<tr><td colspan="4">' . esc_html__('No backups found.', 'seinc-backup') . '</td></tr>';
                        }
                        ?>
                    </tbody>
                </table>
                <?php
            }
            ?>
        </div>
    </div>
    <?php
}

function seinc_backup_list($backup_path) {
    $backups = array();
    if (is_dir($backup_path)) {
        $files = glob($backup_path . 'wp-backup-*.zip');
        foreach ($files as $file) {
            // Get file modification time
            $file_time = filemtime($file);
            
            // Convert to WordPress timezone
            $date_time = new DateTime();
            $date_time->setTimestamp($file_time);
            $date_time->setTimezone(wp_timezone());
            
            $backups[] = array(
                'name' => basename($file),
                'size' => size_format(filesize($file)),
                'date' => wp_date(get_option('date_format') . ' ' . get_option('time_format'), $file_time)
            );
        }
    }
    return $backups;
}

function seinc_backup_download() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Unauthorized access', 'seinc-backup'));
    }

    // Verify nonce
    if (!isset($_GET['file']) || !wp_verify_nonce($_GET['_wpnonce'], 'seinc_backup_download_' . $_GET['file'])) {
        wp_die(esc_html__('Invalid download request', 'seinc-backup'));
    }

    $file = sanitize_file_name(wp_unslash($_GET['file']));
    $backup_path = seinc_backup_get_path();
    $file_path = trailingslashit($backup_path) . $file;

    // Initialize WP_Filesystem
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    WP_Filesystem();
    global $wp_filesystem;

    // Check if file exists and is readable
    if (!$wp_filesystem->exists($file_path)) {
        wp_die(esc_html__('Backup file not found', 'seinc-backup'));
    }

    // Get file size
    $file_size = $wp_filesystem->size($file_path);
    if ($file_size === false || $file_size === 0) {
        wp_die(esc_html__('Invalid backup file', 'seinc-backup'));
    }

    // Clean any previous output
    while (ob_get_level()) {
        ob_end_clean();
    }

    // Set download headers
    nocache_headers();
    header('Content-Description: File Transfer');
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . basename($file_path) . '"');
    header('Content-Transfer-Encoding: binary');
    header('Content-Length: ' . $file_size);
    header('Accept-Ranges: bytes');
    header('Cache-Control: private');
    header('Pragma: private');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');

    // Log download start
    global $seinc_backup_logs;
    $seinc_backup_logs->add_log(
        'manual_download_start',
        $file,
        $backup_path,
        'info',
        'Starting backup download'
    );

    // Disable time limit for large files
    set_time_limit(0);

    // Get and output file contents
    $file_contents = $wp_filesystem->get_contents($file_path);
    if ($file_contents === false) {
        // Log download error
        $seinc_backup_logs->add_log(
            'manual_download_error',
            $file,
            $backup_path,
            'error',
            'Failed to read backup file'
        );
        wp_die(esc_html__('Failed to read backup file', 'seinc-backup'));
    }

    // Output file contents
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Binary data should not be escaped
    echo $file_contents;

    // Log successful download
    $seinc_backup_logs->add_log(
        'manual_download_complete',
        $file,
        $backup_path,
        'success',
        'Backup downloaded successfully'
    );
    exit;
}
add_action('admin_post_seinc_backup_download', 'seinc_backup_download');

function seinc_backup_delete() {
    check_ajax_referer('seinc_backup_delete', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized access', 'seinc-backup'));
    }

    $file = isset($_POST['file']) ? sanitize_file_name($_POST['file']) : '';
    if (empty($file)) {
        wp_send_json_error(esc_html__('Invalid file name', 'seinc-backup'));
    }

    $backup_path = seinc_backup_get_path();
    $file_path = trailingslashit($backup_path) . $file;

    if (!file_exists($file_path)) {
        wp_send_json_error(esc_html__('File not found:', 'seinc-backup') . ' ' . $file_path);
    }

    require_once(ABSPATH . 'wp-admin/includes/file.php');
    WP_Filesystem();

    global $seinc_backup_logs;
    
    // Log delete attempt
    $seinc_backup_logs->add_log(
        'delete_backup_start',
        $file,
        $backup_path,
        'info',
        'Attempting to delete backup file'
    );

    // Simply check if the file no longer exists after deletion attempt
    if (wp_delete_file($file_path) || !file_exists($file_path)) {
        // Log successful deletion
        $seinc_backup_logs->add_log(
            'delete_backup_success',
            $file,
            $backup_path,
            'success',
            'Backup file deleted successfully'
        );
        wp_send_json_success(esc_html__('Backup deleted successfully', 'seinc-backup'));
    } else {
        // Log failed deletion
        $seinc_backup_logs->add_log(
            'delete_backup_failed',
            $file,
            $backup_path,
            'error',
            'Failed to delete backup file'
        );
        wp_send_json_error(esc_html__('Failed to delete backup', 'seinc-backup'));
    }
}
add_action('wp_ajax_seinc_backup_delete', 'seinc_backup_delete');

function seinc_backup_create($backup_path) {
    global $wpdb, $seinc_backup_logs;
    
    // Get WordPress timezone setting
    $wp_timezone = wp_timezone();
    
    // Get the current datetime with WordPress timezone
    $current_time = current_datetime();
    $backup_filename = 'wp-backup-' . $current_time->format('Y-m-d-H-i-s') . '.zip';
    $backup_file = trailingslashit($backup_path) . $backup_filename;

    // Log backup start
    $seinc_backup_logs->add_log(
        'backup_start',
        $backup_filename,
        $backup_path,
        'info',
        'Starting backup creation'
    );

    // Create ZIP archive
    $zip = new ZipArchive();
    if ($zip->open($backup_file, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
        $seinc_backup_logs->add_log(
            'backup_error',
            $backup_filename,
            $backup_path,
            'error',
            'Failed to create ZIP archive. Make sure the backup path is writable.'
        );
        return false;
    }

    // Add WordPress files to the ZIP
    $wp_root = ABSPATH;
    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($wp_root),
        RecursiveIteratorIterator::LEAVES_ONLY
    );

    // Log files backup start
    $seinc_backup_logs->add_log(
        'backup_files',
        $backup_filename,
        $backup_path,
        'info',
        'Backing up WordPress files'
    );

    foreach ($files as $name => $file) {
        if (!$file->isDir()) {
            $filePath = $file->getRealPath();
            $relativePath = substr($filePath, strlen($wp_root));
            $zip->addFile($filePath, $relativePath);
        }
    }

    // Log database backup start
    $seinc_backup_logs->add_log(
        'backup_database',
        $backup_filename,
        $backup_path,
        'info',
        'Backing up database'
    );

    // Backup database
    $db_file = trailingslashit($backup_path) . $wpdb->dbname . '.sql';
    if (!seinc_backup_database($db_file)) {
        $zip->close();
        wp_delete_file($backup_file);
        $seinc_backup_logs->add_log(
            'backup_error',
            $backup_filename,
            $backup_path,
            'error',
            'Failed to backup database'
        );
        return false;
    }

    $zip->addFile($db_file, $wpdb->dbname . '.sql');
    $zip->close();

    // Remove database backup file
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    wp_delete_file($db_file);

    // Log successful completion
    $seinc_backup_logs->add_log(
        'backup_complete',
        $backup_filename,
        $backup_path,
        'success',
        'Backup completed successfully'
    );

    return true;
}

function seinc_backup_database($file) {
    global $wpdb;

    // Initialize WP_Filesystem
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    WP_Filesystem();
    global $wp_filesystem;

    $tables = $wpdb->get_results('SHOW TABLES', ARRAY_N);
    $output = '';

    foreach ($tables as $table) {
        $table_name = $table[0];
        // Use esc_sql() for table names in queries
        $escaped_table_name = esc_sql($table_name);
        $result = $wpdb->get_results("SELECT * FROM `$escaped_table_name`", ARRAY_N);
        $row2 = $wpdb->get_row("SHOW CREATE TABLE `$escaped_table_name`", ARRAY_N);
        $output .= "\n\n" . $row2[1] . ";\n\n";

        foreach ($result as $row) {
            $output .= 'INSERT INTO `' . $escaped_table_name . '` VALUES(';
            for ($j = 0; $j < count($row); $j++) {
                $row[$j] = addslashes($row[$j]);
                $row[$j] = str_replace("\n", "\\n", $row[$j]);
                if (isset($row[$j])) {
                    $output .= '"' . $row[$j] . '"';
                } else {
                    $output .= '""';
                }
                if ($j < (count($row) - 1)) {
                    $output .= ',';
                }
            }
            $output .= ");\n";
        }
    }

    // Use WP_Filesystem to write the file
    return $wp_filesystem->put_contents($file, $output, FS_CHMOD_FILE);
}

function seinc_backup_enqueue_scripts($hook) {
    if ('toplevel_page_seinc-backup' !== $hook) {
        return;
    }
    wp_enqueue_script('seinc-backup-js', plugins_url('js/wp-backup-custom-folder.js', __FILE__), array('jquery'), time(), true);
    wp_localize_script('seinc-backup-js', 'seincBackup', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => esc_js(wp_create_nonce('seinc_backup_browse')),
        'delete_nonce' => esc_js(wp_create_nonce('seinc_backup_delete')),
        'restore_nonce' => esc_js(wp_create_nonce('seinc_backup_verify')),
        'progress_nonce' => esc_js(wp_create_nonce('seinc_backup_progress')),
        'backup_nonce' => esc_js(wp_create_nonce('seinc_backup_start')),
        'list_nonce' => esc_js(wp_create_nonce('seinc_backup_list')),
        'i18n' => array(
            'confirmRestore' => esc_js(__('Are you sure you want to restore this backup? This will overwrite your current WordPress installation.', 'seinc-backup')),
            'verifyingBackup' => esc_js(__('Verifying backup...', 'seinc-backup')),
            'restoreButton' => esc_js(__('Restore', 'seinc-backup')),
            'verificationFailed' => esc_js(__('Backup verification failed. Please try again.', 'seinc-backup')),
            'unknownError' => esc_js(__('An unknown error occurred. Please try again.', 'seinc-backup')),
            'initializingBackup' => esc_js(__('Initializing backup...', 'seinc-backup')),
            'backingUpFiles' => esc_js(__('Backing up files...', 'seinc-backup')),
            'backingUpDatabase' => esc_js(__('Backing up database...', 'seinc-backup')),
            'backupCompleted' => esc_js(__('Backup completed successfully!', 'seinc-backup')),
            'backupError' => esc_js(__('Backup failed', 'seinc-backup')),
            /* translators: %1$s is the current file number being processed, %2$s is the total number of files to process */
            'fileProgress' => esc_js(__('Processing file %1$s of %2$s', 'seinc-backup')),
            'confirmDelete' => esc_js(__('Are you sure you want to delete this backup?', 'seinc-backup')),
            'backupInProgress' => esc_js(__('A backup is already in progress. Please wait for it to complete.', 'seinc-backup')),
            'emptyPath' => esc_js(__('Please select a backup directory first.', 'seinc-backup')),
            'savingPath' => esc_js(__('Saving backup path...', 'seinc-backup')),
            'pathSaved' => esc_js(__('Backup path saved successfully!', 'seinc-backup')),
            'pathError' => esc_js(__('Failed to save backup path. Please try again.', 'seinc-backup'))
        )
    ));
}
add_action('admin_enqueue_scripts', 'seinc_backup_enqueue_scripts');

function seinc_backup_browse_directories() {
    check_ajax_referer('seinc_backup_browse', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized access', 'seinc-backup'));
    }

    // Get the server's root path
    $root_path = '/';  // Linux systems
    if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
        $root_path = 'C:\\'; // Windows systems
    }

    $current_dir = isset($_POST['dir']) ? sanitize_text_field(wp_unslash($_POST['dir'])) : $root_path;
    $current_dir = rtrim($current_dir, '/\\') . DIRECTORY_SEPARATOR;

    // Initialize WP_Filesystem
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    WP_Filesystem();
    global $wp_filesystem;

    // Security check - prevent going above root
    $real_current = realpath($current_dir);
    if ($real_current === false) {
        wp_send_json_error(esc_html__('Invalid directory path', 'seinc-backup'));
        return;
    }

    try {
        $directories = array();
        if ($wp_filesystem->is_readable($real_current)) {
            $items = $wp_filesystem->dirlist($real_current, false, false);
            
            if (is_array($items)) {
                foreach ($items as $item) {
                    if ($item['type'] === 'd' && $item['name'] !== '.' && $item['name'] !== '..') {
                        $full_path = trailingslashit($real_current) . $item['name'];
                        $directories[] = array(
                            'name' => $item['name'],
                            'path' => $full_path,
                            'writable' => $wp_filesystem->is_writable($full_path)
                        );
                    }
                }
            }
        }

        // Sort directories
        usort($directories, function($a, $b) {
            return strcasecmp($a['name'], $b['name']);
        });

        // Get parent directory
        $parent_dir = dirname($real_current);
        if ($parent_dir === $real_current) {
            $parent_dir = $root_path;
        }

        wp_send_json_success(array(
            'current' => $real_current,
            'parent' => $parent_dir,
            'directories' => $directories
        ));

    } catch (Exception $e) {
        wp_send_json_error(esc_html__('Error reading directory:', 'seinc-backup') . ' ' . $e->getMessage());
    }
}
add_action('wp_ajax_seinc_backup_browse', 'seinc_backup_browse_directories');

function seinc_backup_check_path_writable($path) {
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    WP_Filesystem();
    global $wp_filesystem;

    return $wp_filesystem->is_writable($path);
}

// Add activation hook to create default backup directory
register_activation_hook(__FILE__, 'seinc_backup_activate');
function seinc_backup_activate() {
    seinc_backup_get_default_dir();
}

// Add deactivation hook to clean up if needed
register_deactivation_hook(__FILE__, 'seinc_backup_deactivate');
function seinc_backup_deactivate() {
    // Optionally clean up settings
    // delete_option('seinc_backup_folder_path');
}

// Add AJAX endpoint for starting backup
function seinc_backup_start_ajax() {
    check_ajax_referer('seinc_backup_start', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized access', 'seinc-backup'));
    }

    // Get backup path
    $backup_path = seinc_backup_get_path();
    
    // Start backup process
    try {
        if (seinc_backup_create($backup_path)) {
            wp_send_json_success(esc_html__('Backup completed successfully', 'seinc-backup'));
        } else {
            wp_send_json_error(esc_html__('Backup failed', 'seinc-backup'));
        }
    } catch (Exception $e) {
        global $seinc_backup_logs;
        $seinc_backup_logs->add_log(
            'backup_error',
            '',
            $backup_path,
            'error',
            'Backup failed with error: ' . $e->getMessage()
        );
        wp_send_json_error($e->getMessage());
    }
}
add_action('wp_ajax_seinc_backup_start', 'seinc_backup_start_ajax');

// Background process handler
function seinc_backup_background_process() {
    $backup_path = seinc_backup_get_path();
    seinc_backup_create($backup_path, false);
}
add_action('seinc_backup_background_process', 'seinc_backup_background_process');

// Add progress check endpoint
function seinc_backup_check_progress() {
    check_ajax_referer('seinc_backup_progress', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    
    $progress = get_option('seinc_backup_progress', array(
        'status' => 'not_started',
        'progress' => 0,
        'current_file' => '',
        'files_processed' => 0,
        'total_files' => 0
    ));
    
    // If backup is complete or has error, clear the progress data
    if ($progress['status'] === 'completed' || $progress['status'] === 'error') {
        delete_option('seinc_backup_progress');
    }
    
    wp_send_json_success($progress);
}
add_action('wp_ajax_seinc_backup_check_progress', 'seinc_backup_check_progress');

// Add backup list refresh endpoint
function seinc_get_backup_list() {
    check_ajax_referer('seinc_backup_list', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    
    $backup_path = seinc_backup_get_path();
    $backups = seinc_backup_list($backup_path);
    
    ob_start();
    if (empty($backups)) {
        echo '<tr><td colspan="4">' . esc_html__('No backups found.', 'seinc-backup') . '</td></tr>';
    } else {
        foreach ($backups as $backup) {
            ?>
            <tr>
                <td><?php echo esc_html($backup['name']); ?></td>
                <td><?php echo esc_html($backup['size']); ?></td>
                <td><?php echo esc_html($backup['date']); ?></td>
                <td>
                    <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin-post.php?action=seinc_backup_download&file=' . urlencode($backup['name'])), 'seinc_backup_download_' . $backup['name'])); ?>" class="button">
                        <?php echo esc_html__('Download', 'seinc-backup'); ?>
                    </a>
                    <button type="button" class="button delete-backup" data-file="<?php echo esc_attr($backup['name']); ?>">
                        <?php echo esc_html__('Delete', 'seinc-backup'); ?>
                    </button>
                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                        <input type="hidden" name="action" value="seinc_backup_restore">
                        <input type="hidden" name="backup_file" value="<?php echo esc_attr($backup['name']); ?>">
                        <?php wp_nonce_field('seinc_backup_restore', 'restore_nonce'); ?>
                        <input type="hidden" name="verify_nonce" value="<?php echo esc_attr(wp_create_nonce('seinc_backup_verify')); ?>">
                        <input type="submit" class="button restore-backup" value="<?php echo esc_attr__('Restore', 'seinc-backup'); ?>" 
                               onclick="return confirm('<?php echo esc_js(__('Are you sure you want to restore this backup? This will overwrite your current WordPress installation.', 'seinc-backup')); ?>');">
                    </form>
                </td>
            </tr>
            <?php
        }
    }
    $html = ob_get_clean();
    
    wp_send_json_success($html);
}
add_action('wp_ajax_seinc_get_backup_list', 'seinc_get_backup_list');

// Add the save path handler
function seinc_backup_save_path() {
    check_ajax_referer('seinc_backup_save_path', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(esc_html__('Unauthorized access', 'seinc-backup'));
    }

    $backup_path = isset($_POST['backup_path']) ? sanitize_text_field(wp_unslash($_POST['backup_path'])) : '';
    
    if (empty($backup_path)) {
        wp_send_json_error(esc_html__('Backup path cannot be empty', 'seinc-backup'));
        return;
    }

    // Initialize WP_Filesystem
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    WP_Filesystem();
    global $wp_filesystem;

    // Verify the path exists and is writable
    if (!$wp_filesystem->exists($backup_path) || !$wp_filesystem->is_dir($backup_path)) {
        wp_send_json_error(esc_html__('Invalid backup path', 'seinc-backup'));
        return;
    }

    if (!$wp_filesystem->is_writable($backup_path)) {
        wp_send_json_error(esc_html__('Backup path is not writable', 'seinc-backup'));
        return;
    }

    // Save the path
    update_option('seinc_backup_folder_path', $backup_path);
    wp_send_json_success(esc_html__('Backup path saved successfully', 'seinc-backup'));
}
add_action('wp_ajax_seinc_backup_save_path', 'seinc_backup_save_path');