<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class SEInc_Scheduled_Backup {
    private $backup_path;
    private $wpdb;
    private $wp_filesystem;

    public function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
        
        // Initialize WP_Filesystem
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        WP_Filesystem();
        global $wp_filesystem;
        $this->wp_filesystem = $wp_filesystem;
        
        // Get backup path from general settings
        $this->backup_path = get_option('seinc_backup_folder_path', '');
        if (empty($this->backup_path)) {
            $upload_dir = wp_upload_dir();
            $this->backup_path = trailingslashit($upload_dir['basedir']) . 'seinc-backups/';
        }

        // Add actions for scheduling
        add_action('seinc_backup_scheduled_event', array($this, 'perform_scheduled_backup'));
        add_action('admin_init', array($this, 'register_settings'));
        add_filter('seinc_backup_admin_tabs', array($this, 'add_schedule_tab'));

        // Handle settings changes
        add_action('update_option_seinc_backup_schedule_enabled', array($this, 'handle_schedule_change'), 10, 2);
        add_action('update_option_seinc_backup_schedule_frequency', array($this, 'handle_schedule_change'), 10, 2);
        add_action('update_option_seinc_backup_schedule_time', array($this, 'handle_schedule_change'), 10, 2);
        
        // Update backup path when general settings change
        add_action('update_option_seinc_backup_folder_path', array($this, 'update_backup_path'), 10, 2);
    }

    /**
     * Handle schedule setting changes
     */
    public function handle_schedule_change($old_value, $new_value) {
        $this->update_backup_schedule();
    }

    /**
     * Update the backup schedule based on current settings
     */
    public function update_backup_schedule() {
        // Clear existing schedule
        wp_clear_scheduled_hook('seinc_backup_scheduled_event');
        
        // If backups are enabled, schedule the next one
        if (get_option('seinc_backup_schedule_enabled', false)) {
            $frequency = get_option('seinc_backup_schedule_frequency', 'daily');
            $schedule_time = get_option('seinc_backup_schedule_time', '00:00');
            
            // Get WordPress timezone
            $wp_timezone = wp_timezone();
            
            // Get current time in WordPress timezone
            $current_time = new DateTime('now', $wp_timezone);
            
            // Create schedule time object
            list($hours, $minutes) = explode(':', $schedule_time);
            $schedule_datetime = new DateTime('today', $wp_timezone);
            $schedule_datetime->setTime((int)$hours, (int)$minutes);
            
            // If the time has already passed today, schedule for tomorrow
            if ($schedule_datetime < $current_time) {
                $schedule_datetime->modify('+1 day');
            }
            
            // Convert to timestamp
            $timestamp = $schedule_datetime->getTimestamp();
            
            wp_schedule_event($timestamp, $frequency, 'seinc_backup_scheduled_event');
            
            // Log the schedule update
            global $seinc_backup_logs;
            $seinc_backup_logs->add_log(
                'schedule_updated',
                '',
                '',
                'success',
                sprintf(
                    'Backup schedule updated: %s at %s (Server time: %s)',
                    $frequency,
                    $schedule_time,
                    $schedule_datetime->format('Y-m-d H:i:s T')
                )
            );
        }
    }

    /**
     * Update backup path when general settings change
     */
    public function update_backup_path($old_value, $new_value) {
        $this->backup_path = $new_value;
    }

    /**
     * Add Schedule tab to admin page
     */
    public function add_schedule_tab($tabs) {
        $tabs['schedule'] = esc_html__('Schedule', 'seinc-backup');
        return $tabs;
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('seinc_backup_schedule_settings', 'seinc_backup_schedule_enabled');
        register_setting('seinc_backup_schedule_settings', 'seinc_backup_schedule_frequency');
        register_setting('seinc_backup_schedule_settings', 'seinc_backup_retention_days');
        register_setting('seinc_backup_schedule_settings', 'seinc_backup_schedule_time');
    }

    /**
     * Render schedule settings page
     */
    public function render_schedule_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $enabled = get_option('seinc_backup_schedule_enabled', false);
        $frequency = get_option('seinc_backup_schedule_frequency', 'daily');
        $retention = get_option('seinc_backup_retention_days', 30);
        $schedule_time = get_option('seinc_backup_schedule_time', '00:00');

        // Get timezone and time information
        $wp_timezone = wp_timezone();
        $current_time = new DateTime('now', $wp_timezone);
        $timezone_string = $wp_timezone->getName();
        if (empty($timezone_string)) {
            $gmt_offset = get_option('gmt_offset');
            $timezone_string = ($gmt_offset >= 0 ? '+' : '') . $gmt_offset;
            $timezone_string = 'UTC' . $timezone_string;
        }

        ?>
        <div class="wrap">
            <h2><?php echo esc_html__('Scheduled Backup Settings', 'seinc-backup'); ?></h2>
            
            <div class="notice notice-info">
                <p>
                    <strong><?php echo esc_html__('Time Information:', 'seinc-backup'); ?></strong><br>
                    <?php echo esc_html__('Current WordPress Time:', 'seinc-backup'); ?> 
                    <?php echo esc_html($current_time->format('Y-m-d H:i:s')); ?><br>
                    <?php echo esc_html__('Timezone:', 'seinc-backup'); ?> 
                    <?php echo esc_html($timezone_string); ?><br>
                    <em><?php echo esc_html__('Note: Scheduled backups will run based on your WordPress timezone settings.', 'seinc-backup'); ?></em>
                </p>
            </div>

            <form method="post" action="options.php">
                <?php settings_fields('seinc_backup_schedule_settings'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php echo esc_html__('Enable Scheduled Backups', 'seinc-backup'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="seinc_backup_schedule_enabled" value="1" <?php checked($enabled, true); ?>>
                                <?php echo esc_html__('Enable automatic backups', 'seinc-backup'); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php echo esc_html__('Backup Frequency', 'seinc-backup'); ?></th>
                        <td>
                            <select name="seinc_backup_schedule_frequency">
                                <option value="hourly" <?php selected($frequency, 'hourly'); ?>><?php echo esc_html__('Hourly', 'seinc-backup'); ?></option>
                                <option value="twicedaily" <?php selected($frequency, 'twicedaily'); ?>><?php echo esc_html__('Twice Daily', 'seinc-backup'); ?></option>
                                <option value="daily" <?php selected($frequency, 'daily'); ?>><?php echo esc_html__('Daily', 'seinc-backup'); ?></option>
                                <option value="weekly" <?php selected($frequency, 'weekly'); ?>><?php echo esc_html__('Weekly', 'seinc-backup'); ?></option>
                                <option value="monthly" <?php selected($frequency, 'monthly'); ?>><?php echo esc_html__('Monthly', 'seinc-backup'); ?></option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php echo esc_html__('Backup Time', 'seinc-backup'); ?></th>
                        <td>
                            <input type="time" name="seinc_backup_schedule_time" value="<?php echo esc_attr($schedule_time); ?>">
                            <p class="description">
                                <?php echo esc_html__('Select the time of day for the backup to run (in 24-hour format).', 'seinc-backup'); ?><br>
                                <?php echo esc_html__('Time is based on your WordPress timezone setting.', 'seinc-backup'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php echo esc_html__('Backup Retention', 'seinc-backup'); ?></th>
                        <td>
                            <input type="number" name="seinc_backup_retention_days" value="<?php echo esc_attr($retention); ?>" min="1" max="365">
                            <p class="description"><?php echo esc_html__('Number of days to keep backups before they are automatically deleted.', 'seinc-backup'); ?></p>
                        </td>
                    </tr>
                </table>

                <?php 
                // Show next scheduled backup time if enabled
                if ($enabled) {
                    $next_backup = wp_next_scheduled('seinc_backup_scheduled_event');
                    if ($next_backup) {
                        $next_backup_time = new DateTime();
                        $next_backup_time->setTimestamp($next_backup);
                        $next_backup_time->setTimezone($wp_timezone);
                        
                        echo '<div class="notice notice-success inline"><p>';
                        echo esc_html__('Next backup scheduled for: ', 'seinc-backup') . ' ';
                        echo esc_html($next_backup_time->format('F j, Y g:i a T'));
                        echo '</p></div>';
                    }
                }
                ?>

                <?php submit_button(esc_html__('Save Schedule Settings', 'seinc-backup')); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Perform scheduled backup
     */
    public function perform_scheduled_backup() {
        if (!get_option('seinc_backup_schedule_enabled', false)) {
            return;
        }

        // Get the latest backup path from options
        $this->backup_path = get_option('seinc_backup_folder_path', $this->backup_path);

        // Create backup using existing backup function
        if (function_exists('seinc_backup_create')) {
            // Get current time in GMT
            $backup_time = gmdate('Y-m-d H:i:s');
            $backup_filename = 'wp-backup-' . gmdate('Y-m-d-H-i-s') . '.zip';
            
            // Log backup start
            global $seinc_backup_logs;
            $seinc_backup_logs->add_log(
                'scheduled_backup_start',
                $backup_filename,
                $this->backup_path,
                'info',
                'Starting scheduled backup to: ' . $this->backup_path
            );
            
            try {
                seinc_backup_create($this->backup_path);
                
                // Log successful backup
                $seinc_backup_logs->add_log(
                    'scheduled_backup_complete',
                    $backup_filename,
                    $this->backup_path,
                    'success',
                    'Scheduled backup completed successfully'
                );
            } catch (Exception $e) {
                // Log backup failure
                $seinc_backup_logs->add_log(
                    'scheduled_backup_failed',
                    $backup_filename,
                    $this->backup_path,
                    'error',
                    'Scheduled backup failed: ' . $e->getMessage()
                );
            }
        }

        // Clean up old backups
        $this->cleanup_old_backups();
    }

    /**
     * Clean up old backups based on retention period
     */
    private function cleanup_old_backups() {
        $retention_days = get_option('seinc_backup_retention_days', 30);
        $cutoff_time = time() - ($retention_days * DAY_IN_SECONDS);

        $backup_files = glob($this->backup_path . 'wp-backup-*.zip');
        if (is_array($backup_files)) {
            foreach ($backup_files as $file) {
                if (filemtime($file) < $cutoff_time) {
                    $filename = basename($file);
                    if ($this->wp_filesystem->delete($file)) {
                        // Log successful deletion
                        global $seinc_backup_logs;
                        $seinc_backup_logs->add_log(
                            'backup_deleted',
                            $filename,
                            $this->backup_path,
                            'success',
                            'Old backup deleted by retention policy'
                        );
                    }
                }
            }
        }
    }
}

// Initialize the scheduled backup functionality
$seinc_scheduled_backup = new SEInc_Scheduled_Backup(); 