<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class SEInc_Backup_Restore {
    private $backup_path;
    private $temp_dir;
    private $wpdb;
    private $wp_filesystem;

    public function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
        
        // Initialize WP_Filesystem
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        WP_Filesystem();
        global $wp_filesystem;
        $this->wp_filesystem = $wp_filesystem;
        
        // Get backup path
        $this->backup_path = seinc_backup_get_path();
        $this->temp_dir = $this->backup_path . 'temp_restore/';
    }

    /**
     * Initialize the restore functionality
     */
    public function init() {
        add_action('admin_post_seinc_backup_restore', array($this, 'handle_restore'));
        add_action('wp_ajax_seinc_backup_verify', array($this, 'verify_backup'));
    }

    /**
     * Verify backup file before restoration
     */
    public function verify_backup() {
        // Verify nonce
        if (!check_ajax_referer('seinc_backup_verify', 'nonce', false)) {
            wp_send_json_error(esc_html__('Security check failed', 'seinc-backup'));
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(esc_html__('Unauthorized access', 'seinc-backup'));
            return;
        }

        $backup_file = isset($_POST['file']) ? sanitize_file_name(wp_unslash($_POST['file'])) : '';
        if (empty($backup_file)) {
            wp_send_json_error(esc_html__('No backup file specified', 'seinc-backup'));
            return;
        }

        $backup_path = trailingslashit($this->backup_path) . $backup_file;
        
        if (!$this->wp_filesystem->exists($backup_path)) {
            wp_send_json_error(esc_html__('Backup file not found', 'seinc-backup'));
            return;
        }

        if (!class_exists('ZipArchive')) {
            wp_send_json_error(esc_html__('ZipArchive not available on your server', 'seinc-backup'));
            return;
        }

        $zip = new ZipArchive();
        $result = $zip->open($backup_path);
        if ($result !== true) {
            /* translators: %s: Numeric error code returned by ZipArchive::open() */
            wp_send_json_error(sprintf(esc_html__('Failed to open backup file. Error code: %s', 'seinc-backup'), $result));
            return;
        }

        // Check for required files
        $has_sql = false;
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $filename = $zip->getNameIndex($i);
            if (strpos($filename, '.sql') !== false) {
                $has_sql = true;
                break;
            }
        }
        $zip->close();

        if (!$has_sql) {
            wp_send_json_error(esc_html__('Invalid backup file: Missing database backup', 'seinc-backup'));
            return;
        }

        wp_send_json_success(array(
            'message' => esc_html__('Backup file verified successfully', 'seinc-backup')
        ));
    }

    /**
     * Handle the restore process
     */
    public function handle_restore() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized access', 'seinc-backup'));
        }

        // Verify both nonces
        if (!check_admin_referer('seinc_backup_restore', 'restore_nonce')) {
            wp_die(esc_html__('Invalid restore nonce', 'seinc-backup'));
        }

        $verify_nonce = isset($_POST['verify_nonce']) ? sanitize_text_field(wp_unslash($_POST['verify_nonce'])) : '';
        if (empty($verify_nonce) || !wp_verify_nonce($verify_nonce, 'seinc_backup_verify')) {
            wp_die(esc_html__('Invalid verification nonce', 'seinc-backup'));
        }

        $backup_file = isset($_POST['backup_file']) ? sanitize_file_name(wp_unslash($_POST['backup_file'])) : '';
        if (empty($backup_file)) {
            wp_die(esc_html__('No backup file specified', 'seinc-backup'));
        }

        $backup_path = trailingslashit($this->backup_path) . $backup_file;
        
        if (!$this->wp_filesystem->exists($backup_path)) {
            wp_die(esc_html__('Backup file not found', 'seinc-backup'));
        }

        global $seinc_backup_logs;
        
        // Log restore start
        $seinc_backup_logs->add_log(
            'restore_start',
            $backup_file,
            $this->backup_path,
            'info',
            'Starting backup restoration'
        );

        // Create temp directory
        if (!$this->wp_filesystem->exists($this->temp_dir)) {
            $this->wp_filesystem->mkdir($this->temp_dir);
        }

        try {
            // Extract backup
            $zip = new ZipArchive();
            if ($zip->open($backup_path) !== true) {
                $this->cleanup();
                $seinc_backup_logs->add_log(
                    'restore_error',
                    $backup_file,
                    $this->backup_path,
                    'error',
                    'Failed to open backup file'
                );
                wp_die(esc_html__('Failed to open backup file', 'seinc-backup'));
            }

            $zip->extractTo($this->temp_dir);
            $zip->close();

            // Log files extraction
            $seinc_backup_logs->add_log(
                'restore_progress',
                $backup_file,
                $this->backup_path,
                'info',
                'Backup files extracted successfully'
            );

            // Restore database first
            $sql_files = glob($this->temp_dir . '*.sql');
            if (empty($sql_files)) {
                $this->cleanup();
                $seinc_backup_logs->add_log(
                    'restore_error',
                    $backup_file,
                    $this->backup_path,
                    'error',
                    'No database backup found in archive'
                );
                wp_die(esc_html__('No database backup found in archive', 'seinc-backup'));
            }

            $sql_content = $this->wp_filesystem->get_contents($sql_files[0]);
            if ($sql_content === false) {
                $this->cleanup();
                $seinc_backup_logs->add_log(
                    'restore_error',
                    $backup_file,
                    $this->backup_path,
                    'error',
                    'Failed to read database backup file'
                );
                wp_die(esc_html__('Failed to read database backup file', 'seinc-backup'));
            }

            // Log database restore start
            $seinc_backup_logs->add_log(
                'restore_progress',
                $backup_file,
                $this->backup_path,
                'info',
                'Starting database restoration'
            );

            // Drop all existing tables
            $tables = $this->wpdb->get_results('SHOW TABLES', ARRAY_N);
            foreach ($tables as $table) {
                $table_name = $table[0];
                $this->wpdb->query("DROP TABLE IF EXISTS `$table_name`");
            }

            // Split SQL into individual queries
            $delimiter = ';';
            $sql_queries = array();
            $current_query = '';
            
            foreach (explode("\n", $sql_content) as $line) {
                $line = trim($line);
                if (empty($line) || strpos($line, '--') === 0 || strpos($line, '#') === 0) {
                    continue;
                }
                
                $current_query .= $line;
                if (substr($line, -strlen($delimiter)) === $delimiter) {
                    $sql_queries[] = $current_query;
                    $current_query = '';
                } else {
                    $current_query .= ' ';
                }
            }

            // Execute queries with error handling
            foreach ($sql_queries as $query) {
                $query = trim($query);
                if (!empty($query)) {
                    // These queries are from our backup file and include CREATE TABLE and INSERT statements
                    // that cannot be prepared without breaking their structure
                    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Reason: Executing backup file queries that are already properly escaped
                    $result = $this->wpdb->query($query);
                    
                    if ($result === false) {
                        // Re-enable foreign key checks before exiting
                        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
                        $this->wpdb->query('SET foreign_key_checks = 1');
                        
                        $this->cleanup();
                        $seinc_backup_logs->add_log(
                            'restore_error',
                            $backup_file,
                            $this->backup_path,
                            'error',
                            'Database restore failed: ' . $this->wpdb->last_error
                        );
                        wp_die(sprintf(
                            /* translators: %s: MySQL error message returned by $wpdb->last_error */
                            esc_html__('Database restore failed. MySQL Error: %s', 'seinc-backup'),
                            esc_html($this->wpdb->last_error)
                        ));
                    }
                }
            }

            // Log database restore completion
            $seinc_backup_logs->add_log(
                'restore_progress',
                $backup_file,
                $this->backup_path,
                'info',
                'Database restored successfully'
            );

            // Log files restore start
            $seinc_backup_logs->add_log(
                'restore_progress',
                $backup_file,
                $this->backup_path,
                'info',
                'Starting files restoration'
            );

            // Restore files
            $this->restore_files();

            // Log files restore completion
            $seinc_backup_logs->add_log(
                'restore_progress',
                $backup_file,
                $this->backup_path,
                'info',
                'Files restored successfully'
            );

            // Cleanup
            $this->cleanup();

            // Clear object cache
            wp_cache_flush();

            // Log restore completion
            $seinc_backup_logs->add_log(
                'restore_complete',
                $backup_file,
                $this->backup_path,
                'success',
                'Backup restored successfully'
            );

            // Redirect back with success message
            wp_safe_redirect(add_query_arg(
                array(
                    'page' => 'seinc-backup',
                    'restored' => '1'
                ),
                admin_url('admin.php')
            ));
            exit;

        } catch (Exception $e) {
            $this->cleanup();
            $seinc_backup_logs->add_log(
                'restore_error',
                $backup_file,
                $this->backup_path,
                'error',
                'Restore failed: ' . $e->getMessage()
            );
            wp_die(esc_html__('Failed to restore backup: ', 'seinc-backup') . esc_html($e->getMessage()));
        }
    }

    /**
     * Restore WordPress files
     */
    private function restore_files() {
        $wordpress_dir = ABSPATH;
        $exclude = array(
            '.',
            '..',
            basename($this->backup_path),
            'wp-config.php',
            '*.sql' // Exclude SQL files from being copied
        );
        
        // Copy files from temp directory to WordPress directory
        $this->copy_directory($this->temp_dir, $wordpress_dir, $exclude);
    }

    /**
     * Copy directory recursively
     */
    private function copy_directory($source, $destination, $exclude = array()) {
        $source = trailingslashit($source);
        $destination = trailingslashit($destination);

        if (!$this->wp_filesystem->is_dir($source)) {
            return false;
        }

        $dir_handle = opendir($source);
        if ($dir_handle === false) {
            return false;
        }

        while (($file = readdir($dir_handle)) !== false) {
            // Check for exact matches and wildcard patterns
            $skip = false;
            foreach ($exclude as $pattern) {
                if (strpos($pattern, '*') !== false) {
                    // Handle wildcard pattern
                    $regex = '/^' . str_replace('*', '.*', $pattern) . '$/i';
                    if (preg_match($regex, $file)) {
                        $skip = true;
                        break;
                    }
                } elseif ($file === $pattern) {
                    // Handle exact match
                    $skip = true;
                    break;
                }
            }
            
            if ($skip) {
                continue;
            }

            $source_file = $source . $file;
            $destination_file = $destination . $file;

            if (is_dir($source_file)) {
                if (!$this->wp_filesystem->exists($destination_file)) {
                    $this->wp_filesystem->mkdir($destination_file);
                }
                $this->copy_directory($source_file, $destination_file, $exclude);
            } else {
                $this->wp_filesystem->copy($source_file, $destination_file, true);
            }
        }

        closedir($dir_handle);
        return true;
    }

    /**
     * Cleanup temporary files
     */
    private function cleanup() {
        if ($this->wp_filesystem->exists($this->temp_dir)) {
            $this->wp_filesystem->delete($this->temp_dir, true);
        }
    }
}

// Initialize the restore functionality
$seinc_backup_restore = new SEInc_Backup_Restore();
$seinc_backup_restore->init(); 