<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class SEInc_Backup_Logs {
    private $log_option = 'seinc_backup_logs';
    private $max_logs = 100; // Maximum number of logs to keep

    public function __construct() {
        add_filter('seinc_backup_admin_tabs', array($this, 'add_logs_tab'));
    }

    /**
     * Add Logs tab to admin page
     */
    public function add_logs_tab($tabs) {
        $tabs['logs'] = esc_html__('Logs', 'seinc-backup');
        return $tabs;
    }

    /**
     * Add a log entry
     */
    public function add_log($action, $filename, $path, $status = 'success', $message = '') {
        $logs = $this->get_logs();
        
        $log_entry = array(
            'timestamp' => current_time('mysql', false),
            'action' => $action,
            'filename' => $filename,
            'path' => $path,
            'status' => $status,
            'message' => $message
        );

        array_unshift($logs, $log_entry);
        
        // Keep only the latest logs
        if (count($logs) > $this->max_logs) {
            $logs = array_slice($logs, 0, $this->max_logs);
        }

        update_option($this->log_option, $logs);
    }

    /**
     * Get all logs
     */
    public function get_logs() {
        return get_option($this->log_option, array());
    }

    /**
     * Clear all logs
     */
    public function clear_logs() {
        delete_option($this->log_option);
    }

    /**
     * Render logs page
     */
    public function render_logs_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Handle clear logs action
        if (isset($_POST['clear_logs']) && check_admin_referer('seinc_backup_clear_logs')) {
            $this->clear_logs();
            echo '<div class="notice notice-success"><p>' . esc_html__('Logs cleared successfully.', 'seinc-backup') . '</p></div>';
        }

        $logs = $this->get_logs();
        ?>
        <div class="wrap">
            <h2><?php echo esc_html__('Backup Logs', 'seinc-backup'); ?></h2>
            
            <form method="post" action="">
                <?php wp_nonce_field('seinc_backup_clear_logs'); ?>
                <input type="submit" name="clear_logs" class="button" value="<?php echo esc_attr__('Clear Logs', 'seinc-backup'); ?>" 
                       onclick="return confirm('<?php echo esc_js(__('Are you sure you want to clear all logs?', 'seinc-backup')); ?>');">
            </form>
            
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('Date/Time', 'seinc-backup'); ?></th>
                        <th><?php echo esc_html__('Action', 'seinc-backup'); ?></th>
                        <th><?php echo esc_html__('File', 'seinc-backup'); ?></th>
                        <th><?php echo esc_html__('Path', 'seinc-backup'); ?></th>
                        <th><?php echo esc_html__('Status', 'seinc-backup'); ?></th>
                        <th><?php echo esc_html__('Message', 'seinc-backup'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($logs)): ?>
                        <tr>
                            <td colspan="6"><?php echo esc_html__('No logs found.', 'seinc-backup'); ?></td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($logs as $log): ?>
                            <tr>
                                <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($log['timestamp']))); ?></td>
                                <td><?php echo esc_html($log['action']); ?></td>
                                <td><?php echo esc_html($log['filename']); ?></td>
                                <td><?php echo esc_html($log['path']); ?></td>
                                <td>
                                    <span class="log-status <?php echo esc_attr($log['status']); ?>">
                                        <?php echo esc_html($log['status']); ?>
                                    </span>
                                </td>
                                <td><?php echo esc_html($log['message']); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
            
            <style>
                .log-status {
                    padding: 3px 8px;
                    border-radius: 3px;
                    font-weight: bold;
                }
                .log-status.success {
                    background-color: #dff0d8;
                    color: #3c763d;
                }
                .log-status.error {
                    background-color: #f2dede;
                    color: #a94442;
                }
            </style>
        </div>
        <?php
    }
}

// Initialize the logs functionality
global $seinc_backup_logs;
$seinc_backup_logs = new SEInc_Backup_Logs(); 