<?php
/*
Plugin Name: seedtrace
Plugin URI: https://seedtrace.org/
Description: Official seedtrace API integration plugin.
Author: seedtrace
Version: 1.9
Text Domain: seedtrace-transparency
*/

const SEEDTRACE_API_URL    = 'https://prd-api.seedtrace.org/external';
const SEEDTRACE_WIDGET_URL = 'https://products.seedtrace.org/api/widget.js';
define( 'SEEDTRACE_PLUGIN_PATH', plugins_url( '', __FILE__ ) );

class SeedTrace_API {

	private $settings;

	/**
	 * Initialize functions
	 */
	public function __construct() {
		// Init Functions
		add_action( 'init', [ $this, 'save_settings' ] );
		add_action( 'init', [ $this, 'load_settings' ] );

		// Include Styles and Scripts
		add_action( 'admin_enqueue_scripts', [ $this, 'admin_scripts_and_styles' ] );
		add_action( 'wp_enqueue_scripts', [ $this, 'include_scripts_and_styles' ], 99 );

		// Admin menu
		add_action( 'admin_menu', [ $this, 'register_menu' ] );

		// Create shortcode
		add_shortcode( 'seedtrace', [ $this, 'create_shortcode' ] );

		// Visual Hooks
		add_action( 'woocommerce_after_add_to_cart_button', [ $this, 'show_after_add_to_cart' ], 9999);
	}

	/**
	 * Load Saved Settings
	 */
	public function load_settings() {
		$this->settings = (array) json_decode( get_option( 'seedtrace' ), true );
	}

	/**
	 * Create ShortCode
	 */
	public function create_shortcode( $args ) {

		if ( isset( $args['product-id'] ) ) {
			// Get ID from Shortcode Params
			$api_product = $args['product-id'];
		} elseif ( is_product() ) {
			global $post;
			// Search Product ID in Settings
			foreach ( $this->settings['product'] as $item => $product_id ) {
				if ( is_object( $post ) && $product_id == $post->ID ) {
					$api_product = $this->settings['api_product'][ $item ];
				}
			}
		}

		// Include Script if found Settings for Product
		if ( isset( $api_product ) ) {
			ob_start();
			include_once( 'templates/frontend/shortcode.php' );
			return ob_get_clean();
		}
	}

	/**
	 * Show Shortcode After Add to Cart
	 */
	public function show_after_add_to_cart() {
		if ( isset( $this->settings['mode'] ) && $this->settings['mode'] == 'after_cart' ) {
			echo do_shortcode( '[seedtrace]' );
		}
	}

	/**
	 * Add Settings to Admin Menu
	 */
	public function register_menu() {
		add_submenu_page( 'woocommerce', 'seedtrace', 'seedtrace', 'manage_options', 'seed_trace_api', function () {
			// Include Global Variables
			global $wp_version, $wpdb;

			// Load Settings
			$settings = $this->settings;

			// Get Data from API
			$api_products = $this->get_api_data( $settings, 'POST' );

			// Get Products from WooCommerce
			$args = array(
				'post_type'      => 'product',
				'orderby'        => 'desc',
				'posts_per_page' => - 1
			);

			$products = get_posts( $args );

			include 'templates/admin/settings.php';
		} );
	}

	/**
	 * Save data from Settings to Option
	 */
	public function save_settings() {
		if ( isset( $_POST['seedtrace'] ) && is_array( $_POST['seedtrace'] ) ) {
			$tmp_products = [];
			if ( (array) $_POST['seedtrace']['product'] ) {
				foreach ( $_POST['seedtrace']['product'] as $tmp_key => $tmp_val ) {
					$tmp_products[ (int) $tmp_key ] = sanitize_text_field( $tmp_val );
				}
			}

			$tmp_api_products = [];
			if ( (array) $_POST['seedtrace']['api_product'] ) {
				foreach ( $_POST['seedtrace']['api_product'] as $tmp_key => $tmp_val ) {
					$tmp_api_products[ (int) $tmp_key ] = sanitize_text_field( $tmp_val );
				}
			}

			$data = [
				'api_key'     => sanitize_text_field( $_POST['seedtrace']['api_key'] ),
				'mode'        => sanitize_text_field( $_POST['seedtrace']['mode'] ),
				'product'     => $tmp_products,
				'api_product' => $tmp_api_products,
			];

			update_option( 'seedtrace', json_encode( $data ) );
		}
	}

	/**
	 * CURL POST Data from API
	 */
	public function get_api_data( $settings, $method ) {
		if ( ! isset( $settings['api_key'] ) || ! strlen( $settings['api_key'] ) ) {
			return [];
		}

		// Request POST Data
		$post_data = [
			'query' => '
                 query GetProducts {
                  companyProducts(first: 10) {
                    edges {
                      cursor
                      node {
                        id
                        slug
                        title
                        publishedTimestamp
                        createdTimestamp
                        image {
                          url
                        }
                      }
                    }
                    pageInfo {
                      hasNextPage
                      hasPreviousPage
                    }
                  }
                }'
		];

		// Send data to API
		$response = wp_remote_post( SEEDTRACE_API_URL, array(
			'method'  => $method,
			"timeout" => 45,
			'headers' => [
				"Content-type"  => "application/json",
				"Accept"        => "application/json",
				"Authorization" => "Bearer {$settings['api_key']}"
			],
			'body'    => json_encode( $post_data )
		) );

		return json_decode( $response['body'], true );
	}

	/**
	 * Include Scripts And Styles on Admin Pages
	 */
	public function admin_scripts_and_styles() {
		// Register styles
		wp_enqueue_style( 'wpm-core-selectstyle', plugins_url( 'templates/libs/selectstyle/selectstyle.css', __FILE__ ) );
		wp_enqueue_style( 'wpm-font-awesome', plugins_url( 'templates/libs/font-awesome/scripts/all.min.css', __FILE__ ) );
		wp_enqueue_style( 'wpm-core-tips', plugins_url( 'templates/libs/tips/tips.css', __FILE__ ) );
		wp_enqueue_style( 'wpm-core-select2', plugins_url( 'templates/libs/select2/select2.min.css', __FILE__ ) );
		wp_enqueue_style( 'wpm-core-admin', plugins_url( 'templates/assets/css/admin.css', __FILE__ ) );

		// Register Scripts
		wp_enqueue_script( 'wpm-core-selectstyle', plugins_url( 'templates/libs/selectstyle/selectstyle.js', __FILE__ ) );
		wp_enqueue_script( 'wpm-font-awesome', plugins_url( 'templates/libs/font-awesome/scripts/all.min.js', __FILE__ ) );
		wp_enqueue_script( 'wpm-core-tips', plugins_url( 'templates/libs/tips/tips.js', __FILE__ ) );
		wp_enqueue_script( 'wpm-core-select2', plugins_url( 'templates/libs/select2/select2.min.js', __FILE__ ) );
		wp_enqueue_script( 'wpm-core-admin', plugins_url( 'templates/assets/js/admin.js', __FILE__ ) );
	}

	/**
	 * Include Scripts And Styles on FrontEnd
	 */
	public function include_scripts_and_styles() {
		// Register styles
		wp_enqueue_style( 'wpm-core', plugins_url( 'templates/assets/css/frontend.css', __FILE__ ), false, '1.0.0', 'all' );

		// Register scripts
		wp_register_script( 'wpm-core', plugins_url( 'templates/assets/js/frontend.js', __FILE__ ), array( 'jquery' ), '1.0.0', 'all' );
		wp_localize_script( 'wpm-core', 'admin', array( 'ajaxurl' => admin_url( 'admin-ajax.php' ) ) );
		wp_enqueue_script( 'wpm-core' );
	}
}

new SeedTrace_API();
