<?php
/**
 * Core plugin functionality.
 *
 * @package SecuritytxtManager
 */

namespace SecuritytxtManager\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use const SecuritytxtManager\Constants\CAPABILITY;

/**
 * Default setup routine
 *
 * @return void
 */
function setup() {
	add_action( 'init', __NAMESPACE__ . '\\display_security_txt' );
	add_action( 'admin_init', __NAMESPACE__ . '\\add_capability' );
	add_filter( 'security_txt_content', __NAMESPACE__ . '\\maybe_add_credits' );
}

/**
 * Handle security.txt requests
 *
 * @return void
 */
function display_security_txt() {
	$request = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : false;

	$allowed_requests = [ '/security.txt', '/.well-known/security.txt' ];

	if ( in_array( $request, $allowed_requests, true ) || '/security.txt?' === substr( $request, 0, 14 ) || '/.well-known/security.txt?' === substr( $request, 0, 26 ) ) {
		$settings = \SecuritytxtManager\Utils\get_settings();
		if ( ! empty( $settings['content'] ) ) {
			header( 'Content-Type: text/plain' );
			echo esc_html( apply_filters( 'security_txt_content', $settings['content'] ) );
			exit;
		}
	}
}

/**
 * Add credits info to security.txt output
 *
 * @param string $content Security.txt content
 *
 * @return mixed|string
 */
function maybe_add_credits( $content ) {
	$settings = \SecuritytxtManager\Utils\get_settings();
	if ( $settings['credits'] ) {
		$content .= PHP_EOL;
		$content .= esc_html__( '# Generated by Security.txt Manager: https://wordpress.org/plugins/security-txt-manager', 'security-txt-manager' );
	}

	return $content;
}

/**
 * Map the capability administrator role
 *
 * @return void
 */
function add_capability() {
	$role = get_role( 'administrator' );
	if ( ! $role->has_cap( CAPABILITY ) ) {
		$role->add_cap( CAPABILITY );
	}
}

/**
 * Remove capability on deactivation
 *
 * @return void
 */
function deactivate() {
	$role = get_role( 'administrator' );
	$role->remove_cap( CAPABILITY );
}
