<?php
/**
 * Plugin Name: SecurelyWP – all-in-one security
 * Description: WordPress security with hardening, headers, CAPTCHA, firewall, and two-factor authentication.
 * Version: 1.0.10
 * Author: <a href="https://github.com/ashar-official">mdashar</a> | <a href="https://securelywp.com">SecurelyWP</a>
 * License: GPL-3.0
 * Text Domain: securelywp
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SECURELYWP_VERSION', '1.0.10');
define('SECURELYWP_PATH', plugin_dir_path(__FILE__));
define('SECURELYWP_URL', esc_url(plugin_dir_url(__FILE__)));

// Load modules
require_once SECURELYWP_PATH . 'includes/captcha/captcha.php';
require_once SECURELYWP_PATH . 'includes/hardening/hide-wp-version.php';
require_once SECURELYWP_PATH . 'includes/hardening/disable-php-uploads.php';
require_once SECURELYWP_PATH . 'includes/hardening/prevent-user-enum.php';
require_once SECURELYWP_PATH . 'includes/hardening/detect-admin-username.php';
require_once SECURELYWP_PATH . 'includes/hardening/disable-file-edit.php';
require_once SECURELYWP_PATH . 'includes/hardening/force-https.php';
require_once SECURELYWP_PATH . 'includes/hardening/brute-force-lite.php';
require_once SECURELYWP_PATH . 'includes/headers/permissions-policy.php';
require_once SECURELYWP_PATH . 'includes/headers/csp.php';
require_once SECURELYWP_PATH . 'includes/headers/hsts.php';
require_once SECURELYWP_PATH . 'includes/headers/x-frame-options.php';
require_once SECURELYWP_PATH . 'includes/headers/referrer-policy.php';
require_once SECURELYWP_PATH . 'includes/headers/x-content-type-options.php';
require_once SECURELYWP_PATH . 'includes/two-factor/email-codes.php';
require_once SECURELYWP_PATH . 'includes/two-factor/totp.php';
require_once SECURELYWP_PATH . 'includes/two-factor/backup-codes.php';
require_once SECURELYWP_PATH . 'includes/two-factor/render.php';
require_once SECURELYWP_PATH . 'includes/firewall/firewall.php';
require_once SECURELYWP_PATH . 'admin/dashboard.php';
require_once SECURELYWP_PATH . 'admin/settings-pages/system-details.php';
require_once SECURELYWP_PATH . 'admin/settings-pages/hardening-settings.php';
require_once SECURELYWP_PATH . 'admin/settings-pages/captcha-settings.php';
require_once SECURELYWP_PATH . 'admin/settings-pages/headers-settings.php';
require_once SECURELYWP_PATH . 'admin/settings-pages/two-factor-settings.php';
require_once SECURELYWP_PATH . 'admin/settings-pages/firewall-settings.php';
require_once SECURELYWP_PATH . 'includes/cache/purge-cache.php';

class SecurelyWP {

    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('admin_menu', [$this, 'register_admin_menus']);
    }

    public function register_admin_menus() {
        add_submenu_page(
            'securelywp-dashboard',
            esc_html__('Firewall Settings', 'securelywp'),
            esc_html__('Firewall Settings', 'securelywp'),
            'manage_options',
            'securelywp-firewall-settings',
            'securelywp_firewall_settings_page'
        );
        add_submenu_page(
            'securelywp-dashboard',
            esc_html__('Security Headers', 'securelywp'),
            esc_html__('Security Headers', 'securelywp'),
            'manage_options',
            'securelywp-headers-settings',
            'securelywp_headers_settings_page'
        );
        add_submenu_page(
            'profile.php',
            esc_html__('Two-Factor Authentication', 'securelywp'),
            esc_html__('Two-Factor Authentication', 'securelywp'),
            'read',
            'securelywp-two-factor-settings',
            'securelywp_two_factor_settings_page'
        );
        add_submenu_page(
            'securelywp-dashboard',
            esc_html__('Security Hardening', 'securelywp'),
            esc_html__('Security Hardening', 'securelywp'),
            'manage_options',
            'securelywp-hardening-settings',
            'securelywp_hardening_settings_page'
        );
        add_submenu_page(
            'securelywp-dashboard',
            esc_html__('CAPTCHA Settings', 'securelywp'),
            esc_html__('CAPTCHA Settings', 'securelywp'),
            'manage_options',
            'securelywp-captcha-settings',
            'securelywp_captcha_settings_page'
        );
        add_submenu_page(
            'securelywp-dashboard',
            esc_html__('System Details', 'securelywp'),
            esc_html__('System Details', 'securelywp'),
            'manage_options',
            'securelywp-system-details',
            'securelywp_system_details_page'
        );
    }

    public function enqueue_assets($hook) {
        if ($hook !== 'profile.php' && strpos($hook, 'securelywp') === false) {
            return;
        }
        wp_enqueue_style(
            'securelywp-admin-css',
            esc_url(SECURELYWP_URL . 'assets/css/admin.css'),
            [],
            SECURELYWP_VERSION
        );
        wp_enqueue_script(
            'jquery-ui-dialog',
            esc_url(SECURELYWP_URL . 'assets/js/jquery-ui-dialog.js'),
            ['jquery', 'jquery-ui-core'],
            SECURELYWP_VERSION,
            true
        );
        if ($hook === 'profile.php' || strpos($hook, 'securelywp-two-factor-settings') !== false) {
            wp_enqueue_script(
                'securelywp-qrcode-js',
                esc_url(SECURELYWP_URL . 'assets/two-factor/js/qrcode.js'),
                [],
                SECURELYWP_VERSION,
                true
            );
            wp_localize_script(
                'securelywp-qrcode-js',
                'securelywp_2fa',
                [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('securelywp_2fa_nonce'),
                ]
            );
        }
    }

    public function activate() {
        $default_options = [];
        $default_hardening_options = [
            'hide_wp_version' => true,
            'disable_php_uploads' => true,
            'prevent_user_enum' => true,
            'detect_admin_username' => true,
            'disable_file_edit' => true,
            'force_https' => true,
            'brute_force_protection' => true,
        ];
        $default_headers_options = [
            'csp_active' => true,
            'csp' => 'upgrade-insecure-requests;',
            'csp_report_uri' => '',
            'hsts_active' => true,
            'hsts_max_age' => 31536000,
            'hsts_include_subdomains' => true,
            'hsts_preload' => true,
            'x_frame_options_active' => true,
            'x_frame_options' => 'SAMEORIGIN',
            'x_frame_options_allow_from_url' => '',
            'referrer_policy_active' => true,
            'referrer_policy' => 'strict-origin-when-cross-origin',
            'permissions_policy_active' => true,
            'permissions_policy' => 'accelerometer=(), autoplay=(), camera=(), cross-origin-isolated=(), display-capture=(self), encrypted-media=(), fullscreen=*, geolocation=(self), gyroscope=(), keyboard-map=(), magnetometer=(), microphone=(), midi=(), payment=*, picture-in-picture=*, publickey-credentials-get=(), screen-wake-lock=(), sync-xhr=*, usb=(), xr-spatial-tracking=(), gamepad=(), serial=()',
            'x_content_type_options_active' => true,
            'x_content_type_options' => true,
        ];
        $default_2fa_options = [
            'enforce_2fa_network' => false,
        ];
        $default_firewall_options = [
            'enable_firewall' => true,
            'custom_rules' => '',
            'whitelist' => '',
        ];

        update_option('securelywp_options', array_map('sanitize_text_field', $default_options));
        update_option('securelywp_hardening_options', array_map('sanitize_text_field', $default_hardening_options));
        update_option('securelywp_headers_options', array_map(function($value) {
            return is_bool($value) ? filter_var($value, FILTER_VALIDATE_BOOLEAN) : (is_numeric($value) ? absint($value) : ($value === '' ? '' : sanitize_text_field($value)));
        }, $default_headers_options));
        update_option('securelywp_2fa_options', array_map('sanitize_text_field', $default_2fa_options));
        update_option('securelywp_firewall_options', array_map('sanitize_text_field', $default_firewall_options));

        // Initialize default 2FA user meta for all users
        $users = get_users();
        foreach ($users as $user) {
            if (!get_user_meta($user->ID, 'securelywp_2fa_user_options', true)) {
                update_user_meta($user->ID, 'securelywp_2fa_user_options', [
                    'enable_totp' => false,
                    'totp_secret' => '',
                    'enable_email_2fa' => false,
                    'enable_recovery_codes' => false,
                    'recovery_codes' => [],
                    'primary_2fa_method' => 'totp',
                    'totp_verified' => false,
                ]);
            }
        }
    }

    public function deactivate() {
        delete_option('securelywp_2fa_options');
        delete_option('securelywp_firewall_options');
        delete_option('securelywp_blocked_requests');
        // Note: User meta is preserved to avoid accidental data loss
    }
}

new SecurelyWP();

add_action('plugins_loaded', function () {
    load_plugin_textdomain('securelywp', false, dirname(plugin_basename(__FILE__)) . '/languages/');
});
