<?php
/**
 * SecurelyWP 2FA Render Handler
 *
 * Handles rendering and validation of the 2FA challenge form integrated with the WordPress login form.
 *
 * @package SecurelyWP
 * @since 1.0.8
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Hook into login form to add 2FA fields.
 */
add_action('login_form', function () {
    $user_id = get_transient('securelywp_2fa_user_' . session_id());
    if (!$user_id) {
        return;
    }

    $user = get_userdata($user_id);
    if (!$user) {
        delete_transient('securelywp_2fa_user_' . session_id());
        return;
    }

    $user_2fa_options = get_user_meta($user_id, 'securelywp_2fa_user_options', true) ?: [];
    $global_2fa_options = get_option('securelywp_2fa_options', []);
    $enforce_2fa = is_multisite() && ($global_2fa_options['enforce_2fa_network'] ?? false);

    // Determine if 2FA is required
    $require_2fa = $enforce_2fa || 
                   ($user_2fa_options['enable_totp'] && $user_2fa_options['totp_verified']) || 
                   $user_2fa_options['enable_email_2fa'] || 
                   $user_2fa_options['enable_recovery_codes'];

    if (!$require_2fa) {
        delete_transient('securelywp_2fa_user_' . session_id());
        return;
    }

    // Select primary method, fallback to available methods if primary is not configured
    $method = $user_2fa_options['primary_2fa_method'] ?? 'totp';
    if ($method === 'totp' && (!$user_2fa_options['enable_totp'] || !$user_2fa_options['totp_verified'] || empty($user_2fa_options['totp_secret']))) {
        $method = $user_2fa_options['enable_email_2fa'] ? 'email' : ($user_2fa_options['enable_recovery_codes'] ? 'backup_codes' : 'none');
    } elseif ($method === 'email' && !$user_2fa_options['enable_email_2fa']) {
        $method = ($user_2fa_options['enable_totp'] && $user_2fa_options['totp_verified']) ? 'totp' : ($user_2fa_options['enable_recovery_codes'] ? 'backup_codes' : 'none');
    } elseif ($method === 'backup_codes' && !$user_2fa_options['enable_recovery_codes']) {
        $method = ($user_2fa_options['enable_totp'] && $user_2fa_options['totp_verified']) ? 'totp' : ($user_2fa_options['enable_email_2fa'] ? 'email' : 'none');
    }

    if ($method === 'none') {
        delete_transient('securelywp_2fa_user_' . session_id());
        return;
    }

    // Send email code if applicable
    if ($method === 'email' && $user_2fa_options['enable_email_2fa']) {
        securelywp_generate_email_code($user_id);
    }

    $error = isset($_POST['securelywp_2fa_code']) && !isset($_POST['securelywp_resend_code']) ? esc_html__('Invalid 2FA code. Please try again.', 'securelywp') : '';
    ?>
    <div class="securelywp-2fa-challenge" style="margin-top: 20px;">
        <p>
            <label for="securelywp_2fa_code">
                <?php
                if ($method === 'totp') {
                    esc_html_e('Enter the code from your authenticator app:', 'securelywp');
                } elseif ($method === 'email') {
                    esc_html_e('Enter the code sent to your email:', 'securelywp');
                } else {
                    esc_html_e('Enter a recovery code:', 'securelywp');
                }
                ?>
            </label>
            <input type="text" name="securelywp_2fa_code" id="securelywp_2fa_code" class="input" value="" size="20" placeholder="123456" autocomplete="off">
        </p>
        <?php if ($error): ?>
            <p class="error"><?php echo esc_html($error); ?></p>
        <?php endif; ?>
        <?php if ($method === 'email'): ?>
            <p>
                <input type="submit" name="securelywp_resend_code" class="button button-secondary" value="<?php esc_attr_e('Resend Code', 'securelywp'); ?>">
            </p>
        <?php endif; ?>
        <?php wp_nonce_field('securelywp_2fa_challenge', 'securelywp_2fa_nonce'); ?>
    </div>
    <?php
});

/**
 * Hook into login to check for 2FA.
 */
add_filter('authenticate', function ($user, $username, $password) {
    if (is_wp_error($user)) {
        return $user;
    }

    $user_2fa_options = get_user_meta($user->ID, 'securelywp_2fa_user_options', true) ?: [];
    $global_2fa_options = get_option('securelywp_2fa_options', []);
    $enforce_2fa = is_multisite() && ($global_2fa_options['enforce_2fa_network'] ?? false);

    // Check if 2FA is required
    $require_2fa = $enforce_2fa || 
                   ($user_2fa_options['enable_totp'] && $user_2fa_options['totp_verified']) || 
                   $user_2fa_options['enable_email_2fa'] || 
                   $user_2fa_options['enable_recovery_codes'];

    if ($require_2fa) {
        set_transient('securelywp_2fa_user_' . session_id(), $user->ID, 5 * MINUTE_IN_SECONDS);
        // If 2FA code is provided, verify it
        if (isset($_POST['securelywp_2fa_code']) && isset($_POST['securelywp_2fa_nonce']) && wp_verify_nonce($_POST['securelywp_2fa_nonce'], 'securelywp_2fa_challenge')) {
            $code = sanitize_text_field($_POST['securelywp_2fa_code']);
            $method = $user_2fa_options['primary_2fa_method'] ?? 'totp';

            // Fallback logic for primary method
            if ($method === 'totp' && (!$user_2fa_options['enable_totp'] || !$user_2fa_options['totp_verified'] || empty($user_2fa_options['totp_secret']))) {
                $method = $user_2fa_options['enable_email_2fa'] ? 'email' : ($user_2fa_options['enable_recovery_codes'] ? 'backup_codes' : 'none');
            } elseif ($method === 'email' && !$user_2fa_options['enable_email_2fa']) {
                $method = ($user_2fa_options['enable_totp'] && $user_2fa_options['totp_verified']) ? 'totp' : ($user_2fa_options['enable_recovery_codes'] ? 'backup_codes' : 'none');
            } elseif ($method === 'backup_codes' && !$user_2fa_options['enable_recovery_codes']) {
                $method = ($user_2fa_options['enable_totp'] && $user_2fa_options['totp_verified']) ? 'totp' : ($user_2fa_options['enable_email_2fa'] ? 'email' : 'none');
            }

            if ($method === 'none') {
                delete_transient('securelywp_2fa_user_' . session_id());
                return $user;
            }

            $valid = false;
            if ($method === 'totp' && $user_2fa_options['enable_totp'] && $user_2fa_options['totp_verified'] && !empty($user_2fa_options['totp_secret'])) {
                $valid = securelywp_verify_totp_code($user_2fa_options['totp_secret'], $code);
            } elseif ($method === 'email' && $user_2fa_options['enable_email_2fa']) {
                $valid = securelywp_verify_email_code($user->ID, $code);
            } elseif ($method === 'backup_codes' && $user_2fa_options['enable_recovery_codes']) {
                $valid = securelywp_verify_recovery_code($user->ID, $code);
            }

            if ($valid) {
                delete_transient('securelywp_2fa_user_' . session_id());
                return $user;
            } else {
                return new WP_Error('invalid_2fa_code', esc_html__('Invalid 2FA code. Please try again.', 'securelywp'));
            }
        } elseif (isset($_POST['securelywp_resend_code']) && isset($_POST['securelywp_2fa_nonce']) && wp_verify_nonce($_POST['securelywp_2fa_nonce'], 'securelywp_2fa_challenge')) {
            if ($user_2fa_options['enable_email_2fa']) {
                securelywp_generate_email_code($user->ID);
            }
            return new WP_Error('2fa_required', esc_html__('Please enter your 2FA code.', 'securelywp'));
        } else {
            return new WP_Error('2fa_required', esc_html__('Please enter your 2FA code.', 'securelywp'));
        }
    }

    return $user;
}, 100, 3);

/**
 * Admin reset 2FA for a user.
 */
add_action('wp_ajax_securelywp_reset_2fa', function () {
    check_ajax_referer('securelywp_2fa_nonce', 'nonce');
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('Insufficient permissions.', 'securelywp')]);
    }

    $user_id = absint($_POST['user_id']);
    delete_user_meta($user_id, 'securelywp_2fa_user_options');
    wp_send_json_success(['message' => esc_html__('2FA reset successfully.', 'securelywp')]);
});