<?php
/**
 * SecurelyWP Cache Purger
 * Adds a "Purge Cache" button to WordPress admin bar
 */

if (!defined('ABSPATH')) {
    exit;
}

class SecurelyWP_Cache_Purger {

    private $cache_version_key = 'securelywp_cache_version';

    public function __construct() {
        add_action('admin_bar_menu', [$this, 'add_purge_cache_button'], 100);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('wp_enqueue_scripts', [$this, 'force_asset_refresh']);
        add_action('admin_enqueue_scripts', [$this, 'force_admin_asset_refresh']);
        add_action('wp_ajax_securelywp_purge_cache', [$this, 'ajax_purge_cache']);
        add_action('admin_notices', [$this, 'show_purge_notice']);
    }

    /**
     * Add Purge Cache button to admin bar
     */
    public function add_purge_cache_button($wp_admin_bar) {
        if (!current_user_can('manage_options')) {
            return;
        }

        $wp_admin_bar->add_node([
            'id' => 'securelywp-purge-cache',
            'title' => '🗑️ Purge Cache',
            'parent' => 'top-secondary',
            'href' => '#',
            'meta' => [
                'class' => 'securelywp-purge-cache-btn',
                'onclick' => 'securelywpPurgeCache(); return false;'
            ]
        ]);
    }

    /**
     * Enqueue JavaScript for AJAX functionality
     */
    public function enqueue_scripts() {
        if (!current_user_can('manage_options')) {
            return;
        }

        wp_enqueue_script('jquery');

        wp_add_inline_script('jquery', '
            function securelywpPurgeCache() {
                if (!confirm("Are you sure you want to purge all cache layers? This action cannot be undone.")) {
                    return;
                }

                jQuery.ajax({
                    url: ajaxurl,
                    type: "POST",
                    data: {
                        action: "securelywp_purge_cache",
                        nonce: "' . wp_create_nonce('securelywp_purge_cache_nonce') . '"
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert("Cache purge failed: " + (response.data || "Unknown error"));
                        }
                    },
                    error: function() {
                        alert("Cache purge failed due to network error");
                    }
                });
            }
        ');
    }

    /**
     * AJAX handler for cache purging
     */
    public function ajax_purge_cache() {
        // Verify capability
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'securelywp'));
        }

        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'securelywp_purge_cache_nonce')) {
            wp_die(__('Security check failed', 'securelywp'));
        }

        // Purge all cache layers
        $this->purge_all_caches();

        // Force asset refresh for current session
        $this->force_asset_refresh();

        // Set success notice
        set_transient('securelywp_cache_purged', time(), 30);

        wp_send_json_success('Cache purged successfully');
    }

    /**
     * Purge all possible cache layers
     */
    private function purge_all_caches() {
        // WordPress internal cache
        wp_cache_flush();

        // Object cache (Redis/Memcached)
        if (function_exists('wp_cache_init')) {
            wp_cache_init();
        }

        // Delete all transients
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_site_transient_%'");

        // Clean expired transients
        delete_expired_transients(true);

        // Opcode cache (PHP OPcache)
        if (function_exists('opcache_reset')) {
            opcache_reset();
        }

        // Plugin-specific caches
        $this->purge_plugin_caches();

        // Update cache version for asset refresh
        update_option($this->cache_version_key, time());
    }

    /**
     * Purge caches from popular caching plugins
     */
    private function purge_plugin_caches() {
        // WP Super Cache
        if (function_exists('wp_cache_clear_cache')) {
            wp_cache_clear_cache();
        }

        // W3 Total Cache
        if (function_exists('w3tc_flush_all')) {
            w3tc_flush_all();
        }

        // LiteSpeed Cache
        if (class_exists('LiteSpeed_Cache')) {
            LiteSpeed_Cache::get_instance()->purge_all();
        }

        // WP Rocket
        if (function_exists('rocket_clean_domain')) {
            rocket_clean_domain();
        }

        // WP Fastest Cache
        if (class_exists('WpFastestCache')) {
            $wpfc = new WpFastestCache();
            $wpfc->deleteCache();
        }

        // Cache Enabler
        if (class_exists('Cache_Enabler')) {
            Cache_Enabler::clear_cache();
        }

        // Autoptimize
        if (class_exists('autoptimizeCache')) {
            autoptimizeCache::clearall();
        }

        // Hummingbird
        if (class_exists('WP_Hummingbird')) {
            WP_Hummingbird::flush_cache();
        }

        // Swift Performance
        if (class_exists('Swift_Performance_Cache')) {
            Swift_Performance_Cache::clear_all_cache();
        }

        // Comet Cache
        if (class_exists('comet_cache')) {
            comet_cache::clear();
        }

        // Hyper Cache
        if (function_exists('hyper_cache_clear')) {
            hyper_cache_clear();
        }
    }

    /**
     * Force browser cache refresh by updating asset versions
     */
    public function force_asset_refresh() {
        $cache_version = get_option($this->cache_version_key, time());
        if ($cache_version !== time()) {
            update_option($this->cache_version_key, time());
        }
    }

    /**
     * Force admin asset refresh
     */
    public function force_admin_asset_refresh() {
        $this->force_asset_refresh();
    }

    /**
     * Show purge success notice
     */
    public function show_purge_notice() {
        if ($notice_time = get_transient('securelywp_cache_purged')) {
            delete_transient('securelywp_cache_purged');
            ?>
            <div class="notice notice-success is-dismissible">
                <p><?php _e('All cache layers have been purged successfully.', 'securelywp'); ?></p>
            </div>
            <?php
        }
    }
}

// Initialize the cache purger
new SecurelyWP_Cache_Purger();
