<?php
/**
 * SecurelyWP System Details Page
 *
 * Displays detailed system information for the WordPress installation.
 *
 * @package SecurelyWP
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Render the system details page.
 *
 * @return void
 */
function securelywp_system_details_page() {
    // Gather system status data
    global $wpdb;
    $site_url = esc_url_raw(site_url());
    $wp_url = esc_url_raw(home_url());
    $wp_install_dir = wp_normalize_path(ABSPATH);
    $wp_version = get_bloginfo('version');
    $wp_language = get_locale();
    $is_multisite = is_multisite() ? 'Yes' : 'No';
    $active_theme = esc_url_raw(get_template_directory_uri());
    $parent_theme = wp_get_theme()->parent() ? esc_url_raw(get_template_directory_uri()) : $active_theme;
    $user_roles = array_keys($GLOBALS['wp_roles']->roles);
    $user_roles = !empty($user_roles) ? implode(', ', $user_roles) : 'None';
    $mu_plugins = array_keys(get_mu_plugins());
    $mu_plugins = !empty($mu_plugins) ? implode(', ', $mu_plugins) : 'None';
    $php_version = phpversion();
    $web_server = isset($_SERVER['SERVER_SOFTWARE']) ? sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE'])) : 'Unknown';
    $server_os = php_uname('s') . ' ' . php_uname('r') . ' ' . php_uname('m');
    $server_address = isset($_SERVER['SERVER_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['SERVER_ADDR'])) : 'Unknown';
    $server_port = isset($_SERVER['SERVER_PORT']) ? absint($_SERVER['SERVER_PORT']) : 'Unknown';

    // File count and directories
    $all_files = [];
    if (is_dir(ABSPATH)) {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator(ABSPATH, FilesystemIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $all_files[] = wp_normalize_path($file->getPathname());
            }
        }
    }
    $total_files = count($all_files);
    $dir_counts = [];
    $dirs = [
        wp_normalize_path(ABSPATH . 'blog/'),
        wp_normalize_path(ABSPATH . 'wp-admin/'),
        wp_normalize_path(ABSPATH . 'wp-content/'),
        wp_normalize_path(ABSPATH . 'wp-content/plugins/'),
        wp_normalize_path(ABSPATH . 'wp-content/uploads/'),
        wp_normalize_path(ABSPATH . 'wp-includes/')
    ];
    foreach ($dirs as $dir) {
        if (is_dir($dir)) {
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS),
                RecursiveIteratorIterator::SELF_FIRST
            );
            $count = 0;
            foreach ($files as $file) {
                if ($file->isFile()) {
                    $count++;
                }
            }
            $dir_counts[$dir] = $count;
        }
    }

    // Hidden files
    $hidden_files = [];
    $root_files = scandir(ABSPATH);
    foreach ($root_files as $file) {
        if (strpos($file, '.') === 0 && is_file(ABSPATH . $file)) {
            $hidden_files[] = '[FILE] ' . wp_normalize_path(ABSPATH . $file);
        }
    }
    $sub_dirs = [
        wp_normalize_path(ABSPATH . 'wp-content/'),
        wp_normalize_path(ABSPATH . 'blog/'),
        wp_normalize_path(ABSPATH . 'wp-content/plugins/')
    ];
    foreach ($sub_dirs as $dir) {
        if (is_dir($dir)) {
            $sub_files = scandir($dir);
            foreach ($sub_files as $file) {
                if (in_array($file, ['.htaccess', '.gitignore']) && is_file($dir . $file)) {
                    $hidden_files[] = '[FILE] ' . wp_normalize_path($dir . $file);
                }
            }
        }
    }

    // Logged-in users
    $logged_in_users = [];
    $current_user = wp_get_current_user();
    $sessions = WP_Session_Tokens::get_instance(get_current_user_id())->get_all();
    foreach ($sessions as $session) {
        $logged_in_users[] = [
            'ip' => sanitize_text_field($session['ip']),
            'ua' => sanitize_text_field($session['ua']),
            'start' => gmdate('c', absint($session['login'])),
            'expiration' => gmdate('c', absint($session['expiration']))
        ];
    }

    ?>
    <div class="wrap securelywp-dashboard securelywp-system-details">
        <h1><?php esc_html_e('System Details', 'securelywp'); ?></h1>

        <div class="securelywp-card">
            <h2><?php esc_html_e('System Details', 'securelywp'); ?></h2>
            <table class="wp-list-table widefat fixed">
                <tbody>
                    <tr><th><?php esc_html_e('Website URL', 'securelywp'); ?></th><td><?php echo esc_html($site_url); ?></td></tr>
                    <tr><th><?php esc_html_e('WP URL', 'securelywp'); ?></th><td><?php echo esc_html($wp_url); ?></td></tr>
                    <tr><th><?php esc_html_e('WP Installation DIR', 'securelywp'); ?></th><td><?php echo esc_html($wp_install_dir); ?></td></tr>
                    <tr><th><?php esc_html_e('WP Version', 'securelywp'); ?></th><td><?php echo esc_html($wp_version); ?></td></tr>
                    <tr><th><?php esc_html_e('WP Language', 'securelywp'); ?></th><td><?php echo esc_html($wp_language); ?></td></tr>
                    <tr><th><?php esc_html_e('WP Multisite', 'securelywp'); ?></th><td><?php echo esc_html($is_multisite); ?></td></tr>
                    <tr><th><?php esc_html_e('Active Theme', 'securelywp'); ?></th><td><?php echo esc_html($active_theme); ?></td></tr>
                    <tr><th><?php esc_html_e('Parent Theme', 'securelywp'); ?></th><td><?php echo esc_html($parent_theme); ?></td></tr>
                    <tr><th><?php esc_html_e('User Roles', 'securelywp'); ?></th><td><?php echo esc_html($user_roles); ?></td></tr>
                    <tr><th><?php esc_html_e('Must-Use Plugins', 'securelywp'); ?></th><td><?php echo esc_html($mu_plugins); ?></td></tr>
                    <tr><th><?php esc_html_e('PHP Version', 'securelywp'); ?></th><td><?php echo esc_html($php_version); ?></td></tr>
                    <tr><th><?php esc_html_e('Web Server', 'securelywp'); ?></th><td><?php echo esc_html($web_server); ?></td></tr>
                    <tr><th><?php esc_html_e('Server OS', 'securelywp'); ?></th><td><?php echo esc_html($server_os); ?></td></tr>
                    <tr><th><?php esc_html_e('Server Address', 'securelywp'); ?></th><td><?php echo esc_html($server_address); ?></td></tr>
                    <tr><th><?php esc_html_e('Server Port', 'securelywp'); ?></th><td><?php echo esc_html($server_port); ?></td></tr>
                    <tr><th><?php esc_html_e('Total Files', 'securelywp'); ?></th><td><?php echo esc_html($total_files); ?></td></tr>
                    <tr><th colspan="2"><?php esc_html_e('Directory File Counts', 'securelywp'); ?></th></tr>
                    <?php foreach ($dir_counts as $dir => $count) : ?>
                        <tr><td><?php echo esc_html($dir); ?></td><td><?php echo esc_html($count); ?></td></tr>
                    <?php endforeach; ?>
                    <tr><th colspan="2"><?php esc_html_e('Hidden Files & Folders', 'securelywp'); ?></th></tr>
                    <?php if (empty($hidden_files)) : ?>
                        <tr><td colspan="2"><?php esc_html_e('None', 'securelywp'); ?></td></tr>
                    <?php else : ?>
                        <?php foreach ($hidden_files as $file) : ?>
                            <tr><td colspan="2"><?php echo esc_html($file); ?></td></tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    <tr><th colspan="2"><?php esc_html_e('Logged-In Users', 'securelywp'); ?></th></tr>
                    <tr><td><?php esc_html_e('User ID', 'securelywp'); ?></td><td><?php echo esc_html($current_user->ID); ?></td></tr>
                    <tr><td><?php esc_html_e('User Roles', 'securelywp'); ?></td><td><?php echo esc_html(implode(', ', $current_user->roles)); ?></td></tr>
                    <tr><td><?php esc_html_e('User Login', 'securelywp'); ?></td><td><?php echo esc_html($current_user->user_login); ?></td></tr>
                    <tr><td><?php esc_html_e('User Email', 'securelywp'); ?></td><td><?php echo esc_html($current_user->user_email); ?></td></tr>
                    <tr><td><?php esc_html_e('Display Name', 'securelywp'); ?></td><td><?php echo esc_html($current_user->display_name); ?></td></tr>
                    <tr><td><?php esc_html_e('Registered', 'securelywp'); ?></td><td><?php echo esc_html($current_user->user_registered); ?></td></tr>
                    <?php if (!empty($logged_in_users)) : ?>
                        <?php foreach ($logged_in_users as $session) : ?>
                            <tr><td><?php esc_html_e('IP Address', 'securelywp'); ?></td><td><?php echo esc_html($session['ip']); ?></td></tr>
                            <tr><td><?php esc_html_e('User-Agent', 'securelywp'); ?></td><td><?php echo esc_html($session['ua']); ?></td></tr>
                            <tr><td><?php esc_html_e('Session Start', 'securelywp'); ?></td><td><?php echo esc_html($session['start']); ?></td></tr>
                            <tr><td><?php esc_html_e('Session Expiration', 'securelywp'); ?></td><td><?php echo esc_html($session['expiration']); ?></td></tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr><td colspan="2"><?php esc_html_e('No active sessions found.', 'securelywp'); ?></td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php
}