<?php
/**
 * SecurelyWP Dashboard
 *
 * Handles admin dashboard, menu, and AJAX status updates.
 *
 * @package SecurelyWP
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class SecurelyWP_Dashboard {
    public static function init() {
        add_action('admin_menu', [__CLASS__, 'register_menu']);
        add_action('wp_ajax_securelywp_get_dashboard_status', [__CLASS__, 'ajax_get_dashboard_status']);
    }

    public static function register_menu() {
        add_menu_page(
            esc_html__('SecurelyWP Dashboard', 'securelywp'),
            esc_html__('SecurelyWP', 'securelywp'),
            'manage_options',
            'securelywp-dashboard',
            [__CLASS__, 'render'],
            'dashicons-shield',
            30
        );
    }

    public static function render() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'securelywp'));
        }

        $hardening_options = get_option('securelywp_hardening_options', []);
        $headers_options = get_option('securelywp_headers_options', []);
        $captcha_options = securelywp_captcha_get_settings();
        $hardening_enabled_count = count(array_filter($hardening_options));
        $hardening_total = 7;
        $hardening_indicator_class = $hardening_enabled_count === $hardening_total ? 'enabled' : ($hardening_enabled_count > 0 ? 'enabled' : 'disabled');
        $headers_enabled_count = count(array_filter([
            $headers_options['csp_active'],
            $headers_options['hsts_active'],
            $headers_options['x_frame_options_active'],
            $headers_options['referrer_policy_active'],
            $headers_options['permissions_policy_active'],
            $headers_options['x_content_type_options_active'],
        ]));
        $headers_total = 6;
        $headers_indicator_class = $headers_enabled_count === $headers_total ? 'enabled' : ($headers_enabled_count > 0 ? 'enabled' : 'disabled');
        $captcha_enabled_count = count(array_filter($captcha_options, 'is_bool'));
        $captcha_total = count($captcha_options) - 2; // Exclude site_key, secret_key
        $captcha_configured = !empty($captcha_options['site_key']) && !empty($captcha_options['secret_key']);
        $captcha_indicator_class = $captcha_configured && $captcha_enabled_count > 0 ? 'enabled' : 'disabled';

        $twofa_options = get_option('securelywp_2fa_options', []);
        $users_with_2fa = count(get_users(['meta_key' => 'securelywp_2fa_user_options', 'meta_compare' => 'EXISTS']));
        $twofa_enabled_count = $users_with_2fa;
        $twofa_total = count(get_users());
        $twofa_indicator_class = $twofa_enabled_count > 0 || ($twofa_options['enforce_2fa_network'] ?? false) ? 'enabled' : 'disabled';
        $twofa_status = $twofa_enabled_count > 0 || ($twofa_options['enforce_2fa_network'] ?? false) ? esc_html__('Enabled', 'securelywp') : esc_html__('Disabled', 'securelywp');

        ?>
        <div class="wrap securelywp-dashboard">
            <h1><?php esc_html_e('SecurelyWP Dashboard', 'securelywp'); ?></h1>
            <div class="securelywp-dashboard-grid">
                <div class="securelywp-card" data-feature="system-details">
                    <h2><?php esc_html_e('System Details', 'securelywp'); ?></h2>
                    <div class="feature-status">
                        <span class="status-indicator enabled"></span>
                        <span><?php esc_html_e('Active', 'securelywp'); ?></span>
                    </div>
                    <ul>
                        <li><strong><?php esc_html_e('PHP Version:', 'securelywp'); ?></strong>
                            <span class="dynamic-data" data-field="php_version"><?php echo esc_html(phpversion()); ?></span>
                        </li>
                        <li><strong><?php esc_html_e('WP Version:', 'securelywp'); ?></strong>
                            <span class="dynamic-data" data-field="wp_version"><?php echo esc_html(get_bloginfo('version')); ?></span>
                        </li>
                    </ul>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=securelywp-system-details')); ?>" class="button"><?php esc_html_e('View Details', 'securelywp'); ?></a>
                </div>
                <div class="securelywp-card" data-feature="hardening-settings">
                    <h2><?php esc_html_e('Hardening Settings', 'securelywp'); ?></h2>
                    <div class="feature-status">
                        <span class="status-indicator <?php echo esc_attr($hardening_indicator_class); ?>"></span>
                        <span class="dynamic-data" data-field="hardening_status">
                            <?php echo $hardening_enabled_count === $hardening_total ? esc_html__('All Enabled', 'securelywp') : esc_html__("$hardening_enabled_count/$hardening_total Enabled", 'securelywp'); ?>
                        </span>
                    </div>
                    <ul>
                        <li><strong><?php esc_html_e('Features:', 'securelywp'); ?></strong>
                            <span class="dynamic-data" data-field="hardening_count"><?php echo esc_html("$hardening_enabled_count/$hardening_total"); ?></span>
                        </li>
                    </ul>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=securelywp-hardening-settings')); ?>" class="button"><?php esc_html_e('Manage', 'securelywp'); ?></a>
                </div>
                <div class="securelywp-card" data-feature="headers-settings">
                    <h2><?php esc_html_e('Security Headers', 'securelywp'); ?></h2>
                    <div class="feature-status">
                        <span class="status-indicator <?php echo esc_attr($headers_indicator_class); ?>"></span>
                        <span class="dynamic-data" data-field="headers_status">
                            <?php echo $headers_enabled_count === $headers_total ? esc_html__('All Enabled', 'securelywp') : esc_html__("$headers_enabled_count/$headers_total Enabled", 'securelywp'); ?>
                        </span>
                    </div>
                    <ul>
                        <li><strong><?php esc_html_e('Headers:', 'securelywp'); ?></strong>
                            <span class="dynamic-data" data-field="headers_count"><?php echo esc_html("$headers_enabled_count/$headers_total"); ?></span>
                        </li>
                    </ul>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=securelywp-headers-settings')); ?>" class="button"><?php esc_html_e('Manage', 'securelywp'); ?></a>
                </div>
                <div class="securelywp-card" data-feature="captcha-settings">
                    <h2><?php esc_html_e('CAPTCHA Settings', 'securelywp'); ?></h2>
                    <div class="feature-status">
                        <span class="status-indicator <?php echo esc_attr($captcha_indicator_class); ?>"></span>
                        <span class="dynamic-data" data-field="captcha_status">
                            <?php echo $captcha_configured && $captcha_enabled_count > 0 ? esc_html__('Enabled', 'securelywp') : esc_html__('Disabled', 'securelywp'); ?>
                        </span>
                    </div>
                    <ul>
                        <li><strong><?php esc_html_e('Forms Protected:', 'securelywp'); ?></strong>
                            <span class="dynamic-data" data-field="captcha_count"><?php echo esc_html("$captcha_enabled_count/$captcha_total"); ?></span>
                        </li>
                        <li><strong><?php esc_html_e('Configured:', 'securelywp'); ?></strong>
                            <span class="dynamic-data" data-field="captcha_configured"><?php echo $captcha_configured ? esc_html__('Yes', 'securelywp') : esc_html__('No', 'securelywp'); ?></span>
                        </li>
                    </ul>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=securelywp-captcha-settings')); ?>" class="button"><?php esc_html_e('Manage', 'securelywp'); ?></a>
                </div>
                <div class="securelywp-card" data-feature="twofa-settings">
                    <h2><?php esc_html_e('Two-Factor Authentication', 'securelywp'); ?></h2>
                    <div class="feature-status">
                        <span class="status-indicator <?php echo esc_attr($twofa_indicator_class); ?>"></span>
                        <span class="dynamic-data" data-field="twofa_status">
                            <?php echo esc_html($twofa_status); ?>
                        </span>
                    </div>
                    <ul>
                        <li><strong><?php esc_html_e('Users with 2FA:', 'securelywp'); ?></strong>
                            <span class="dynamic-data" data-field="twofa_count"><?php echo esc_html("$twofa_enabled_count/$twofa_total"); ?></span>
                        </li>
                        <li><strong><?php esc_html_e('Network Enforced:', 'securelywp'); ?></strong>
                            <span class="dynamic-data" data-field="twofa_enforced"><?php echo ($twofa_options['enforce_2fa_network'] ?? false) ? esc_html__('Yes', 'securelywp') : esc_html__('No', 'securelywp'); ?></span>
                        </li>
                    </ul>
                    <a href="<?php echo esc_url(admin_url('profile.php?page=securelywp-two-factor-settings')); ?>" class="button"><?php esc_html_e('Manage', 'securelywp'); ?></a>
                </div>
            </div>
            <?php wp_nonce_field('securelywp_nonce', 'securelywp_nonce_field', false); ?>
            <script>
            jQuery(document).ready(function($) {
                function updateDashboard() {
                    $.ajax({
                        url: ajaxurl,
                        method: 'POST',
                        data: {
                            action: 'securelywp_get_dashboard_status',
                            nonce: $('#securelywp_nonce_field').val()
                        },
                        success: function(response) {
                            if (response.success) {
                                $('[data-field="php_version"]').text(response.data.php_version);
                                $('[data-field="wp_version"]').text(response.data.wp_version);
                                $('[data-field="hardening_status"]').text(response.data.hardening_status);
                                $('[data-field="hardening_count"]').text(response.data.hardening_count);
                                var hardeningIndicator = response.data.hardening_count.split('/')[0] == response.data.hardening_count.split('/')[1] ? 'enabled' : 'disabled';
                                $('[data-feature="hardening-settings"] .status-indicator').removeClass('enabled disabled').addClass(hardeningIndicator);
                                $('[data-field="headers_status"]').text(response.data.headers_status);
                                $('[data-field="headers_count"]').text(response.data.headers_count);
                                var headersIndicator = response.data.headers_count.split('/')[0] == response.data.headers_count.split('/')[1] ? 'enabled' : 'disabled';
                                $('[data-feature="headers-settings"] .status-indicator').removeClass('enabled disabled').addClass(headersIndicator);
                                $('[data-field="captcha_status"]').text(response.data.captcha_status);
                                $('[data-field="captcha_count"]').text(response.data.captcha_count);
                                $('[data-field="captcha_configured"]').text(response.data.captcha_configured);
                                var captchaIndicator = response.data.captcha_status === '<?php esc_html_e('Enabled', 'securelywp'); ?>' ? 'enabled' : 'disabled';
                                $('[data-feature="captcha-settings"] .status-indicator').removeClass('enabled disabled').addClass(captchaIndicator);
                                $('[data-field="twofa_status"]').text(response.data.twofa_status);
                                $('[data-field="twofa_count"]').text(response.data.twofa_count);
                                $('[data-field="twofa_enforced"]').text(response.data.twofa_enforced);
                                var twofaIndicator = response.data.twofa_status === '<?php esc_proto_e('Enabled', 'securelywp'); ?>' ? 'enabled' : 'disabled';
                                $('[data-feature="twofa-settings"] .status-indicator').removeClass('enabled disabled').addClass(twofaIndicator);
                            }
                        }
                    });
                }
                updateDashboard();
                setInterval(updateDashboard, 5000);
            });
            </script>
        <?php
    }

    public static function ajax_get_dashboard_status() {
        if (!check_ajax_referer('securelywp_nonce', 'nonce', false) || !current_user_can(' basket_options')) {
            wp_send_json_error(['message' => esc_html__('Unauthorized.', 'securelywp')]);
        }

        $hardening_options = get_option('securelywp_hardening_options', []);
        $headers_options = get_option('securelywp_headers_options', []);
        $captcha_options = securelywp_captcha_get_settings();
        $twofa_options = get_option('securelywp_2fa_options', []);
        $users_with_2fa = count(get_users(['meta_key' => 'securelywp_2fa_user_options', 'meta_compare' => 'EXISTS']));
        $twofa_enabled_count = $users_with_2fa;
        $twofa_total = count(get_users());

        $hardening_enabled_count = count(array_filter($hardening_options));
        $hardening_total = 7;
        $hardening_status = $hardening_enabled_count === $hardening_total ? esc_html__('All Enabled', 'securelywp') : esc_html__("$hardening_enabled_count/$hardening_total Enabled", 'securelywp');
        $headers_enabled_count = count(array_filter([
            $headers_options['csp_active'],
            $headers_options['hsts_active'],
            $headers_options['x_frame_options_active'],
            $headers_options['referrer_policy_active'],
            $headers_options['permissions_policy_active'],
            $headers_options['x_content_type_options_active'],
        ]));
        $headers_total = 6;
        $headers_status = $headers_enabled_count === $headers_total ? esc_html__('All Enabled', 'securelywp') : esc_html__("$headers_enabled_count/$headers_total Enabled", 'securelywp');
        $captcha_enabled_count = count(array_filter($captcha_options, 'is_bool'));
        $captcha_total = count($captcha_options) - 2;
        $captcha_configured = !empty($captcha_options['site_key']) && !empty($captcha_options['secret_key']);
        $captcha_status = $captcha_configured && $captcha_enabled_count > 0 ? esc_html__('Enabled', 'securelywp') : esc_html__('Disabled', 'securelywp');
        $twofa_status = $twofa_enabled_count > 0 || ($twofa_options['enforce_2fa_network'] ?? false) ? esc_html__('Enabled', 'securelywp') : esc_html__('Disabled', 'securelywp');
        $twofa_enforced = ($twofa_options['enforce_2fa_network'] ?? false) ? esc_html__('Yes', 'securelywp') : esc_html__('No', 'securelywp');

        wp_send_json_success([
            'php_version' => esc_html(phpversion()),
            'wp_version' => esc_html(get_bloginfo('version')),
            'hardening_status' => esc_html($hardening_status),
            'hardening_count' => esc_html("$hardening_enabled_count/$hardening_total"),
            'headers_status' => esc_html($headers_status),
            'headers_count' => esc_html("$headers_enabled_count/$headers_total"),
            'captcha_status' => esc_html($captcha_status),
            'captcha_count' => esc_html("$captcha_enabled_count/$captcha_total"),
            'captcha_configured' => $captcha_configured ? esc_html__('Yes', 'securelywp') : esc_html__('No', 'securelywp'),
            'twofa_status' => esc_html($twofa_status),
            'twofa_count' => esc_html("$twofa_enabled_count/$twofa_total"),
            'twofa_enforced' => esc_html($twofa_enforced),
        ]);
    }
}

SecurelyWP_Dashboard::init();
