<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class ITC_SLAWP_Login_Authorization {

	public static function init() {
		add_action( 'rest_api_init', [ self::class, 'itc_slawp_register_routes' ] );
		add_action( 'admin_init', [ self::class, 'itc_slawp_restrict_dashboard_access' ] );
		add_action( 'admin_enqueue_scripts', [ self::class, 'itc_slawp_enqueue_warning_script' ] );
		add_filter( 'authenticate', [ self::class, 'itc_slawp_restrict_login' ], 100, 3 );
		add_action( 'admin_footer', [ self::class, 'itc_slawp_show_warning_notice' ] );
		add_action( 'wp_ajax_itc_slawp_pause_authorization_all', [ self::class, 'itc_slawp_pause_authorization_all' ] );
		add_action( 'wp_ajax_continue_authorization_all', [ self::class, 'itc_slawp_continue_authorization_all' ] );
		add_action( 'rest_api_init', [ self::class, 'itc_slawp_register_custom_api_route_itc' ] );
		add_filter( 'rest_authentication_errors', [ self::class, 'itc_slawp_allow_custom_route_for_rest_api' ] );
		add_action( 'rest_api_init', [ self::class, 'itc_slawp_register_force_logout_route' ] );
	}

	/**
	 * Register the POST /authorize endpoint with a secret key–based permission callback.
	 */
	public static function itc_slawp_register_routes() {
		register_rest_route( 'secure-login-authorization/v1', '/authorize', [
			'methods'             => 'POST',
			'callback'            => [ self::class, 'itc_slawp_authorize_user' ],
			'permission_callback' => [ self::class, 'itc_slawp_authorize_permission_callback' ],
		] );
	}

	/**
	 * Register the GET /authorize endpoint.
	 */
	public static function itc_slawp_register_custom_api_route_itc() {
		register_rest_route( 'secure-login-authorization/v1', '/authorize', [
			'methods'             => 'GET',
			'callback'            => [ self::class, 'itc_slawp_custom_authorize_callback' ],
			'permission_callback' => [ self::class, 'allow_custom_route_access_itc' ],
		] );
	}

	/**
	 * Register the POST /force-logout endpoint with a secret key–based permission callback.
	 */
	public static function itc_slawp_register_force_logout_route() {
		register_rest_route( 'secure-login-authorization/v1', '/force-logout', [
			'methods'             => 'POST',
			'callback'            => [ self::class, 'itc_slawp_force_logout_user' ],
			'permission_callback' => [ self::class, 'itc_slawp_force_logout_permission_callback' ],
		] );
	}

	/**
	 * Permission callback for the POST /authorize endpoint.
	 * Validates that a secret key is provided and is valid.
	 */
	public static function itc_slawp_authorize_permission_callback( WP_REST_Request $request ) {
		$params = $request->get_json_params() ?: [];
		if ( empty( $params['key'] ) ) {
			return new WP_Error(
				'missing_key',
				esc_html__( 'Secret key is required', 'secure-login-authorization' ),
				[ 'status' => 403 ]
			);
		}
		$key = sanitize_text_field( $params['key'] );
		if ( $key !== get_option( 'itc_slawp_secure_login_authorization' ) ) {
			return new WP_Error(
				'invalid_key',
				esc_html__( 'Invalid secret key', 'secure-login-authorization' ),
				[ 'status' => 403 ]
			);
		}
		return true;
	}

	/**
	 * Permission callback for the POST /force-logout endpoint.
	 * Validates that a secret key is provided and is valid.
	 */
	public static function itc_slawp_force_logout_permission_callback( WP_REST_Request $request ) {
		$params = $request->get_json_params() ?: [];
		if ( empty( $params['key'] ) ) {
			return new WP_Error(
				'missing_key',
				esc_html__( 'Secret key is required', 'secure-login-authorization' ),
				[ 'status' => 403 ]
			);
		}
		$key = sanitize_text_field( $params['key'] );
		if ( $key !== get_option( 'itc_slawp_secure_login_authorization' ) ) {
			return new WP_Error(
				'invalid_key',
				esc_html__( 'Invalid secret key', 'secure-login-authorization' ),
				[ 'status' => 403 ]
			);
		}
		return true;
	}

	/**
	 * GET /authorize permission callback.
	 */
	public static function allow_custom_route_access_itc() {
		return true;
	}

	/**
	 * GET /authorize callback.
	 */
	public static function itc_slawp_custom_authorize_callback( WP_REST_Request $request ) {
	return new WP_Error(
		'rest_no_route',
		__( 'No route was found matching the URL and request method.', 'secure-login-authorization' ),
		[ 'status' => 404 ]
	);
}


	/**
	 * Allow our custom endpoints access via REST API.
	 */
	public static function itc_slawp_allow_custom_route_for_rest_api( $access ) {
		$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		if ( strpos( $request_uri, '/secure-login-authorization/v1/authorize' ) !== false ||
		     strpos( $request_uri, '/secure-login-authorization/v1/force-logout' ) !== false ) {
			return true;
		}
		return $access;
	}

	/**
	 * POST /authorize callback.
	 */
	public static function itc_slawp_authorize_user( $request ) {
		$params   = $request->get_json_params();
		$key      = sanitize_text_field( $params['key'] );
		$username = isset( $params['username'] ) ? sanitize_text_field( $params['username'] ) : '';
		$ip       = isset( $params['ip'] ) ? sanitize_text_field( $params['ip'] ) : '';
		$duration = isset( $params['duration'] ) ? absint( $params['duration'] ) : 0;

		// The secret key is already validated by the permission callback.
		if ( $ip && ! filter_var( $ip, FILTER_VALIDATE_IP ) ) {
			return new WP_Error(
				'invalid_ip',
				esc_html__( 'Invalid IP address', 'secure-login-authorization' ),
				[ 'status' => 400 ]
			);
		}

		$authorized = get_transient( 'itc_slawp_clear_transient_secure_login_authorization' ) ?: [];
		$found      = false;

		foreach ( $authorized as &$auth ) {
			if ( $auth['username'] === $username && ( $auth['ip'] === $ip || empty( $auth['ip'] ) || empty( $ip ) ) ) {
				$auth['expires'] = time() + ( $duration * 60 );
				$found           = true;
				break;
			}
		}

		if ( ! $found ) {
			$authorized[] = [
				'username' => $username,
				'ip'       => $ip,
				'expires'  => time() + ( $duration * 60 ),
			];
		}

		set_transient( 'itc_slawp_clear_transient_secure_login_authorization', $authorized, DAY_IN_SECONDS );

		return [
			'success' => true,
			'message' => 'Authorization updated successfully',
		];
	}

	/**
	 * Filter the authentication process to restrict login if the user is not authorized.
	 */
	public static function itc_slawp_restrict_login( $user, $username, $password ) {
		$authorized = get_transient( 'itc_slawp_clear_transient_secure_login_authorization' ) ?: [];
		$ip         = self::itc_slawp_get_user_ip();

		foreach ( $authorized as $key => $auth ) {
			if ( $auth['expires'] < time() ) {
				unset( $authorized[ $key ] );
				continue;
			}

			if ( $auth['username'] === $username && ( ! $auth['ip'] || $auth['ip'] === $ip ) ) {
				update_user_meta( $user->ID, 'itc_slawp_wp_secure_access_expiry', $auth['expires'] );
				set_transient( 'itc_slawp_clear_transient_secure_login_authorization', $authorized, DAY_IN_SECONDS );
				return $user;
			}
		}

		set_transient( 'itc_slawp_clear_transient_secure_login_authorization', $authorized, DAY_IN_SECONDS );
		return new WP_Error(
			'access_denied',
			esc_html__( 'Access denied. Authorization required.', 'secure-login-authorization' )
		);
	}

	/**
	 * Restrict dashboard access if the session has expired.
	 */
	public static function itc_slawp_restrict_dashboard_access() {
		if ( is_admin() && is_user_logged_in() ) {
			$user_id = get_current_user_id();
			$expiry  = get_user_meta( $user_id, 'itc_slawp_wp_secure_access_expiry', true );

			if ( get_option( 'itc_slawp_login_authorization_paused', 0 ) ) {
				return;
			}

			if ( $expiry && time() > $expiry ) {
				wp_logout();
				wp_die(
					esc_html__( 'Your session has expired. Please contact your administrator to regain access.', 'secure-login-authorization' ),
					esc_html__( 'Session Expired', 'secure-login-authorization' ),
					[ 'response' => 403 ]
				);
			}
		}
	}

	/**
	 * Enqueue a warning script when the session is about to expire.
	 */
	public static function itc_slawp_enqueue_warning_script() {
		if ( is_user_logged_in() && is_admin() ) {
			$user_id = get_current_user_id();
			$expiry  = get_user_meta( $user_id, 'itc_slawp_wp_secure_access_expiry', true );

			if ( $expiry && ( $expiry - time() <= 60 ) && ! get_option( 'itc_slawp_login_authorization_paused', 0 ) ) {
				wp_enqueue_script( 'wp-secure-warning-itc', plugin_dir_url( __FILE__ ) . 'assets/js/wp-login-authorization.js', [ 'jquery' ], '1.0.0', true );
				wp_localize_script( 'wp-secure-warning-itc', 'wpSecureAccessITC', [
					'logoutTime' => $expiry,
					'logoutUrl'  => wp_logout_url(),
				] );
			}
		}
	}

	/**
	 * Show a warning notice in the dashboard when the session is about to expire.
	 */
	public static function itc_slawp_show_warning_notice() {
		if ( is_user_logged_in() && is_admin() ) {
			$user_id = get_current_user_id();
			$expiry  = get_user_meta( $user_id, 'itc_slawp_wp_secure_access_expiry', true );

			if ( $expiry && ( $expiry - time() <= 60 ) && ! get_option( 'itc_slawp_login_authorization_paused', 0 ) ) {
				?>
				<div class="notice notice-warning is-dismissible" id="session-expiry-warning">
					<p><?php esc_html_e( 'Your session will expire in 1 minute. Please save your work.', 'secure-login-authorization' ); ?></p>
				</div>
				<?php
			}
		}
	}

	/**
	 * AJAX callback to pause login authorization.
	 */
	public static function itc_slawp_pause_authorization_all() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'You do not have sufficient permissions.', 'secure-login-authorization' ) ] );
		}

		update_option( 'itc_slawp_login_authorization_paused', 1 );
		wp_send_json_success( [ 'paused' => 'all' ] );
	}

	/**
	 * AJAX callback to continue login authorization.
	 */
	public static function itc_slawp_continue_authorization_all() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'You do not have sufficient permissions.', 'secure-login-authorization' ) ] );
		}

		update_option( 'itc_slawp_login_authorization_paused', 0 );
		wp_send_json_success( [ 'paused' => 'none' ] );
	}

	/**
	 * Helper function to get the user IP.
	 */
	public static function itc_slawp_get_user_ip() {
		$ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
		if ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$forwarded_ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
			$ip           = filter_var( $forwarded_ip, FILTER_VALIDATE_IP ) ? $forwarded_ip : $ip;
		}
		return $ip;
	}

	/**
	 * POST /force-logout callback.
	 */
	public static function itc_slawp_force_logout_user( WP_REST_Request $request ) {
		// Get the username from the request.
		$username = sanitize_text_field( $request->get_param( 'username' ) );

		// Get the user by username.
		$user = get_user_by( 'login', $username );

		if ( $user ) {
			// Destroy all sessions for this user.
			if ( function_exists( 'wp_destroy_user_session' ) ) {
				wp_destroy_user_session( $user->ID );
			} else {
				// Fallback for older WordPress versions.
				$sessions = WP_Session_Tokens::get_instance( $user->ID );
				$sessions->destroy_all();
			}

			// Clear transient if using any custom session storage.
			$authorized = get_transient( 'itc_slawp_clear_transient_secure_login_authorization' ) ?: [];
			foreach ( $authorized as $key => $auth ) {
				if ( $auth['username'] === $username ) {
					unset( $authorized[ $key ] );
					break;
				}
			}
			set_transient( 'itc_slawp_clear_transient_secure_login_authorization', $authorized, DAY_IN_SECONDS );

			return new WP_REST_Response( 'User logged out successfully', 200 );
		}

		return new WP_REST_Response( 'User not found', 404 );
	}
}
