<?php
namespace BSDBB;

if (!defined('ABSPATH')) exit;

final class Settings {
    const OPT_KEY = 'bsdbb_settings';
    const PAGE_SLUG = 'bsdbb-settings';

    public static function init(): void {
        add_action('admin_init', [self::class, 'register']);
    }

    public static function defaults(): array {
        return [
            'enabled'   => true,
            'key_id'    => '',
            'secret'    => '',
            'ip_allow'  => '',
            'time_skew' => 300,
            'rate_per5' => 5,
        ];
    }

    public static function get(): array {
        $opt = get_option(self::OPT_KEY, []);
        return wp_parse_args($opt, self::defaults());
    }

    public static function register(): void {
        register_setting('bsdbb_group', self::OPT_KEY, [
            'type'              => 'array',
            'sanitize_callback' => [self::class, 'sanitize'],
        ]);

        add_settings_section(
            'bsdbb_sec',
            esc_html__('Secure DB Bridge For Bervice', 'secure-db-bridge-for-bervice'),
            function(){},
            self::PAGE_SLUG
        );

        add_settings_field('enabled', esc_html__('Enabled', 'secure-db-bridge-for-bervice'), function(){
            $o     = self::get();
            $name  = self::OPT_KEY . '[enabled]';
            ?>
<label>
    <input type="checkbox" name="<?php echo esc_attr($name); ?>" value="1" <?php checked(!empty($o['enabled'])); ?> />
    <?php echo esc_html__('Enable bridge', 'secure-db-bridge-for-bervice'); ?>
</label>
<?php
        }, self::PAGE_SLUG, 'bsdbb_sec');

        add_settings_field('key_id', esc_html__('Key ID', 'secure-db-bridge-for-bervice'), function(){
            $o     = self::get();
            $name  = self::OPT_KEY . '[key_id]';
            $value = $o['key_id'];
            ?>
<input type="text" name="<?php echo esc_attr($name); ?>" value="<?php echo esc_attr($value); ?>" class="regular-text" />
<?php
        }, self::PAGE_SLUG, 'bsdbb_sec');

        add_settings_field('secret', esc_html__('Shared Secret (Base64)', 'secure-db-bridge-for-bervice'), function(){
            $o     = self::get();
            $name  = self::OPT_KEY . '[secret]';
            $value = $o['secret'];
            ?>
<input type="text" name="<?php echo esc_attr($name); ?>" value="<?php echo esc_attr($value); ?>" class="regular-text" />
<p class="description">
    <?php echo esc_html__('Generate a 32–64 byte random secret and Base64-encode it.', 'secure-db-bridge-for-bervice'); ?>
</p>
<?php
        }, self::PAGE_SLUG, 'bsdbb_sec');

        add_settings_field('ip_allow', esc_html__('IP Allowlist', 'secure-db-bridge-for-bervice'), function(){
            $o     = self::get();
            $name  = self::OPT_KEY . '[ip_allow]';
            $ph    = "203.0.113.10\n198.51.100.0/24";
            ?>
<textarea name="<?php echo esc_attr($name); ?>" rows="5" class="large-text code"
    placeholder="<?php echo esc_attr($ph); ?>"><?php
                echo esc_textarea($o['ip_allow']);
            ?></textarea>
<?php
        }, self::PAGE_SLUG, 'bsdbb_sec');

        add_settings_field('time_skew', esc_html__('Clock Skew (sec)', 'secure-db-bridge-for-bervice'), function(){
            $o     = self::get();
            $name  = self::OPT_KEY . '[time_skew]';
            $value = (string)$o['time_skew'];
            ?>
<input type="number" min="60" step="1" name="<?php echo esc_attr($name); ?>" value="<?php echo esc_attr($value); ?>" />
<?php
        }, self::PAGE_SLUG, 'bsdbb_sec');

        add_settings_field('rate_per5', esc_html__('Rate limit /5min', 'secure-db-bridge-for-bervice'), function(){
            $o     = self::get();
            $name  = self::OPT_KEY . '[rate_per5]';
            $value = (string)$o['rate_per5'];
            ?>
<input type="number" min="1" step="1" name="<?php echo esc_attr($name); ?>" value="<?php echo esc_attr($value); ?>" />
<?php
        }, self::PAGE_SLUG, 'bsdbb_sec');
    }

    public static function page(): void { ?>
<div class="wrap">
    <h1><?php echo esc_html__('Secure DB Bridge', 'secure-db-bridge-for-bervice'); ?></h1>
    <form method="post" action="options.php">
        <?php settings_fields('bsdbb_group'); do_settings_sections(self::PAGE_SLUG); submit_button(); ?>
    </form>
</div>
<?php }

    public static function sanitize($in): array {
        $d = self::defaults();
        $secret = trim((string)($in['secret'] ?? ''));
        if ($secret !== '' && base64_decode($secret, true) === false) {
            add_settings_error(self::OPT_KEY, 'invalid_secret', esc_html__('Secret must be valid Base64.', 'secure-db-bridge-for-bervice'));
            $secret = '';
        }
        return [
            'enabled'   => !empty($in['enabled']),
            'key_id'    => sanitize_text_field($in['key_id'] ?? ''),
            'secret'    => $secret,
            'ip_allow'  => trim((string)($in['ip_allow'] ?? '')),
            'time_skew' => max(60, (int)($in['time_skew'] ?? $d['time_skew'])),
            'rate_per5' => max(1,  (int)($in['rate_per5'] ?? $d['rate_per5'])),
        ];
    }
}