<?php
namespace BSDBB;

if (!defined('ABSPATH')) exit;

final class Rest {
    public static function init(): void {
        add_action('rest_api_init', [self::class, 'routes']);
    }

    public static function routes(): void {
        register_rest_route('bbridge/v1', '/backup/export', [
            'methods'  => 'POST',
            'callback' => [self::class, 'handleExport'],
            'permission_callback' => [self::class, 'permissionExport'],
        ]);
    }

    /**
     * permission callback - MUST perform all auth checks and return true on success
     */
    public static function permissionExport(\WP_REST_Request $req) {
        // 1) Require HTTPS for API calls
        if (!is_ssl()) {
            return new \WP_Error('ssl_required', 'HTTPS required', ['status' => 403]);
        }

        $cfg = Settings::get();
        if (empty($cfg['enabled'])) {
            return new \WP_Error('disabled', 'Bridge disabled', ['status' => 403]);
        }

        // read headers (case-insensitive)
        $keyId = (string)$req->get_header('X-BBridge-KeyId');
        $ts    = (string)$req->get_header('X-BBridge-Timestamp');
        $nonce = (string)$req->get_header('X-BBridge-Nonce');
        $sig   = (string)$req->get_header('X-BBridge-Signature');

        if (!$keyId || !$ts || !$nonce || !$sig) {
            return new \WP_Error('auth_missing', 'Missing auth headers', ['status' => 401]);
        }

        if ($keyId !== $cfg['key_id']) {
            return new \WP_Error('key_mismatch', 'KeyId mismatch', ['status' => 401]);
        }

        if (!ctype_digit($ts)) {
            return new \WP_Error('bad_ts', 'Bad timestamp', ['status' => 401]);
        }

        $now = time();
        if (abs($now - (int)$ts) > (int)$cfg['time_skew']) {
            return new \WP_Error('ts_range', 'Timestamp out of range', ['status' => 401]);
        }

        // nonce replay protection
        if (NonceStore::seen($nonce, (int)$cfg['time_skew'])) {
            return new \WP_Error('nonce_replay', 'Nonce replayed', ['status' => 401]);
        }

        // IP allowlist check - allow all if empty
        $ip = Signer::clientIp();
        if (!Signer::ipAllowed($ip, $cfg['ip_allow'])) {
            return new \WP_Error('ip_denied', 'IP not allowed', ['status' => 403]);
        }

        // rate limiting
        if (!NonceStore::rateCheck((int)$cfg['rate_per5'])) {
            return new \WP_Error('rate_limit', 'Rate limit exceeded', ['status' => 429]);
        }

        // body hash + verify signature (route/path must match what WP actually saw)
        $bodyRaw  = $req->get_body() ?? '';
        $bodyHash = base64_encode(hash('sha256', $bodyRaw, true));

        // Build the canonical path the REST server recognized, e.g. "/wp-json/bbridge/v1/backup/export"
        $path = '/' . rest_get_url_prefix() . $req->get_route();

        $base = $req->get_method() . "\n" . $path . "\n" . $ts . "\n" . $nonce . "\n" . $bodyHash;

        $secretRaw = base64_decode($cfg['secret'], true);
        if (!$secretRaw) {
            return new \WP_Error('bad_secret', 'Bad secret', ['status' => 500]);
        }

        if (!Signer::verify($secretRaw, $base, $sig)) {
            return new \WP_Error('sig_invalid', 'Signature invalid', ['status' => 401]);
        }

        return true;
    }

    public static function handleExport(\WP_REST_Request $req) {
        // These are acceptable for long-running exports.
        if (function_exists('set_time_limit')) { @set_time_limit(0); } // phpcs:ignore
        if (function_exists('ini_set')) { @ini_set('memory_limit', '1024M'); } // phpcs:ignore

        $cfg = Settings::get();
        if (empty($cfg['enabled'])) {
            return self::fail('Disabled', 403);
        }

        // All authentication, replay, IP allowlist, and signature checks already passed
        // inside permissionExport(). Do not duplicate them here.

        // 1) Create DB dump (.sql.gz)
        [$ok, $res] = Exporter::makeGzDump();
        if (!$ok) {
            return self::fail('Export failed: ' . $res, 500);
        }

        // 2) Encrypt the dump -> produces $encPath and $b64iv
        [$eok, $encPath, $b64iv] = Exporter::encryptFile($res, $cfg['secret']);
        if (!$eok) {
            self::safeDelete($res);
            return self::fail('Encrypt failed: ' . $encPath, 500);
        }

        // Prepare stable, sanitized filename
        $fname = 'wpdb_' . site_url() . '_' . gmdate('Ymd_His') . '.sql.gz.enc';
        $fname = preg_replace('/[^A-Za-z0-9_\.\-]+/', '_', $fname);

        // Send headers only after we have a valid file to stream
        nocache_headers();
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $fname . '"');
        header('X-BBridge-IV: ' . $b64iv);
        header('X-BBridge-File: ' . $fname);
        header('X-BBridge-Algo: AES-256-GCM');
        header('X-BBridge-Note: Body=ciphertext||tag(16B)');

        /**
         * We intentionally use fopen/fpassthru/fclose here to stream large encrypted archives.
         * WP_Filesystem lacks a chunked-read API; get_contents() would load the full file into memory,
         * causing OOM on large sites. Narrowly scoped PHPCS ignores are applied ONLY for these lines.
         */

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        $fp = fopen($encPath, 'rb');
        if ($fp) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            fpassthru($fp);
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
            fclose($fp);
        } else {
            // Best effort cleanup if the stream couldn't be opened
            self::safeDelete($encPath);
            self::safeDelete($res);
            return self::fail('Cannot open encrypted file for streaming', 500);
        }

        // Cleanup temp files after streaming
        self::safeDelete($encPath);
        self::safeDelete($res);

        // Hard exit to avoid REST server appending JSON
        exit;
    }

    private static function fail(string $msg, int $code) {
        return new \WP_REST_Response(['error' => $msg], $code);
    }

    /**
     * Delete a file safely, making sure the necessary functions are available.
     */
    private static function safeDelete(string $path): void {
        if ($path === '' || !file_exists($path)) {
            return;
        }
        if (!function_exists('wp_delete_file')) {
            // Ensure file.php is loaded where wp_delete_file() lives.
            if (defined('ABSPATH')) {
                @require_once ABSPATH . 'wp-admin/includes/file.php';
            }
        }
        if (function_exists('wp_delete_file')) {
            @wp_delete_file($path);
        } else {
            // Fall back if wp_delete_file is not available in this context.
            // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
            @unlink($path);
        }
    }
}
