<?php
namespace BSDBB;

if (!defined('ABSPATH')) exit;

final class Exporter {

    public static function makeGzDump(): array {
        global $wpdb;

        $uploads = wp_upload_dir();
        $tmpDir  = trailingslashit($uploads['basedir']) . 'bervice-db-bridge/tmp/';
        if (!is_dir($tmpDir)) {
            if (!wp_mkdir_p($tmpDir)) {
                return [false, 'Cannot create tmp dir: '.$tmpDir];
            }
        }
        $fname = 'dump_'.gmdate('Ymd_His').'_'.wp_generate_password(6, false).'.sql.gz';
        $out   = $tmpDir.$fname;

        $gz = @gzopen($out, 'wb9');
        if (!$gz) return [false, 'Cannot open gz file for write: '.$out];

        $site = site_url();
        gzwrite($gz, "-- Bervice DB Bridge dump\n");
        gzwrite($gz, "-- Site: {$site}\n");
        gzwrite($gz, "-- Date (UTC): ".gmdate('c')."\n\n");
        gzwrite($gz, "SET SQL_MODE='NO_AUTO_VALUE_ON_ZERO';\n");
        gzwrite($gz, "SET time_zone = '+00:00';\n\n");
        gzwrite($gz, "START TRANSACTION;\n\n");

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $tables = $wpdb->get_col('SHOW TABLES');
        if (!is_array($tables) || empty($tables)) {
            gzclose($gz);
            wp_delete_file($out);
            return [false, 'No tables found or SHOW TABLES denied.'];
        }

        foreach ($tables as $table) {
            // Make sure the table name is a string
            $table = (string) $table;

            // Double-check that the table is actually in the list we got from "SHOW TABLES".
            // This acts as a whitelist, so we don’t allow any unexpected or malicious names.
            if (!in_array($table, $tables, true)) {
                // This should never happen, but if it does we skip the table
                gzwrite($gz, "\n-- WARN: skipped unknown table name\n");
                continue;
            }

            // Wrap the table name safely in backticks to avoid SQL injection issues.
            $safeTable = self::backtick($table);
            
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
            $row = $wpdb->get_row("SHOW CREATE TABLE {$safeTable}", ARRAY_N);
            if (!$row || !isset($row[1])) {
                gzwrite($gz, "\n-- WARN: cannot SHOW CREATE TABLE for {$safeTable}\n");
                continue;
            }
            $create = $row[1];

            gzwrite($gz, "\n--\n-- Table structure for table {$safeTable}\n--\n");
            gzwrite($gz, "DROP TABLE IF EXISTS {$safeTable};\n");
            gzwrite($gz, $create . ";\n");

            gzwrite($gz, "\n--\n-- Dumping data for table {$safeTable}\n--\n");

            $batch = 1000;
            $offset = 0;

            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $columns = $wpdb->get_results("SHOW COLUMNS FROM {$safeTable}", ARRAY_A);
            if (!is_array($columns) || empty($columns)) {
                continue;
            }
            $colNames = array_map(function($c){
                return self::backtick((string)$c['Field']);
            }, $columns);
            $colList = implode(', ', $colNames);
            
            while (true) {
                // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results(
                    $wpdb->prepare("SELECT * FROM {$safeTable} LIMIT %d OFFSET %d", $batch, $offset),
                    ARRAY_A
                );
                // phpcs:enable
            
                if (!$rows) {
                    break;
                }
            
                foreach ($rows as $r) {
                    $values = [];
                    foreach ($columns as $c) {
                        $name = (string)$c['Field'];
                        if (!array_key_exists($name, $r) || is_null($r[$name])) {
                            $values[] = 'NULL';
                            continue;
                        }
                        $val = (string)$r[$name];
                        $values[] = self::sqlValue($val, $wpdb);
                    }
                    $line = "INSERT INTO {$safeTable} ({$colList}) VALUES (".implode(', ', $values).");\n";
                    gzwrite($gz, $line);
                }
            
                $offset += $batch;
            }
        }

        gzwrite($gz, "\nCOMMIT;\n");
        gzclose($gz);

        if (!file_exists($out) || filesize($out) < 64 * 1024) {
            wp_delete_file($out);
            return [false, 'Dump file missing or too small (possible permission/DB error).'];
        }
        return [true, $out];
    }

public static function encryptFile(string $fileGzPath, string $b64Secret): array {
    $secretRaw = base64_decode($b64Secret, true);
    if (!$secretRaw || strlen($secretRaw) < 32) {
        return [false, 'Invalid secret (Base64 >= 32 bytes required).'];
    }

    if (!is_readable($fileGzPath)) return [false, 'Dump file not readable'];

    // derive a 32-byte key via HKDF (use hash_hkdf directly)
    $key = hash_hkdf('sha256', $secretRaw, 32, 'bbridge-enc', 'bbridge-salt');
    if ($key === false) return [false, 'HKDF failed'];
    // hash_hkdf returns hex-string; convert to raw binary if length==64 hex
    if (strlen($key) === 64 && ctype_xdigit($key)) {
        $keyRaw = hex2bin($key);
    } else {
        $keyRaw = $key;
    }

    $iv = random_bytes(12);
    $b64iv = base64_encode($iv);

    $encPath = $fileGzPath . '.enc';

    // If openssl binary exists, use streaming encryption via it (AES-256-GCM)
    $openssl_bin = trim(shell_exec('which openssl 2>/dev/null') ?: '');
    if ($openssl_bin) {
        // Build command: openssl enc -aes-256-gcm -K <hexkey> -iv <hexiv> -in <in> -out <out>
        $hexKey = bin2hex($keyRaw);
        $hexIv  = bin2hex($iv);

        // Some openssl versions support -aes-256-gcm, but older may not.
        $cmd = sprintf('%s enc -aes-256-gcm -K %s -iv %s -in %s -out %s', 
            escapeshellcmd($openssl_bin),
            escapeshellarg($hexKey),
            escapeshellarg($hexIv),
            escapeshellarg($fileGzPath),
            escapeshellarg($encPath)
        );

        // run command
        $ret = null;
        system($cmd, $ret);
        if ($ret !== 0 || !file_exists($encPath)) {
            // fallback: try PHP in-memory method (if file small)
            // continue to php method below
        } else {
            // OpenSSL `enc -aes-256-gcm` writes tag? Newer versions append tag as trailing bytes;
            // we assume it did; return iv to client for verification.
            return [true, $encPath, $b64iv];
        }
    }

    // Fallback: if file size is small enough -> use PHP openssl_encrypt
    $filesize = filesize($fileGzPath);
    $maxInMemory = 50 * 1024 * 1024; // 50MB threshold - tune as needed
    if ($filesize > $maxInMemory) {
        return [false, 'Server cannot encrypt large dump in-memory and openssl binary not available'];
    }

    $plain = @file_get_contents($fileGzPath);
    if ($plain === false) return [false, 'Read dump failed'];

    if (!function_exists('openssl_encrypt')) {
        return [false, 'OpenSSL not available on PHP.'];
    }

    $tag = '';
    $cipher = openssl_encrypt($plain, 'aes-256-gcm', $keyRaw, OPENSSL_RAW_DATA, $iv, $tag);
    if ($cipher === false || strlen($tag) !== 16) {
        return [false, 'AES-256-GCM encryption failed'];
    }

    // write cipher||tag
    $ok = @file_put_contents($encPath, $cipher . $tag);
    if ($ok === false) return [false, 'Write enc failed'];

    return [true, $encPath, $b64iv];
}
    private static function backtick(string $name): string {
        return '`' . str_replace('`', '``', $name) . '`';
    }

    private static function sqlValue(string $val, \wpdb $wpdb): string {
        $dbh = $wpdb->dbh;
        if ($dbh instanceof \mysqli) {
            $esc = $dbh->real_escape_string($val);
        } else {
            $esc = addslashes($val);
        }
        return "'{$esc}'";
    }
}
