<?php
/**
 * @package Newway
 * @version 1.1.0
 */
/*
Plugin Name: Section-Specific Dashboard Lock
Plugin URI: https://newwaypmsco.com/
Description: Locks specific sections of the WordPress admin dashboard with custom passwords, including submenus.
Author: New-way
Version: 1.1.0
License: GPL-2.0-or-later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Requires at least: 5.0
Tested up to: 6.8
Text Domain: section-specific-dashboard-lock
*/

// Prevent direct access to the file
if (!defined('ABSPATH')) {
    exit;
}

// Helper function to get current page slug
function sectdalo_get_current_page() {
    $current_page = '';
    $screen = get_current_screen();
    if ($screen && !empty($screen->base)) {
        $current_page = $screen->base;
        if (!empty($screen->post_type)) {
            $current_page .= '?post_type=' . sanitize_key($screen->post_type);
        }
    } elseif (!empty($GLOBALS['pagenow'])) {
        $current_page = sanitize_text_field($GLOBALS['pagenow']);
        // Handle query string pages (e.g., WooCommerce, Site Kit)
        if (isset($_GET['page'])) {
            $current_page = sanitize_text_field($_GET['page']);
        }
    }
    return $current_page;
}

// Disable Admin Bar for locked sections
add_action('admin_menu', 'sectdalo_disable_admin_bar_for_locked_sections', 1);
function sectdalo_disable_admin_bar_for_locked_sections() {
    if (!is_admin() || !is_user_logged_in() || !current_user_can('manage_options')) {
        return;
    }

    $settings = get_option('sectdalo_section_lock_settings', []);
    if (empty($settings)) {
        return;
    }

    $current_page = sectdalo_get_current_page();
    global $menu, $submenu;
    $is_locked = false;

    if (!is_array($menu) || empty($menu)) {
        return;
    }

    foreach ($settings as $section => $data) {
        if (empty($data['enabled']) || empty($data['password'])) {
            continue;
        }
        if ($section === $current_page) {
            $is_locked = true;
            break;
        } else {
            foreach ($menu as $menu_item) {
                if ($menu_item[2] === $section && isset($submenu[$section]) && is_array($submenu[$section])) {
                    foreach ($submenu[$section] as $submenu_item) {
                        if ($submenu_item[2] === $current_page || (isset($_GET['page']) && $submenu_item[2] === $_GET['page'])) {
                            $is_locked = true;
                            break 2;
                        }
                    }
                }
            }
        }
    }

    if ($is_locked) {
        add_filter('show_admin_bar', '__return_false', 999);
    }
}

// Add settings menu
add_action('admin_menu', 'sectdalo_section_lock_menu');
function sectdalo_section_lock_menu() {
    add_options_page(
        __('Section Lock Settings', 'section-specific-dashboard-lock'),
        __('Section Lock', 'section-specific-dashboard-lock'),
        'manage_options',
        'section-specific-dashboard-lock',
        'sectdalo_section_lock_settings_page'
    );
}

// Settings page to configure locked sections and passwords
function sectdalo_section_lock_settings_page() {
    if (isset($_POST['sectdalo_section_lock_settings_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['sectdalo_section_lock_settings_nonce'])), 'sectdalo_section_lock_settings_action')) {
        if (isset($_POST['section_lock_settings']) && is_array($_POST['section_lock_settings'])) {
            $existing_settings = get_option('sectdalo_section_lock_settings', []);
            $new_settings = [];

            $section_lock_settings = map_deep(wp_unslash($_POST['section_lock_settings']), 'sanitize_text_field');
            foreach ($section_lock_settings as $section => $data) {
                $section = esc_attr($section);
                $enabled = isset($data['enabled']) && $data['enabled'] == '1' ? true : false;
                $password = isset($data['password']) ? sanitize_text_field($data['password']) : '';

                if ($enabled && !empty($password)) {
                    $password_hash = password_hash($password, PASSWORD_DEFAULT);
                    $new_settings[$section] = [
                        'enabled' => true,
                        'password' => $password_hash
                    ];
                } elseif ($enabled && empty($password) && isset($existing_settings[$section]['password'])) {
                    $new_settings[$section] = [
                        'enabled' => true,
                        'password' => $existing_settings[$section]['password']
                    ];
                } else {
                    $new_settings[$section] = [
                        'enabled' => false,
                        'password' => ''
                    ];
                }
            }

            $final_settings = array_filter($new_settings, function($data) {
                return $data['enabled'] === true;
            });
            update_option('sectdalo_section_lock_settings', $final_settings);
            echo '<div class="updated"><p>' . esc_html__('Settings saved successfully!', 'section-specific-dashboard-lock') . '</p></div>';
        }
    }

    $settings = get_option('sectdalo_section_lock_settings', []);
    global $menu;
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Section Lock Settings', 'section-specific-dashboard-lock'); ?></h1>
        <form method="post">
            <?php wp_nonce_field('sectdalo_section_lock_settings_action', 'sectdalo_section_lock_settings_nonce'); ?>
            <table class="form-table">
                <tr>
                    <th><?php esc_html_e('Section', 'section-specific-dashboard-lock'); ?></th>
                    <th><?php esc_html_e('Enable Lock', 'section-specific-dashboard-lock'); ?></th>
                    <th><?php esc_html_e('Password', 'section-specific-dashboard-lock'); ?></th>
                </tr>
                <?php
                if (!empty($menu) && is_array($menu)) {
                    foreach ($menu as $menu_item) {
                        if (!empty($menu_item[0]) && !empty($menu_item[2])) {
                            $section_slug = $menu_item[2];
                            $section_name = wp_strip_all_tags($menu_item[0]);
                            $is_enabled = isset($settings[$section_slug]['enabled']) && $settings[$section_slug]['enabled'] === true;
                            ?>
                            <tr>
                                <td><?php echo esc_html($section_name); ?> (<?php echo esc_html($section_slug); ?>)</td>
                                <td><input type="checkbox" name="section_lock_settings[<?php echo esc_attr($section_slug); ?>][enabled]" value="1" <?php checked($is_enabled); ?>></td>
                                <td><input type="password" name="section_lock_settings[<?php echo esc_attr($section_slug); ?>][password]" value="" placeholder="<?php esc_attr_e('Leave blank to keep existing password', 'section-specific-dashboard-lock'); ?>"></td>
                            </tr>
                            <?php
                        }
                    }
                } else {
                    echo '<tr><td colspan="3">' . esc_html__('No menu items found.', 'section-specific-dashboard-lock') . '</td></tr>';
                }
                ?>
            </table>
            <input type="submit" value="<?php esc_attr_e('Save Settings', 'section-specific-dashboard-lock'); ?>" class="button button-primary">
        </form>
    </div>
    <?php
}

// Enqueue CSS and JS for lock screen
add_action('admin_enqueue_scripts', 'sectdalo_enqueue_lock_screen_assets');
function sectdalo_enqueue_lock_screen_assets() {
    $settings = get_option('sectdalo_section_lock_settings', []);
    if (!empty($settings)) {
        wp_enqueue_style(
            'sectdalo-lock-screen',
            plugin_dir_url(__FILE__) . 'assets/css/lock-screen.css',
            [],
            '1.1.0'
        );
        wp_enqueue_script(
            'sectdalo-lock-screen',
            plugin_dir_url(__FILE__) . 'assets/js/lock-screen.js',
            [],
            '1.1.0',
            true
        );
    }
}

// Lock specific sections and their submenus
add_action('admin_init', 'sectdalo_lock_specific_sections', 1);
function sectdalo_lock_specific_sections() {
    if (!is_user_logged_in() || !is_admin()) {
        return;
    }

    $settings = get_option('sectdalo_section_lock_settings', []);
    if (empty($settings)) {
        return;
    }

    $current_page = sectdalo_get_current_page();
    global $menu, $submenu;
    $locked_parent = false;
    $current_section_password = false;

    if (!is_array($menu) || empty($menu)) {
        return;
    }

    foreach ($settings as $section => $data) {
        if (empty($data['enabled']) || empty($data['password'])) {
            continue;
        }
        if ($section === $current_page) {
            $current_section_password = $data['password'];
            $locked_parent = $section;
            break;
        } else {
            foreach ($menu as $menu_item) {
                if ($menu_item[2] === $section && isset($submenu[$section]) && is_array($submenu[$section])) {
                    foreach ($submenu[$section] as $submenu_item) {
                        if ($submenu_item[2] === $current_page || (isset($_GET['page']) && $submenu_item[2] === $_GET['page'])) {
                            $current_section_password = $data['password'];
                            $locked_parent = $section;
                            break 2;
                        }
                    }
                }
            }
        }
    }

    if (!$current_section_password || !$locked_parent) {
        return;
    }

    if (!session_id()) {
        session_start();
    }

    if (isset($_SESSION['sectdalo_section_unlocked'][$locked_parent]) && $_SESSION['sectdalo_section_unlocked'][$locked_parent] === true) {
        return;
    }

    $error = '';
    $success = '';
    if (isset($_POST['sectdalo_section_password_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['sectdalo_section_password_nonce'])), 'sectdalo_section_password_action')) {
        if (isset($_POST['section_password']) && !empty($_POST['section_password'])) {
            $submitted_password = sanitize_text_field(wp_unslash($_POST['section_password']));
            if (password_verify($submitted_password, $current_section_password)) {
                $_SESSION['sectdalo_section_unlocked'][$locked_parent] = true;
                return;
            } else {
                $error = __('Incorrect password!', 'section-specific-dashboard-lock');
            }
        } else {
            $error = __('Password is required.', 'section-specific-dashboard-lock');
        }
    }

    if (isset($_POST['sectdalo_change_password_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['sectdalo_change_password_nonce'])), 'sectdalo_change_password_action')) {
        if (isset($_POST['current_password']) && isset($_POST['new_password']) && !empty($_POST['current_password']) && !empty($_POST['new_password'])) {
            $current_password = sanitize_text_field(wp_unslash($_POST['current_password']));
            $new_password = sanitize_text_field(wp_unslash($_POST['new_password']));
            if (password_verify($current_password, $current_section_password)) {
                $settings[$locked_parent]['password'] = password_hash($new_password, PASSWORD_DEFAULT);
                update_option('sectdalo_section_lock_settings', $settings);
                $success = __('Password changed successfully! Please use the new password to unlock.', 'section-specific-dashboard-lock');
            } else {
                $error = __('Current password is incorrect!', 'section-specific-dashboard-lock');
            }
        } else {
            $error = __('Both current and new passwords are required.', 'section-specific-dashboard-lock');
        }
    }

    add_filter('show_admin_bar', '__return_false', PHP_INT_MAX);
    remove_action('wp_footer', 'wp_admin_bar_render', 1000);
    remove_action('admin_footer', 'wp_admin_bar_render', 1000);

    wp_enqueue_style('sectdalo-lock-screen');
    wp_enqueue_script('sectdalo-lock-screen');
    ?>
    <!DOCTYPE html>
    <html <?php language_attributes(); ?>>
    <head>
        <meta charset="<?php bloginfo('charset'); ?>">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title><?php esc_html_e('Section Lock', 'section-specific-dashboard-lock'); ?></title>
        <?php wp_head(); ?>
    </head>
    <body>
    <div class="lock-container">
        <h2>
            <?php
            echo esc_html(sprintf(__('Enter password for %s', 'section-specific-dashboard-lock'), esc_html($locked_parent)));
            ?>
        </h2>
        <?php
        if ($error) {
            echo '<p class="error">' . esc_html($error) . '</p>';
        }
        if ($success) {
            echo '<p class="success">' . esc_html($success) . '</p>';
        }
        ?>
        <form method="post">
            <?php wp_nonce_field('sectdalo_section_password_action', 'sectdalo_section_password_nonce'); ?>
            <input type="password" name="section_password" required><br><br>
            <input type="submit" value="<?php esc_attr_e('Unlock', 'section-specific-dashboard-lock'); ?>" style="padding: 5px 10px;">
        </form>
        <button onclick="toggleChangePasswordForm()" style="margin-top: 10px;"><?php esc_html_e('Change Password', 'section-specific-dashboard-lock'); ?></button>
        <div id="change-password-form" class="change-password-form">
            <form method="post">
                <?php wp_nonce_field('sectdalo_change_password_action', 'sectdalo_change_password_nonce'); ?>
                <label for="current_password"><?php esc_html_e('Current Password:', 'section-specific-dashboard-lock'); ?></label><br>
                <input type="password" name="current_password" required><br><br>
                <label for="new_password"><?php esc_html_e('New Password:', 'section-specific-dashboard-lock'); ?></label><br>
                <input type="password" name="new_password" required><br><br>
                <input type="submit" value="<?php esc_attr_e('Update Password', 'section-specific-dashboard-lock'); ?>" style="padding: 5px 10px;">
            </form>
        </div>
    </div>
    <?php wp_footer(); ?>
    </body>
    </html>
    <?php
    exit;
}

// Reset session on logout
add_action('wp_logout', 'sectdalo_reset_section_lock');
function sectdalo_reset_section_lock() {
    if (session_id()) {
        session_unset();
        session_destroy();
    }
}

// Reset session on login to enforce lock
add_action('wp_login', 'sectdalo_reset_session_on_login');
function sectdalo_reset_session_on_login() {
    if (!session_id()) {
        session_start();
    }
    unset($_SESSION['sectdalo_section_unlocked']);
}