<?php
/*
Plugin Name: Searchify AI
Description: Tracks live citations from AI conversations.
Version: 1.1.0
Author: Searchify
Author URI: https://searchify.ai/
License: GPL2
*/

// Define tracking function
if (!function_exists('searchify_track_request')) {
    function searchify_track_request() {
        // Avoid tracking admin, cron, or AJAX requests
        if (is_admin() || wp_doing_cron() || wp_doing_ajax()) {
            return;
        }

        // Avoid duplicate tracking by checking if we've already tracked this request
        static $tracked = false;
        if ($tracked) {
            return;
        }
        $tracked = true;

        // Collect and sanitize request info
        $scheme     = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';

        $host       = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
        $uri        = isset($_SERVER['REQUEST_URI']) ? esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $full_url   = esc_url_raw($scheme . '://' . $host . $uri);

        $domain     = $host; // already sanitized
        $user_agent = isset($_SERVER['HTTP_USER_AGENT'])
            ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT']))
            : 'unknown';

        $data = array(
            'full_url'   => $full_url,
            'domain'     => $domain,
            'user_agent' => $user_agent,
        );

        // Send POST request to API
        // Use blocking with short timeout to ensure request completes before PHP exits
        // Short timeout (1s) minimizes page load impact while ensuring delivery
        wp_remote_post(
            'https://api.searchify.ai/v1/track/citation',
            array(
                'headers'  => array('Content-Type' => 'application/json'),
                'body'     => wp_json_encode($data),
                'timeout'  => 1, // Short timeout to minimize delay
                'blocking' => true, // Blocking ensures request completes before PHP exits
            )
        );

    }
}

// Start output buffering immediately to ensure we can track even cached pages
if (!headers_sent() && !ob_get_level()) {
    ob_start('searchify_output_buffer_handler');
}

// Track on the earliest possible hooks - before any caching logic
// These fire in order: muplugins_loaded -> plugins_loaded -> init -> setup_theme -> template_redirect
add_action('muplugins_loaded', 'searchify_track_request', 1);
add_action('plugins_loaded', 'searchify_track_request', 1);
add_action('init', 'searchify_track_request', 1);
add_action('setup_theme', 'searchify_track_request', 1);
add_action('template_redirect', 'searchify_track_request', 1);

// Track on shutdown hook - ALWAYS fires if PHP executes at all
add_action('shutdown', 'searchify_track_request', 999);


/**
 * Output buffer handler - ensures tracking happens even with output buffering
 */
function searchify_output_buffer_handler($buffer) {
    // Track when buffer is processed
    searchify_track_request();
    return $buffer; // Return buffer unchanged
}

