<?php

namespace SearchFilterSort\Filter;

if ( ! defined('ABSPATH') ) {
    exit;
}

abstract class BaseSettings implements TabInterface{

    protected $page;

    protected $group;

    protected $optionName;

    protected $options;

    protected function registerSettings($settings, $page, $optionName)
    {
        $callbacks = array(
            'checkbox' => array($this, 'checkboxCallback'),
            'text'     => array($this, 'inputCallback'),
            'license'  => array($this, 'licenseCallback'),
            'textarea' => array($this, 'textareaCallback'),
            'editor'   => array($this, 'textEditorCallback'),
            'select'   => array($this, 'selectCallback'),
            'hidden'   => array($this, 'hiddenCallback')
        );

        foreach ($settings as $sectionId => $section) {
            $callback = isset($section['callback']) ? $section['callback'] : null;
            add_settings_section($sectionId, $section['label'], $callback, $page);

            if( isset( $section['fields'] ) ){
                foreach ( $section['fields'] as $fieldId => $field) {
                    if(empty($field['type']) || !isset($callbacks[$field['type']])){
                        continue;
                    }
                    $title                = isset($field['title']) ? $field['title'] : '';
                    $field['label_for']   = $fieldId;
                    $field['option_name'] = $optionName;
                    $field['placeholder'] = isset($field['placeholder']) ? $field['placeholder'] : '';
                    $field['id']          = isset($field['id']) ? $field['id'] : $fieldId;
                    add_settings_field($fieldId, $title, $callbacks[$field['type']], $page, $sectionId, $field);
                }
            }
        }
    }

    public function checkBoxCallback($args)
    {

        $checkbox = '<label><input type="checkbox" name="%s[%s]" %s id="%s">%s</label>';
        $checkbox = apply_filters( 'sfswp_settings_field_checkbox', $checkbox, $args );
        $checked  = $this->getOption($args['label_for']);

        if ($checked == '1') {
            $checked = 'on';
        }

        printf(
            $checkbox,
            esc_attr($args['option_name']),
            esc_attr($args['label_for']),
            checked('on', $checked, false),
            esc_attr($args['id']),
            esc_html($args['label'])
        );

        if( isset( $args['description'] ) ){
            printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
        }
    }

    public function inputCallback($args)
    {
        $input = '<input class="%s" type="text" name="%s[%s]" value="%s" placeholder="%s" id="%s">';
        $class = isset( $args['class'] ) ? $args['class'] : 'regular-text';
        if( isset( $args['default'] ) ){
            $value = $this->getOption($args['label_for']) ? $this->getOption($args['label_for']) : $args['default'];
        }else{
            $value = $this->getOption($args['label_for']);
        }

        $value = $value ? trim($value) : $value;

        printf(
            $input,
            esc_attr($class),
            esc_attr($this->optionName),
            esc_attr($args['label_for']),
            esc_attr($value),
            esc_attr($args['placeholder']),
            esc_attr($args['id'])
        );

        if( isset($args['id']) && $args['id'] === 'posts_container' ){

            if( sfs_get_option('enable_ajax') === 'on' && ! $value ){
                printf( '<p class="sfswp-warning">%s</p>', esc_html__( 'You must specify Posts Container, otherwise AJAX will not work properly', 'searchfiltersort' ) );
            }
        }

        if( isset( $args['description'] ) ){
            printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
        }

    }

    public function licenseCallback( $args )
    {
        $input = '<input class="%s" type="text" readonly="readonly" name="%s[%s]" value="%s" placeholder="%s" id="%s">';
        $class = isset( $args['class'] ) ? $args['class'] : 'regular-text';

        $value = str_repeat( '*', 80 );
        $value = trim($value);

        printf(
            $input,
            esc_attr($class),
            esc_attr($this->optionName),
            esc_attr($args['label_for']),
            esc_attr($value),
            esc_attr($args['placeholder']),
            esc_attr($args['id'])
        );

        if( isset( $args['description'] ) ){
            printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
        }

    }

    public function hiddenCallback( $args )
    {
        $input = '<input class="%s" type="hidden" name="%s[%s]" value="%s" placeholder="%s" id="%s">';
        $class = isset( $args['class'] ) ? $args['class'] : 'regular-text';
        printf(
            $input,
            esc_attr($class),
            esc_attr($this->optionName),
            esc_attr($args['label_for']),
            esc_attr($this->getOption($args['label_for'])),
            esc_attr($args['placeholder']),
            esc_attr($args['id'])
        );
    }

    public function selectCallback( $args )
    {
        $options = isset($args['options']) ? $args['options'] : [];
        $class = isset( $args['class'] ) ? $args['class'] : 'sfs-settings-select';
        $value = $this->getOption($args['label_for']);

        $multiple = ! empty($args['multiple']) ? 'multiple' : null;

        $select = '<select name="%s[%s]%s" class="%s" placeholder="%s" id="%s" %s>';

        printf(
            $select,
            esc_attr($this->optionName),
            esc_attr($args['label_for']),
            $multiple ? '[]' : '',
            $class,
            esc_attr($args['placeholder']),
            esc_attr($args['id']),
            $multiple
        );

        foreach ($options as $key => $option) {
            if (! empty($value)) {
                if (is_null($multiple)) {
                    $selected = $key === $value ? 'selected' : '';
                } else {
                    $selected = in_array($key, $value) ? 'selected' : '';
                }
            } else {
                $selected = '';
            }

            printf('<option value="%s" %s>%s</option>', esc_attr($key), $selected, esc_html($option));
        }

        print('</select>');

        if (isset($args['help'])) {
            printf('<p>%s</p>', esc_html($args['help']));
        }

        if( isset( $args['description'] ) ){
            printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
        }

    }

    public function textareaCallback($args)
    {
        $textarea = '<textarea class="sfs-settings-input large-text code" name="%s[%s]" placeholder="%s" cols="30" rows="10" id="%s">%s</textarea>';

        printf(
            $textarea,
            esc_attr($this->optionName),
            esc_attr($args['label_for']),
            esc_attr($args['placeholder']),
            esc_attr($args['id']),
            esc_textarea( $this->getOption($args['label_for']) )
        );

    }

    public function textEditorCallback($args)
    {
        wp_editor(
            $this->getOption( $args['label_for'] ),
            esc_attr( $args['id'] ),
            array( 'textarea_name' => esc_attr( $this->optionName . '[' . $args['label_for'] . ']' ) )
        );

    }

    public function render()
    {
        print('<form action="' . admin_url('options.php') . '" method="post">');

        settings_errors();

        settings_fields($this->group);

        do_action('sfswp_before_sections_settings_fields', $this->page );

        $this->doSettingsSections($this->page);

        do_action('sfswp_after_sections_settings_fields', $this->page );

        if( apply_filters('sfswp_settings_submit_button', true ) ){
            submit_button();
        }

        print('</form>');
    }

    public function doSettingsSections( $page ) {
        global $wp_settings_sections, $wp_settings_fields;

        if ( ! isset( $wp_settings_sections[ $page ] ) ) {
            return;
        }

        foreach ( (array) $wp_settings_sections[ $page ] as $section ) {

            do_action('sfswp_before_settings_fields_title', $page );

            if ( $section['title'] ) {
                echo "<h2>". wp_kses( $section['title'], array( 'span' => array( 'class' => true ) )) ."</h2>\n";
            }

            do_action('sfswp_after_settings_fields_title', $page );

            if ( $section['callback'] ) {
                call_user_func( $section['callback'], $section );
            }

            if ( ! isset( $wp_settings_fields ) || ! isset( $wp_settings_fields[ $page ] ) || ! isset( $wp_settings_fields[ $page ][ $section['id'] ] ) ) {
                continue;
            }

            $sortable = ( $section['id'] === 'sfswp_slugs' ) ? ' sfswp-sortable-table' : '';

            echo '<table class="sfswp-form-table form-table'.esc_attr($sortable).'" role="presentation">';
            $this->doSettingsFields( $page, $section['id'] );
            echo '</table>';
        }
    }

    public function doSettingsFields( $page, $section ) {
        global $wp_settings_fields;

        if ( ! isset( $wp_settings_fields[ $page ][ $section ] ) ) {
            return;
        }

        $i = 1;

        foreach ( (array) $wp_settings_fields[ $page ][ $section ] as $field ) {
            $class = '';

            if( $field['id'] === 'bottom_widget_compatibility' ){
                if( sfs_get_option('mobile_filter_settings') === 'show_bottom_widget' ){
                    $field['args']['class'] .= ' sfswp-opened';
                }
            }

            if( $field['id'] === 'color_swatches_taxonomies' || $field['id'] === 'rounded_swatches' ) {
                if( sfs_get_experimental_option('use_color_swatches') === 'on' ) {
                    $field['args']['class'] .= ' sfswp-opened';
                }
            }

            if ( ! empty( $field['args']['class'] ) ) {
                $class = ' class="' . esc_attr( $field['args']['class'] ) . '"';
            }

            do_action('sfswp_before_settings_field', $field );

            echo "<tr{$class}>";

            $sortable = isset( $field['args']['sortable'] ) ? $i : '';

            if( $sortable ){
                echo '<td class="sfswp-order-td">'.esc_html($sortable).'</td>';
            }

            $tooltip = isset( $field['args']['tooltip'] ) ? sfs_tooltip( array( 'tooltip' => $field['args']['tooltip'] ) ) : '';
            $tooltip = wp_kses(
                $tooltip,
                array(
                    'strong' => array(),
                    'br'     => array(),
                    'a'      => array( 'href'=>true, 'title'=>true, 'class'=>true ),
                    'span'   => array( 'class'=>true, 'data-tip'=>true)
                )
            );

            if ( ! empty( $field['args']['label_for'] ) ) {
                echo '<th scope="row"><label for="' . esc_attr( $field['args']['label_for'] ) . '">' . esc_html($field['title']) . '</label> ' .$tooltip. '</th>'; // $tooltip already escaped
            } else {
                echo '<th scope="row">'. esc_html($field['title'] ) . ' ' .$tooltip. '</th>'; // $tooltip already escaped
            }

            echo '<td>';
            call_user_func( $field['callback'], $field['args'] );
            echo '</td>';

            if( $sortable ){
                $title = ( ! defined( 'SFS_FILTERS_PRO' ) ) ? __( 'Editing the order of URL prefixes is available in the PRO version', 'searchfiltersort' ) : '';
                echo '<td class="sfswp-order-sortable-handle-icon"><span class="dashicons dashicons-menu sfswp-field-sortable-handle" title="'.esc_attr( $title ).'">&nbsp;</span></td>';
            }

            echo '</tr>';

            do_action('sfswp_after_settings_field', $field );

            $i++;
        }
    }

    public function render_tab($tabs)
    {
        print('<div id="sfs-nav-settings" class="navtabcontent sfs-nav-settings-content">');

        print('<form action="' . admin_url('options.php') . '" method="post" class="filter-settings-tab sfs-filter-settings-tab">');

        settings_errors();
       
        settings_fields($this->group);

        $this->doSettingsSectionsTab($this->page);

        /*if( apply_filters('sfswp_settings_submit_button', true ) ){
            submit_button();
        }*/

        print('</form>');
        print('</div>');
    }
    public function doSettingsSectionsTab( $page ) {
        global $wp_settings_sections;

        // Step 1: Group sections by title
        $grouped_sections = [];

        foreach ( (array) $wp_settings_sections as $page_key => $sections ) {

            // Only your plugin pages
            if ( false === strpos( $page_key, 'sfswp-filter-' ) ) {
                continue;
            }

            foreach ( (array) $sections as $section ) {
                $title = $section['title'];

                if ( empty( $title ) || empty( $section['id'] ) ) {
                    continue;
                }

                $grouped_sections[ $title ][] = [
                    'page'    => $page_key,
                    'section' => $section['id'],
                ];
            }
        }

      
        if ( empty( $grouped_sections ) ) {
            return;
        }
        $section_order = [
            'Mobile Devices',
            'AJAX Settings',
            'Appearance Settings',
            'WooCommerce',
            'Other Settings',
        ];

        uksort( $grouped_sections, function ( $a, $b ) use ( $section_order ) {

            $pos_a = array_search( $a, $section_order, true );
            $pos_b = array_search( $b, $section_order, true );

            // Titles not found go to the end
            $pos_a = ( $pos_a === false ) ? PHP_INT_MAX : $pos_a;
            $pos_b = ( $pos_b === false ) ? PHP_INT_MAX : $pos_b;

            return $pos_a <=> $pos_b;
        });

        ?>
        <div class="vertical-tabs sfs-nav-settings-content">

            <!-- Tabs -->
            <div class="tab">
                <?php
                $i = 1;
                foreach ( $grouped_sections as $title => $sections_group ) {

                    $title_id = sanitize_title( $title );

                    $active   = ( $i === 1 ) ? 'active' : '';
                    $svg_icon = '';
                    switch ($title_id) {
                        case 'ajax':
                        case 'ajax-settings':
                            $svg_icon = '<svg class="sfs-ajax-svg" width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <rect width="18" height="18" fill="#1E293B"/>
                                <path d="M10.8568 11.4831C11.3115 12.2256 11.9031 12.7713 12.9494 12.7713C13.8284 12.7713 14.3899 12.332 14.3899 11.725C14.3899 10.9976 13.813 10.7399 12.8455 10.3168L12.3152 10.0892C10.7844 9.43708 9.76752 8.62009 9.76752 6.89298C9.76752 5.30203 10.9797 4.09091 12.8742 4.09091C14.2229 4.09091 15.1925 4.5603 15.8912 5.78936L14.2394 6.84999C13.8757 6.19785 13.4833 5.94094 12.8742 5.94094C12.2528 5.94094 11.859 6.33511 11.859 6.84999C11.859 7.48639 12.2532 7.74401 13.1633 8.13818L13.6936 8.36533C15.496 9.13824 16.5136 9.92617 16.5136 11.6977C16.5136 13.6076 15.0133 14.6539 12.9985 14.6539C11.0284 14.6539 9.75566 13.7151 9.13289 12.4846L10.8568 11.4831ZM3.36323 11.6669C3.69646 12.2582 3.99963 12.758 4.72844 12.758C5.42538 12.758 5.86504 12.4854 5.86504 11.4251V4.21207H7.98632V11.4537C7.98632 13.6502 6.69853 14.65 4.81874 14.65C3.12029 14.65 2.13672 13.771 1.63647 12.7123L3.36323 11.6669Z" fill="white"/>
                                </svg>';
                            break;
                        case 'appearance':
                        case 'appearance-settings':
                            $svg_icon = '<svg width="22" height="22" viewBox="0 0 22 22" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <g clip-path="url(#clip0_1858_1561)">
                                <path d="M7.5625 1.375H3.4375C2.29841 1.375 1.375 2.29841 1.375 3.4375V7.5625C1.375 8.70159 2.29841 9.625 3.4375 9.625H7.5625C8.70159 9.625 9.625 8.70159 9.625 7.5625V3.4375C9.625 2.29841 8.70159 1.375 7.5625 1.375Z" fill="#1E293B"/>
                                <path d="M7.5625 12.375H3.4375C2.29831 12.375 1.375 13.2983 1.375 14.4375V18.5625C1.375 19.7017 2.29831 20.625 3.4375 20.625H7.5625C8.70169 20.625 9.625 19.7017 9.625 18.5625V14.4375C9.625 13.2983 8.70169 12.375 7.5625 12.375ZM6.56356 18.3803H4.43506C4.05556 18.3803 3.74756 18.0723 3.74756 17.6928C3.74756 17.3133 4.05556 17.0053 4.43506 17.0053H6.56356C6.94306 17.0053 7.25106 17.3133 7.25106 17.6928C7.25106 18.0723 6.94306 18.3803 6.56356 18.3803Z" fill="#1E293B"/>
                                <path d="M7.5625 10.3125H3.4375C1.92087 10.3125 0.6875 9.07913 0.6875 7.5625V3.4375C0.6875 1.92087 1.92087 0.6875 3.4375 0.6875H7.5625C9.07913 0.6875 10.3125 1.92087 10.3125 3.4375V7.5625C10.3125 9.07913 9.07913 10.3125 7.5625 10.3125ZM3.4375 2.0625C2.67919 2.0625 2.0625 2.67919 2.0625 3.4375V7.5625C2.0625 8.32081 2.67919 8.9375 3.4375 8.9375H7.5625C8.32081 8.9375 8.9375 8.32081 8.9375 7.5625V3.4375C8.9375 2.67919 8.32081 2.0625 7.5625 2.0625H3.4375Z" fill="#1E293B"/>
                                <path d="M18.5625 10.3125H14.4375C12.9209 10.3125 11.6875 9.07913 11.6875 7.5625V3.4375C11.6875 1.92087 12.9209 0.6875 14.4375 0.6875H18.5625C20.0791 0.6875 21.3125 1.92087 21.3125 3.4375V7.5625C21.3125 9.07913 20.0791 10.3125 18.5625 10.3125ZM14.4375 2.0625C13.6792 2.0625 13.0625 2.67919 13.0625 3.4375V7.5625C13.0625 8.32081 13.6792 8.9375 14.4375 8.9375H18.5625C19.3208 8.9375 19.9375 8.32081 19.9375 7.5625V3.4375C19.9375 2.67919 19.3208 2.0625 18.5625 2.0625H14.4375Z" fill="#1E293B"/>
                                <path d="M7.5625 21.3125H3.4375C1.92087 21.3125 0.6875 20.0791 0.6875 18.5625V14.4375C0.6875 12.9209 1.92087 11.6875 3.4375 11.6875H7.5625C9.07913 11.6875 10.3125 12.9209 10.3125 14.4375V18.5625C10.3125 20.0791 9.07913 21.3125 7.5625 21.3125ZM3.4375 13.0625C2.67919 13.0625 2.0625 13.6792 2.0625 14.4375V18.5625C2.0625 19.3208 2.67919 19.9375 3.4375 19.9375H7.5625C8.32081 19.9375 8.9375 19.3208 8.9375 18.5625V14.4375C8.9375 13.6792 8.32081 13.0625 7.5625 13.0625H3.4375Z" fill="#1E293B"/>
                                <path d="M16.5 21.3125C13.8469 21.3125 11.6875 19.1531 11.6875 16.5C11.6875 13.8469 13.8469 11.6875 16.5 11.6875C19.1531 11.6875 21.3125 13.8469 21.3125 16.5C21.3125 19.1531 19.1531 21.3125 16.5 21.3125ZM16.5 13.0625C14.6046 13.0625 13.0625 14.6046 13.0625 16.5C13.0625 18.3954 14.6046 19.9375 16.5 19.9375C18.3954 19.9375 19.9375 18.3954 19.9375 16.5C19.9375 14.6046 18.3954 13.0625 16.5 13.0625Z" fill="#1E293B"/>
                                <path d="M18.5625 1.375H14.4375C13.2984 1.375 12.375 2.29841 12.375 3.4375V7.5625C12.375 8.70159 13.2984 9.625 14.4375 9.625H18.5625C19.7016 9.625 20.625 8.70159 20.625 7.5625V3.4375C20.625 2.29841 19.7016 1.375 18.5625 1.375Z" fill="#1E293B"/>
                                </g>
                                <defs>
                                <clipPath id="clip0_1858_1561">
                                <rect width="22" height="22" fill="white"/>
                                </clipPath>
                                </defs>
                                </svg>';
                            break;
                        case 'woocommerce':
                            $svg_icon = '<svg width="25" height="25" viewBox="0 0 25 25" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path fill-rule="evenodd" clip-rule="evenodd" d="M21.7461 6C20.5932 6 19.843 6.37607 19.1728 7.6386L16.1173 13.4139V8.28327C16.1173 6.75214 15.3934 6 14.0531 6C12.7133 6 12.1505 6.45663 11.4803 7.74602L8.5854 13.4139V8.33698C8.5854 6.69843 7.91524 6 6.2801 6H2.95661C1.69686 6 1 6.59096 1 7.66546C1 8.73991 1.67007 9.38462 2.90292 9.38462H4.27009V15.8583C4.27009 17.685 5.50303 18.7594 7.27192 18.7594C9.04128 18.7594 9.84519 18.061 10.7296 16.4224L12.6596 12.7961V15.8583C12.6596 17.6581 13.8388 18.7594 15.6346 18.7594C17.4308 18.7594 18.1005 18.1416 19.1191 16.4224L23.5687 8.90109C24.5336 7.26253 23.8635 6 21.7192 6C21.7192 6 21.7192 6 21.7461 6Z" fill="#1E293B"/>
                            </svg> ';
                            break;
                        case 'mobile-devices':
                            $svg_icon = '<svg width="25" height="25" viewBox="0 0 25 25" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M12.5 22.2656C12.7157 22.2656 12.8906 22.0907 12.8906 21.875C12.8906 21.6593 12.7157 21.4844 12.5 21.4844C12.2843 21.4844 12.1094 21.6593 12.1094 21.875C12.1094 22.0907 12.2843 22.2656 12.5 22.2656Z" fill="#13648A"/>
                            <path d="M16.4062 1.17188H8.59375C7.97234 1.1725 7.37656 1.41962 6.93715 1.85903C6.49775 2.29843 6.25062 2.89421 6.25 3.51562H18.75C18.7494 2.89421 18.5023 2.29843 18.0628 1.85903C17.6234 1.41962 17.0277 1.1725 16.4062 1.17188ZM13.6719 2.73438H11.3281C11.2245 2.73438 11.1252 2.69322 11.0519 2.61996C10.9787 2.54671 10.9375 2.44735 10.9375 2.34375C10.9375 2.24015 10.9787 2.14079 11.0519 2.06754C11.1252 1.99428 11.2245 1.95312 11.3281 1.95312H13.6719C13.7755 1.95312 13.8748 1.99428 13.9481 2.06754C14.0213 2.14079 14.0625 2.24015 14.0625 2.34375C14.0625 2.44735 14.0213 2.54671 13.9481 2.61996C13.8748 2.69322 13.7755 2.73438 13.6719 2.73438Z" fill="#13648A"/>
                            <path d="M6.25 4.29688H18.75V19.1406H6.25V4.29688Z" fill="#13648A"/>
                            <path d="M6.25 21.4844C6.25062 22.1058 6.49775 22.7016 6.93715 23.141C7.37656 23.5804 7.97234 23.8275 8.59375 23.8281H16.4062C17.0277 23.8275 17.6234 23.5804 18.0628 23.141C18.5023 22.7016 18.7494 22.1058 18.75 21.4844V19.9219H6.25V21.4844ZM12.5 20.7031C12.7318 20.7031 12.9583 20.7719 13.1511 20.9006C13.3438 21.0294 13.494 21.2124 13.5827 21.4265C13.6714 21.6407 13.6946 21.8763 13.6494 22.1036C13.6041 22.3309 13.4925 22.5398 13.3286 22.7036C13.1648 22.8675 12.9559 22.9791 12.7286 23.0244C12.5013 23.0696 12.2657 23.0464 12.0515 22.9577C11.8374 22.869 11.6544 22.7188 11.5256 22.5261C11.3969 22.3333 11.3281 22.1068 11.3281 21.875C11.3281 21.5642 11.4516 21.2661 11.6714 21.0464C11.8911 20.8266 12.1892 20.7031 12.5 20.7031Z" fill="#13648A"/>
                            </svg>
                            ';
                            break;
                        default:
                            $svg_icon = '<svg width="21" height="21" viewBox="0 0 21 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M8.82582 0L7.56858 2.99223C7.31713 3.06767 7.09083 3.19339 6.86453 3.31911L3.8723 2.06187L2.06187 3.8723L3.31911 6.86453C3.19339 7.11598 3.09281 7.31713 2.99223 7.56858L0 8.82582V11.3403L2.99223 12.5975C3.09281 12.849 3.19339 13.0501 3.31911 13.3016L2.06187 16.2938L3.8723 18.1043L6.86453 16.847C7.09083 16.9476 7.31713 17.0733 7.56858 17.1739L8.82582 20.1661H11.3403L12.5975 17.1739C12.8238 17.0733 13.0753 16.9727 13.3016 16.847L16.2938 18.1043L18.1043 16.2938L16.847 13.3016C16.9476 13.0753 17.0733 12.8238 17.1739 12.5975L20.1661 11.3403V8.82582L17.1739 7.56858C17.0985 7.34228 16.9727 7.09083 16.847 6.86453L18.1043 3.8723L16.2938 2.06187L13.3016 3.31911C13.0753 3.21853 12.8238 3.09281 12.5975 2.99223L11.3403 0L8.82582 0ZM10.0831 6.2862C12.1701 6.2862 13.8548 7.9709 13.8548 10.0579C13.8548 12.1449 12.1701 13.8296 10.0831 13.8296C7.99605 13.8296 6.31134 12.1449 6.31134 10.0579C6.31134 7.9709 7.99605 6.2862 10.0831 6.2862Z" fill="#1E293B"/>
                            </svg>
                            ';
                            break;
                    }
                    if( !in_array( $title_id, array('global-filter-prefixes-and-order', 'global-url-var-names' ), true) ){
                    ?>
                    <button
                        class="tablinks <?php echo esc_attr( $active ); ?>"
                        onclick="sfsTab(event, '<?php echo esc_attr( $title_id ); ?>')">
                        <span class="tabicon">
                            <?php echo $svg_icon; ?>
                        </span>
                        <?php echo $title; ?>
                    </button>
                    <?php
                    }
                    $i++;
                }
                ?>
            </div>

            <!-- Tab contents -->
            <?php
            $i = 1;
            foreach ( $grouped_sections as $title => $sections_group ) {

                $title_id = sanitize_title( $title );
                $active   = ( $i === 1 ) ? 'active' : '';
                ?>
                <div id="<?php echo esc_attr( $title_id ); ?>" class="tabcontent <?php echo esc_attr( $active ); ?>">
                    <div class="generalSetting-section">
                        <table class="sfswp-form-table form-table" role="presentation">
                            <?php
                            
                            // Render ALL sections with the same title
                            foreach ( $sections_group as $group ) {
                                // Capture BEFORE action output
                                ob_start();
                                do_action( 'sfswp_before_sections_settings_fields', $group['page'] );
                                $before_output = trim( ob_get_clean() );

                                if ( $before_output !== '' ) {
                                    echo '<tr><td colspan="2">';
                                    echo $before_output;
                                    echo '</td></tr>';
                                }
                                $this->doSettingsFieldsTab( $group['page'], $group['section'] );
                                // Capture AFTER action output
                                ob_start();
                                do_action( 'sfswp_after_sections_settings_fields', $group['page'] );
                                $after_output = trim( ob_get_clean() );

                                if ( $after_output !== '' ) {
                                    echo '<tr><td colspan="2">';
                                    echo $after_output;
                                    echo '</td></tr>';
                                }
                            }

                            
                            ?>
                        </table>
                    </div>

                    <?php if( apply_filters('sfswp_settings_submit_button', true ) ){
                        echo '<div class="generalSetting-section generalSetting-section-button">';
                        submit_button();
                        echo '</div>';
                    } ?>
                </div>
                <?php
                $i++;
            }
            ?>
        </div>
        <?php
    }

    public function doSettingsFieldsTab( $page, $section ) {
        global $wp_settings_fields;

        $i = 1;

        foreach ( (array) $wp_settings_fields[ $page ][ $section ] as $field ) {
            $class = '';

            if( $field['id'] === 'bottom_widget_compatibility' ){
                if( sfs_get_option('mobile_filter_settings') === 'show_bottom_widget' ){
                    $field['args']['class'] .= ' sfswp-opened';
                }
            }

            if( $field['id'] === 'color_swatches_taxonomies' || $field['id'] === 'rounded_swatches' ) {
                if( sfs_get_experimental_option('use_color_swatches') === 'on' ) {
                    $field['args']['class'] .= ' sfswp-opened';
                }
            }

            if ( ! empty( $field['args']['class'] ) ) {
                $class = ' class="' . esc_attr( $field['args']['class'] ) . '"';
            }

            do_action('sfswp_before_settings_field', $field );

            echo "<tr{$class}>";

            $sortable = isset( $field['args']['sortable'] ) ? $i : '';

            if( $sortable ){
                echo '<td class="sfswp-order-td">'.esc_html($sortable).'</td>';
            }

            $tooltip = isset( $field['args']['tooltip'] ) ? sfs_tooltip( array( 'tooltip' => $field['args']['tooltip'] ) ) : '';
            $tooltip = wp_kses(
                $tooltip,
                array(
                    'strong' => array(),
                    'br'     => array(),
                    'a'      => array( 'href'=>true, 'title'=>true, 'class'=>true ),
                    'span'   => array( 'class'=>true, 'data-tip'=>true)
                )
            );

            if ( ! empty( $field['args']['label_for'] ) ) {
                echo '<th scope="row"><label for="' . esc_attr( $field['args']['label_for'] ) . '">' . esc_html($field['title']) . '</label> ' .$tooltip. '</th>'; // $tooltip already escaped
            } else {
                echo '<th scope="row">'. esc_html($field['title'] ) . ' ' .$tooltip. '</th>'; // $tooltip already escaped
            }

            echo '<td>';
            call_user_func( $field['callback'], $field['args'] );
            echo '</td>';

            if( $sortable ){
                $title = ( ! defined( 'SFS_FILTERS_PRO' ) ) ? __( 'Editing the order of URL prefixes is available in the PRO version', 'searchfiltersort' ) : '';
                echo '<td class="sfswp-order-sortable-handle-icon"><span class="dashicons dashicons-menu sfswp-field-sortable-handle" title="'.esc_attr( $title ).'">&nbsp;</span></td>';
            }

            echo '</tr>';

            do_action('sfswp_after_settings_field', $field );

            $i++;
        }
    }

    public function getOption($key, $default = null)
    {
        if (is_null($this->options)) {
            $this->options = get_option($this->optionName);
        }

        return isset($this->options[$key]) ? $this->options[$key] : $default;
    }
    public function get_group() {
        return $this->group;
    }
}
