<?php


namespace SearchFilterSort\Filter;

if ( ! defined('ABSPATH') ) {
    exit;
}

class SFSFields
{

    const SFS_NEW_FILTER_ID = 'sfswp_new_id';

    const FILTER_FIELD_KEY = 'sfswp_filter_fields';

    private $defaultFields = [];

    private $searchFields = [];

    private $sortFields = [];

    private $fse;

    private $em;

    private $hooksRegistered = false;

    private $errors;

    public function __construct()
    {
        $this->fse = Container::instance()->getFilterService();
        $this->em  = Container::instance()->getEntityManager();
        $this->setupDefaultFields();
        $this->setupSearchFields();
        $this->setupSortFields();
    }

    public function registerHooks()
    {
        if ( ! $this->hooksRegistered ) {
            add_filter( 'sfswp_input_type_select', [ $this, 'addSpinnerToSelect' ], 10, 2 );
            add_filter( 'sfswp_input_type_radio', [ $this, 'addSpinnerToDateFormats' ], 10, 2 );

            add_action( 'wp_ajax_sfswp-delete-filter',  [ $this, 'ajaxDeleteFilter' ] );
            add_action( 'wp_ajax_sfswp-load-exclude-terms', [ $this, 'sendExcludedTerms' ] );
            add_action( 'wp_ajax_sfswp_get_date_formats', [ $this, 'sendDateFormats' ] );
            add_action( 'wp_ajax_sfswp-validate-filters', [ $this, 'ajaxValidateFilters' ] );
            add_action( 'after_delete_post', [ $this, 'deleteRelatedFilters' ], 10, 2 );

            $this->hooksRegistered = true;
        }
    }

    private function setupDefaultFields()
    {
        // maybe add filter in future to allow change default fields
        do_action( 'sfswp_before_setup_filter_fields' );
        $entity     = $this->em->getEntityByFilter( array('entity' => 'taxonomy', 'e_name' => 'category'), 'post' );
        $entityTerms = $entity->getTermsForSelect();
        asort( $entityTerms, SORT_NATURAL );

        $defaultFields = array(
            'ID' => array(
                'type'          => 'Hidden'
            ),
            'parent' => array(
                'type'          => 'Hidden'
            ),
            'menu_order' => array(
                'type'          => 'Hidden',
                'class'         => 'sfswp-menu-order-field'
            ),
            'entity' => array(
                'type'          => 'Select',
                'label'         => esc_html__(  'Filter by', 'searchfiltersort' ),
                'class'         => 'sfswp-field-entity',
                'options'       => $this->em->getPossibleEntities(),
                'default'       => 'taxonomy_category',
                'instructions'  => esc_html__(  'A thing by which posts will be filtered', 'searchfiltersort' ),
                'required'      => true
            ),
            'e_name' => array(
                'type'          => 'Text',
                'label'         => esc_html__(  'Meta Key', 'searchfiltersort' ),
                'class'         => 'sfswp-field-ename',
                'instructions'  => esc_html__( 'Name of the Custom Field. Please, see the Popular Meta keys at the bottom', 'searchfiltersort'),
                'required'      => true
            ),
            'label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Filter Title', 'searchfiltersort' ),
                'class'         => 'sfswp-field-label',
                'placeholder'   => esc_html__( 'New Filter', 'searchfiltersort'),
                'instructions'  => esc_html__(  'Will appear in the Filters widget', 'searchfiltersort' )
            ),
            'slug' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Var Name for URL', 'searchfiltersort' ),
                'class'         => 'sfswp-field-slug',
                'instructions'  => esc_html__( 'Name of a part of the URL responsible for this filter', 'searchfiltersort'),
                'tooltip'       => wp_kses(
                    __( 'For example, in the URL path:<br />/?color=blue&size=large<br />"color" and "size" are the names of the URL vars.<br />In the PRO version of the plugin, the URL part after "?" becomes <br />/color-blue/size-large/', 'searchfiltersort'),
                    array( 'br' => array() )
                ),
                'required'      => true
            ), // Optional
            'view' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'View in Widget', 'searchfiltersort' ),
                'class'         => 'sfswp-field-view',
                'options'       => $this->getViewOptions(),
                'default'       => 'checkboxes',
                'instructions'  => '',
            ),
            'dropdown_label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Dropdown Label', 'searchfiltersort' ),
                'class'         => 'sfswp-field-dropdown-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. - Select color -', 'searchfiltersort' ),
                'instructions'  => esc_html__( 'Label for the default dropdown option', 'searchfiltersort' ),
            ),
            'date_type'         => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Date Type', 'searchfiltersort' ),
                'class'         => 'sfswp-date-type',
                'options'       => array(
                    'date'      => esc_html__( 'Date', 'searchfiltersort' ),
                    'datetime'  => esc_html__( 'Date Time', 'searchfiltersort' ),
                    'time'      => esc_html__( 'Time', 'searchfiltersort' ),
                ),
                'default'       => 'date',
                'instructions'  => '',
            ),
            'date_format'         => array(
                'type'          => 'Radio',
                'label'         => esc_html__( 'Date Format', 'searchfiltersort' ),
                'class'         => 'sfswp-date-format',
                'options'       => $this->getDateFormatOptions(),
                'default'       => __( 'F j, Y' ),
                'instructions'  => esc_html__(  'How the date will be displayed in the Filters widget', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    sprintf( __( 'More PHP date formats can be found on <a href="%1$s" target="_blank">this page</a>.', 'searchfiltersort' ), 'https://wordpress.org/documentation/article/customize-date-and-time-format/' ),
                    array(
                        'a' => array(
                            'href'     => true,
                        ) )
                )
            ),
            'show_term_names' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show Term names', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-term-names',
                'default'       => 'yes',
                'instructions'  => esc_html__(  'Term names for filters with swatches or brand images may be hidden', 'searchfiltersort' ),
                'tooltip'       => '',
            ),
            'selected_and_above' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Mode «All stars+»', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-selected-and-above',
                'default'       => 'no',
                'instructions'  => esc_html__(  'E.g. if 3 stars are selected, all products with rating 3+ will be shown', 'searchfiltersort' ),
                'tooltip'       => '',
            ),
            'logic' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Filter Logic', 'searchfiltersort' ),
                'class'         => 'sfswp-field-logic',
                'options'       => array('or' => esc_html__('OR', 'searchfiltersort' ), 'and' => esc_html__('AND', 'searchfiltersort') ),
                'default'       => 'or',
                'instructions'  => esc_html__( 'Determines how to select posts when two or more terms of this filter are selected', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    __( '«OR» means to show posts if they are at least in one of the selected terms. <br />«AND» means that posts should belong to all selected terms at the same time.', 'searchfiltersort' ),
                    array( 'br' => array() )
                ),
            ), // AND
            'orderby' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Sort Terms by', 'searchfiltersort' ),
                'class'         => 'sfswp-field-orderby',
                'options'       => $this->getOrderByOptions(),
                'default'       => 'default',
                'instructions'  => esc_html__('The order in which terms appear in the widget', 'searchfiltersort'),
                'tooltip'       => wp_kses(
                    __( 'The "Default" option means that terms will be sorted in the order as they were received from the Database<br />The "Menu order" option is available for WooCommerce attributes only.', 'searchfiltersort' ),
                    array( 'br' => array() )
                )
            ),
            'in_path' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'In URL', 'searchfiltersort' ),
                'class'         => 'sfswp-field-path',
                'default'       => 'yes',
                'instructions'  => ''
            ),
            'exclude' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Include/Exclude Terms', 'searchfiltersort' ),
                'class'         => 'sfswp-field-exclude',
                'options'       => $entityTerms, //$entity->getTermsForSelect(),
                'multiple'      => 'multiple',
                'instructions'  => '',
                'skip_view'     => true
            ),
            'include' => array(
                'type'          => 'Select',
                'label'         => '',
                'class'         => 'sfswp-field-include',
                'options'       => [ 'no' => esc_html__( 'exclude', 'searchfiltersort' ), 'yes' => esc_html__('include', 'searchfiltersort' ) ],
                'default'       => 'no',
                'instructions'  => esc_html__( 'Include selected terms only', 'searchfiltersort' ),
                'skip_view'     => true
            ),
            'collapse' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Folding', 'searchfiltersort' ),
                'class'         => 'sfswp-field-collapse',
                'default'       => 'no',
                'instructions'  =>  esc_html__( 'Makes filter collapsible in the widget', 'searchfiltersort'),
                'tooltip'       => esc_html__( 'Useful in situations when the filter is rarely applied but takes up some space in the widget. Collapsed by default.', 'searchfiltersort' )
            ),
            'hierarchy' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show Hierarchy', 'searchfiltersort' ),
                'class'         => 'sfswp-field-hierarchy',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Display the term hierarchy in the filter widget. Child terms will be collapsed by default', 'searchfiltersort' )
            ),
            'range_slider' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Enable Range Slider?', 'searchfiltersort' ),
                'class'         => 'sfswp-field-range-slider',
                'default'       => 'yes',
                'instructions'  => esc_html__( 'If disabled, visitors must type numeric values in text inputs', 'searchfiltersort' )
            ),
            'step'      => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Slider Step', 'searchfiltersort' ),
                'class'         => 'sfswp-field-value-step',
                'instructions'  => esc_html__( 'Determines how the numeric value will be changed when you move slider controls', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    __('Step 1 means possible values are 1,2,3,4 ...<br />Step 0.1 means possible values are 5.1, 5.2, 5.3, 5.4 ...<br />Step 15 means possible values are 15, 30, 45, 60...', 'searchfiltersort'),
                    array( 'br' => array() )
                ),
                'default'       => 1
            ),
            'search' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Terms Search', 'searchfiltersort' ),
                'class'         => 'sfswp-field-search',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Adds a search field that allows you to quickly find filter terms', 'searchfiltersort' )
            ),
            'parent_filter' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Parent Filter', 'searchfiltersort' ),
                'class'         => 'sfswp-field-parent-filter',
                'options'       => [ 'no' => esc_html__( 'Please, add filters first', 'searchfiltersort' ) ],
                'instructions'  => esc_html__( 'If specified, current Filter terms become available only after the parent Filter selected', 'searchfiltersort' )
            ),
            'hide_until_parent' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Hide until the Parent selected', 'searchfiltersort' ),
                'class'         => 'sfswp-field-hide-until-parent',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Do not show this Filter until a parent is selected. Useful for step-by-step filtering', 'searchfiltersort' )
            ),
            'min_num_label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Labels for Chips', 'searchfiltersort' ),
                'class'         => 'sfswp-field-min-num-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. From {value}', 'searchfiltersort' ),
                'instructions'  => '',
                'skip_view'     => true,
            ),
            'max_num_label' => array(
                'type'          => 'Text',
                'label'         => '',
                'class'         => 'sfswp-field-max-num-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. To {value}', 'searchfiltersort' ),
                'instructions'  => '',
                'skip_view'     => true,
            ),
            'tooltip' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Tooltip', 'searchfiltersort' ),
                'class'         => 'sfswp-field-tooltip',
                'instructions'  => esc_html__( 'Will appear next to the Filter Title', 'searchfiltersort' ),
                'default'       => ''
            ),
            'more_less' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'More/Less', 'searchfiltersort' ),
                'class'         => 'sfswp-field-more-less',
                'default'       => 'no',
                'instructions'  => sprintf( esc_html__( 'Show a toggle link after %s terms', 'searchfiltersort' ), sfs_more_less_count() )
            ),
            'show_chips' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show in Chips', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-chips',
                'default'       => 'yes',
                'instructions'  => esc_html__( 'Show filter selected terms in the list of all chosen items', 'searchfiltersort' )
            ),
            'acf_fields' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'ACF Fields', 'searchfiltersort' ),
                'class'         => 'sfswp-field-acf',
                'default'       => '',
                'instructions'  => esc_html__( 'Contains the IDs of ACF fields that are related to the Custom Field', 'searchfiltersort' )
            )
        );

        $this->defaultFields = apply_filters( 'sfswp_filter_default_fields', $defaultFields, $this );
    }
    private function setupSearchFields()
    {
        // maybe add filter in future to allow change default fields
        do_action( 'sfswp_before_setup_filter_fields' );
        $entity     = $this->em->getEntityByFilter( array('entity' => 'taxonomy', 'e_name' => 'category'), 'post' );
        $entityTerms = $entity->getTermsForSelect();
        asort( $entityTerms, SORT_NATURAL );

        $searchFields = array(
            'ID' => array(
                'type'          => 'Hidden'
            ),
            'parent' => array(
                'type'          => 'Hidden'
            ),
            'menu_order' => array(
                'type'          => 'Hidden',
                'class'         => 'sfswp-menu-order-field'
            ),
            'entity' => array(
                'type'          => 'Select',
                'label'         => esc_html__(  'Filter by', 'searchfiltersort' ),
                'class'         => 'sfswp-field-entity',
                'options'       => $this->em->getPossibleEntities(),
                'default'       => 'taxonomy_category',
                'instructions'  => esc_html__(  'A thing by which posts will be filtered', 'searchfiltersort' ),
                'required'      => true
            ),
            'e_name' => array(
                'type'          => 'Text',
                'label'         => esc_html__(  'Meta Key', 'searchfiltersort' ),
                'class'         => 'sfswp-field-ename',
                'instructions'  => esc_html__( 'Name of the Custom Field. Please, see the Popular Meta keys at the bottom', 'searchfiltersort'),
                'required'      => true
            ),
            'label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Filter Title', 'searchfiltersort' ),
                'class'         => 'sfswp-field-label',
                'placeholder'   => esc_html__( 'New Filter', 'searchfiltersort'),
                'instructions'  => esc_html__(  'Will appear in the Filters widget', 'searchfiltersort' )
            ),
            'slug' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Var Name for URL', 'searchfiltersort' ),
                'class'         => 'sfswp-field-slug',
                'instructions'  => esc_html__( 'Name of a part of the URL responsible for this filter', 'searchfiltersort'),
                'tooltip'       => wp_kses(
                    __( 'For example, in the URL path:<br />/?color=blue&size=large<br />"color" and "size" are the names of the URL vars.<br />In the PRO version of the plugin, the URL part after "?" becomes <br />/color-blue/size-large/', 'searchfiltersort'),
                    array( 'br' => array() )
                ),
                'required'      => true
            ), // Optional
            'view' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'View in Widget', 'searchfiltersort' ),
                'class'         => 'sfswp-field-view',
                'options'       => $this->getViewOptions(),
                'default'       => 'checkboxes',
                'instructions'  => '',
            ),
            'dropdown_label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Dropdown Label', 'searchfiltersort' ),
                'class'         => 'sfswp-field-dropdown-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. - Select color -', 'searchfiltersort' ),
                'instructions'  => esc_html__( 'Label for the default dropdown option', 'searchfiltersort' ),
            ),
            'date_type'         => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Date Type', 'searchfiltersort' ),
                'class'         => 'sfswp-date-type',
                'options'       => array(
                    'date'      => esc_html__( 'Date', 'searchfiltersort' ),
                    'datetime'  => esc_html__( 'Date Time', 'searchfiltersort' ),
                    'time'      => esc_html__( 'Time', 'searchfiltersort' ),
                ),
                'default'       => 'date',
                'instructions'  => '',
            ),
            'date_format'         => array(
                'type'          => 'Radio',
                'label'         => esc_html__( 'Date Format', 'searchfiltersort' ),
                'class'         => 'sfswp-date-format',
                'options'       => $this->getDateFormatOptions(),
                'default'       => __( 'F j, Y' ),
                'instructions'  => esc_html__(  'How the date will be displayed in the Filters widget', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    sprintf( __( 'More PHP date formats can be found on <a href="%1$s" target="_blank">this page</a>.', 'searchfiltersort' ), 'https://wordpress.org/documentation/article/customize-date-and-time-format/' ),
                    array(
                        'a' => array(
                            'href'     => true,
                        ) )
                )
            ),
            'show_term_names' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show Term names', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-term-names',
                'default'       => 'yes',
                'instructions'  => esc_html__(  'Term names for filters with swatches or brand images may be hidden', 'searchfiltersort' ),
                'tooltip'       => '',
            ),
            'selected_and_above' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Mode «All stars+»', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-selected-and-above',
                'default'       => 'no',
                'instructions'  => esc_html__(  'E.g. if 3 stars are selected, all products with rating 3+ will be shown', 'searchfiltersort' ),
                'tooltip'       => '',
            ),
            'logic' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Filter Logic', 'searchfiltersort' ),
                'class'         => 'sfswp-field-logic',
                'options'       => array('or' => esc_html__('OR', 'searchfiltersort' ), 'and' => esc_html__('AND', 'searchfiltersort') ),
                'default'       => 'or',
                'instructions'  => esc_html__( 'Determines how to select posts when two or more terms of this filter are selected', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    __( '«OR» means to show posts if they are at least in one of the selected terms. <br />«AND» means that posts should belong to all selected terms at the same time.', 'searchfiltersort' ),
                    array( 'br' => array() )
                ),
            ), // AND
            'orderby' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Sort Terms by', 'searchfiltersort' ),
                'class'         => 'sfswp-field-orderby',
                'options'       => $this->getOrderByOptions(),
                'default'       => 'default',
                'instructions'  => esc_html__('The order in which terms appear in the widget', 'searchfiltersort'),
                'tooltip'       => wp_kses(
                    __( 'The "Default" option means that terms will be sorted in the order as they were received from the Database<br />The "Menu order" option is available for WooCommerce attributes only.', 'searchfiltersort' ),
                    array( 'br' => array() )
                )
            ),
            'in_path' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'In URL', 'searchfiltersort' ),
                'class'         => 'sfswp-field-path',
                'default'       => 'yes',
                'instructions'  => ''
            ),
            'exclude' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Include/Exclude Terms', 'searchfiltersort' ),
                'class'         => 'sfswp-field-exclude',
                'options'       => $entityTerms, //$entity->getTermsForSelect(),
                'multiple'      => 'multiple',
                'instructions'  => '',
                'skip_view'     => true
            ),
            'include' => array(
                'type'          => 'Select',
                'label'         => '',
                'class'         => 'sfswp-field-include',
                'options'       => [ 'no' => esc_html__( 'exclude', 'searchfiltersort' ), 'yes' => esc_html__('include', 'searchfiltersort' ) ],
                'default'       => 'no',
                'instructions'  => esc_html__( 'Include selected terms only', 'searchfiltersort' ),
                'skip_view'     => true
            ),
            'collapse' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Folding', 'searchfiltersort' ),
                'class'         => 'sfswp-field-collapse',
                'default'       => 'no',
                'instructions'  =>  esc_html__( 'Makes filter collapsible in the widget', 'searchfiltersort'),
                'tooltip'       => esc_html__( 'Useful in situations when the filter is rarely applied but takes up some space in the widget. Collapsed by default.', 'searchfiltersort' )
            ),
            'hierarchy' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show Hierarchy', 'searchfiltersort' ),
                'class'         => 'sfswp-field-hierarchy',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Display the term hierarchy in the filter widget. Child terms will be collapsed by default', 'searchfiltersort' )
            ),
            'range_slider' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Enable Range Slider?', 'searchfiltersort' ),
                'class'         => 'sfswp-field-range-slider',
                'default'       => 'yes',
                'instructions'  => esc_html__( 'If disabled, visitors must type numeric values in text inputs', 'searchfiltersort' )
            ),
            'step'      => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Slider Step', 'searchfiltersort' ),
                'class'         => 'sfswp-field-value-step',
                'instructions'  => esc_html__( 'Determines how the numeric value will be changed when you move slider controls', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    __('Step 1 means possible values are 1,2,3,4 ...<br />Step 0.1 means possible values are 5.1, 5.2, 5.3, 5.4 ...<br />Step 15 means possible values are 15, 30, 45, 60...', 'searchfiltersort'),
                    array( 'br' => array() )
                ),
                'default'       => 1
            ),
            'search' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Terms Search', 'searchfiltersort' ),
                'class'         => 'sfswp-field-search',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Adds a search field that allows you to quickly find filter terms', 'searchfiltersort' )
            ),
            'parent_filter' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Parent Filter', 'searchfiltersort' ),
                'class'         => 'sfswp-field-parent-filter',
                'options'       => [ 'no' => esc_html__( 'Please, add filters first', 'searchfiltersort' ) ],
                'instructions'  => esc_html__( 'If specified, current Filter terms become available only after the parent Filter selected', 'searchfiltersort' )
            ),
            'hide_until_parent' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Hide until the Parent selected', 'searchfiltersort' ),
                'class'         => 'sfswp-field-hide-until-parent',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Do not show this Filter until a parent is selected. Useful for step-by-step filtering', 'searchfiltersort' )
            ),
            'min_num_label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Labels for Chips', 'searchfiltersort' ),
                'class'         => 'sfswp-field-min-num-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. From {value}', 'searchfiltersort' ),
                'instructions'  => '',
                'skip_view'     => true,
            ),
            'max_num_label' => array(
                'type'          => 'Text',
                'label'         => '',
                'class'         => 'sfswp-field-max-num-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. To {value}', 'searchfiltersort' ),
                'instructions'  => '',
                'skip_view'     => true,
            ),
            'tooltip' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Tooltip', 'searchfiltersort' ),
                'class'         => 'sfswp-field-tooltip',
                'instructions'  => esc_html__( 'Will appear next to the Filter Title', 'searchfiltersort' ),
                'default'       => ''
            ),
            'more_less' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'More/Less', 'searchfiltersort' ),
                'class'         => 'sfswp-field-more-less',
                'default'       => 'no',
                'instructions'  => sprintf( esc_html__( 'Show a toggle link after %s terms', 'searchfiltersort' ), sfs_more_less_count() )
            ),
            'show_chips' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show in Chips', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-chips',
                'default'       => 'yes',
                'instructions'  => esc_html__( 'Show filter selected terms in the list of all chosen items', 'searchfiltersort' )
            ),
            'acf_fields' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'ACF Fields', 'searchfiltersort' ),
                'class'         => 'sfswp-field-acf',
                'default'       => '',
                'instructions'  => esc_html__( 'Contains the IDs of ACF fields that are related to the Custom Field', 'searchfiltersort' )
            )
        );

        $this->searchFields = apply_filters( 'sfswp_filter_search_fields', $searchFields, $this );
    }

    private function setupSortFields()
    {
        // maybe add filter in future to allow change default fields
        do_action( 'sfswp_before_setup_filter_fields' );
        $entity     = $this->em->getEntityByFilter( array('entity' => 'taxonomy', 'e_name' => 'category'), 'post' );
        $entityTerms = $entity->getTermsForSelect();
        asort( $entityTerms, SORT_NATURAL );

        $sortFields = array(
            'ID' => array(
                'type'          => 'Hidden'
            ),
            'parent' => array(
                'type'          => 'Hidden'
            ),
            'menu_order' => array(
                'type'          => 'Hidden',
                'class'         => 'sfswp-menu-order-field'
            ),
            'entity' => array(
                'type'          => 'Select',
                'label'         => esc_html__(  'Filter by', 'searchfiltersort' ),
                'class'         => 'sfswp-field-entity',
                'options'       => $this->em->getPossibleEntities(),
                'default'       => 'taxonomy_category',
                'instructions'  => esc_html__(  'A thing by which posts will be filtered', 'searchfiltersort' ),
                'required'      => true
            ),
            'e_name' => array(
                'type'          => 'Text',
                'label'         => esc_html__(  'Meta Key', 'searchfiltersort' ),
                'class'         => 'sfswp-field-ename',
                'instructions'  => esc_html__( 'Name of the Custom Field. Please, see the Popular Meta keys at the bottom', 'searchfiltersort'),
                'required'      => true
            ),
            'label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Filter Title', 'searchfiltersort' ),
                'class'         => 'sfswp-field-label',
                'placeholder'   => esc_html__( 'New Filter', 'searchfiltersort'),
                'instructions'  => esc_html__(  'Will appear in the Filters widget', 'searchfiltersort' )
            ),
            'slug' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Var Name for URL', 'searchfiltersort' ),
                'class'         => 'sfswp-field-slug',
                'instructions'  => esc_html__( 'Name of a part of the URL responsible for this filter', 'searchfiltersort'),
                'tooltip'       => wp_kses(
                    __( 'For example, in the URL path:<br />/?color=blue&size=large<br />"color" and "size" are the names of the URL vars.<br />In the PRO version of the plugin, the URL part after "?" becomes <br />/color-blue/size-large/', 'searchfiltersort'),
                    array( 'br' => array() )
                ),
                'required'      => true
            ), // Optional
            'view' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'View in Widget', 'searchfiltersort' ),
                'class'         => 'sfswp-field-view',
                'options'       => $this->getViewOptions(),
                'default'       => 'checkboxes',
                'instructions'  => '',
            ),
            'dropdown_label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Dropdown Label', 'searchfiltersort' ),
                'class'         => 'sfswp-field-dropdown-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. - Select color -', 'searchfiltersort' ),
                'instructions'  => esc_html__( 'Label for the default dropdown option', 'searchfiltersort' ),
            ),
            'date_type'         => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Date Type', 'searchfiltersort' ),
                'class'         => 'sfswp-date-type',
                'options'       => array(
                    'date'      => esc_html__( 'Date', 'searchfiltersort' ),
                    'datetime'  => esc_html__( 'Date Time', 'searchfiltersort' ),
                    'time'      => esc_html__( 'Time', 'searchfiltersort' ),
                ),
                'default'       => 'date',
                'instructions'  => '',
            ),
            'date_format'         => array(
                'type'          => 'Radio',
                'label'         => esc_html__( 'Date Format', 'searchfiltersort' ),
                'class'         => 'sfswp-date-format',
                'options'       => $this->getDateFormatOptions(),
                'default'       => __( 'F j, Y' ),
                'instructions'  => esc_html__(  'How the date will be displayed in the Filters widget', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    sprintf( __( 'More PHP date formats can be found on <a href="%1$s" target="_blank">this page</a>.', 'searchfiltersort' ), 'https://wordpress.org/documentation/article/customize-date-and-time-format/' ),
                    array(
                        'a' => array(
                            'href'     => true,
                        ) )
                )
            ),
            'show_term_names' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show Term names', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-term-names',
                'default'       => 'yes',
                'instructions'  => esc_html__(  'Term names for filters with swatches or brand images may be hidden', 'searchfiltersort' ),
                'tooltip'       => '',
            ),
            'selected_and_above' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Mode «All stars+»', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-selected-and-above',
                'default'       => 'no',
                'instructions'  => esc_html__(  'E.g. if 3 stars are selected, all products with rating 3+ will be shown', 'searchfiltersort' ),
                'tooltip'       => '',
            ),
            'logic' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Filter Logic', 'searchfiltersort' ),
                'class'         => 'sfswp-field-logic',
                'options'       => array('or' => esc_html__('OR', 'searchfiltersort' ), 'and' => esc_html__('AND', 'searchfiltersort') ),
                'default'       => 'or',
                'instructions'  => esc_html__( 'Determines how to select posts when two or more terms of this filter are selected', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    __( '«OR» means to show posts if they are at least in one of the selected terms. <br />«AND» means that posts should belong to all selected terms at the same time.', 'searchfiltersort' ),
                    array( 'br' => array() )
                ),
            ), // AND
            'orderby' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Sort Terms by', 'searchfiltersort' ),
                'class'         => 'sfswp-field-orderby',
                'options'       => $this->getOrderByOptions(),
                'default'       => 'default',
                'instructions'  => esc_html__('The order in which terms appear in the widget', 'searchfiltersort'),
                'tooltip'       => wp_kses(
                    __( 'The "Default" option means that terms will be sorted in the order as they were received from the Database<br />The "Menu order" option is available for WooCommerce attributes only.', 'searchfiltersort' ),
                    array( 'br' => array() )
                )
            ),
            'in_path' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'In URL', 'searchfiltersort' ),
                'class'         => 'sfswp-field-path',
                'default'       => 'yes',
                'instructions'  => ''
            ),
            'exclude' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Include/Exclude Terms', 'searchfiltersort' ),
                'class'         => 'sfswp-field-exclude',
                'options'       => $entityTerms, //$entity->getTermsForSelect(),
                'multiple'      => 'multiple',
                'instructions'  => '',
                'skip_view'     => true
            ),
            'include' => array(
                'type'          => 'Select',
                'label'         => '',
                'class'         => 'sfswp-field-include',
                'options'       => [ 'no' => esc_html__( 'exclude', 'searchfiltersort' ), 'yes' => esc_html__('include', 'searchfiltersort' ) ],
                'default'       => 'no',
                'instructions'  => esc_html__( 'Include selected terms only', 'searchfiltersort' ),
                'skip_view'     => true
            ),
            'collapse' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Folding', 'searchfiltersort' ),
                'class'         => 'sfswp-field-collapse',
                'default'       => 'no',
                'instructions'  =>  esc_html__( 'Makes filter collapsible in the widget', 'searchfiltersort'),
                'tooltip'       => esc_html__( 'Useful in situations when the filter is rarely applied but takes up some space in the widget. Collapsed by default.', 'searchfiltersort' )
            ),
            'hierarchy' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show Hierarchy', 'searchfiltersort' ),
                'class'         => 'sfswp-field-hierarchy',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Display the term hierarchy in the filter widget. Child terms will be collapsed by default', 'searchfiltersort' )
            ),
            'range_slider' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Enable Range Slider?', 'searchfiltersort' ),
                'class'         => 'sfswp-field-range-slider',
                'default'       => 'yes',
                'instructions'  => esc_html__( 'If disabled, visitors must type numeric values in text inputs', 'searchfiltersort' )
            ),
            'step'      => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Slider Step', 'searchfiltersort' ),
                'class'         => 'sfswp-field-value-step',
                'instructions'  => esc_html__( 'Determines how the numeric value will be changed when you move slider controls', 'searchfiltersort' ),
                'tooltip'       => wp_kses(
                    __('Step 1 means possible values are 1,2,3,4 ...<br />Step 0.1 means possible values are 5.1, 5.2, 5.3, 5.4 ...<br />Step 15 means possible values are 15, 30, 45, 60...', 'searchfiltersort'),
                    array( 'br' => array() )
                ),
                'default'       => 1
            ),
            'search' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Terms Search', 'searchfiltersort' ),
                'class'         => 'sfswp-field-search',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Adds a search field that allows you to quickly find filter terms', 'searchfiltersort' )
            ),
            'parent_filter' => array(
                'type'          => 'Select',
                'label'         => esc_html__( 'Parent Filter', 'searchfiltersort' ),
                'class'         => 'sfswp-field-parent-filter',
                'options'       => [ 'no' => esc_html__( 'Please, add filters first', 'searchfiltersort' ) ],
                'instructions'  => esc_html__( 'If specified, current Filter terms become available only after the parent Filter selected', 'searchfiltersort' )
            ),
            'hide_until_parent' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Hide until the Parent selected', 'searchfiltersort' ),
                'class'         => 'sfswp-field-hide-until-parent',
                'default'       => 'no',
                'instructions'  => esc_html__( 'Do not show this Filter until a parent is selected. Useful for step-by-step filtering', 'searchfiltersort' )
            ),
            'min_num_label' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Labels for Chips', 'searchfiltersort' ),
                'class'         => 'sfswp-field-min-num-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. From {value}', 'searchfiltersort' ),
                'instructions'  => '',
                'skip_view'     => true,
            ),
            'max_num_label' => array(
                'type'          => 'Text',
                'label'         => '',
                'class'         => 'sfswp-field-max-num-label',
                'default'       => '',
                'placeholder'   => esc_html__( 'e.g. To {value}', 'searchfiltersort' ),
                'instructions'  => '',
                'skip_view'     => true,
            ),
            'tooltip' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'Tooltip', 'searchfiltersort' ),
                'class'         => 'sfswp-field-tooltip',
                'instructions'  => esc_html__( 'Will appear next to the Filter Title', 'searchfiltersort' ),
                'default'       => ''
            ),
            'more_less' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'More/Less', 'searchfiltersort' ),
                'class'         => 'sfswp-field-more-less',
                'default'       => 'no',
                'instructions'  => sprintf( esc_html__( 'Show a toggle link after %s terms', 'searchfiltersort' ), sfs_more_less_count() )
            ),
            'show_chips' => array(
                'type'          => 'Checkbox',
                'label'         => esc_html__( 'Show in Chips', 'searchfiltersort' ),
                'class'         => 'sfswp-field-show-chips',
                'default'       => 'yes',
                'instructions'  => esc_html__( 'Show filter selected terms in the list of all chosen items', 'searchfiltersort' )
            ),
            'acf_fields' => array(
                'type'          => 'Text',
                'label'         => esc_html__( 'ACF Fields', 'searchfiltersort' ),
                'class'         => 'sfswp-field-acf',
                'default'       => '',
                'instructions'  => esc_html__( 'Contains the IDs of ACF fields that are related to the Custom Field', 'searchfiltersort' )
            )
        );

        $this->sortFields = apply_filters( 'sfswp_filter_sort_fields', $sortFields, $this );
    }

    public static function getViewOptions()
    {
        $viewOptions = array(
            'checkboxes'    => esc_html__('Checkboxes', 'searchfiltersort'),
            'radio'         => esc_html__('Radio buttons', 'searchfiltersort'),
            'labels'        => esc_html__('Labels list', 'searchfiltersort'),
            'dropdown'      => esc_html__('Dropdown', 'searchfiltersort'),
            'range'         => esc_html__('Numeric range', 'searchfiltersort'),
            'date'          => esc_html__('Date range', 'searchfiltersort'),
        );

        if(sfs_is_woocommerce()){
            $viewOptions['rating'] = esc_html__('Rating', 'searchfiltersort');
        }

        return $viewOptions;
    }

    public static function getDateFormatOptions( $type = 'date' )
    {
        switch( $type ){
            case 'date':
                $F_j_Y = date_i18n( __( 'F j, Y' ) );
                $d_m_Y = date_i18n( 'd/m/Y' );
                $m_d_Y = date_i18n( 'm/d/Y' );

                $formatOptions = array(
                    __( 'F j, Y' ) => '<span>' . $F_j_Y . '</span><code>'.__( 'F j, Y' ).'</code>',
                    'd/m/Y'  => '<span>' . $d_m_Y . '</span><code>d/m/Y</code>',
                    'm/d/Y'  => '<span>' . $m_d_Y . '</span><code>m/d/Y</code>',
                );

                break;
            case 'datetime':
                $F_j_Y = date_i18n( __('F j, Y g:i a') );
                $d_m_Y = date_i18n( 'd/m/Y g:i a' );
                $m_d_Y = date_i18n( 'm/d/Y g:i a' );

                $formatOptions = array(
                    __('F j, Y g:i a') => '<span>' . $F_j_Y . '</span><code>'.__('F j, Y g:i a').'</code>',
                    'd/m/Y g:i a'  => '<span>' . $d_m_Y . '</span><code>d/m/Y g:i a</code>',
                    'm/d/Y g:i a'  => '<span>' . $m_d_Y . '</span><code>m/d/Y g:i a</code>',
                );
                break;
            case 'time':
                $g_i_a = date_i18n( __('g:i a') );
                $H_i_s = date_i18n( 'H:i:s' );

                $formatOptions = array(
                    __('g:i a') => '<span>' . $g_i_a . '</span><code>'.__('g:i a').'</code>',
                    'H:i:s' => '<span>' . $H_i_s . '</span><code>H:i:s</code>',
                );
                break;
        }

        $formatOptions['other'] = '<span>' . esc_html__( 'Custom:', 'searchfiltersort' ) . '</span>';

        return $formatOptions;
    }

    public function getOrderByOptions()
    {
        $orderBy = array(
            'default'       => esc_html__( 'Default (no sorting)', 'searchfiltersort' ),
            'nameasc'       => esc_html__( 'Term name &laquo;abc&raquo;', 'searchfiltersort' ),
            'postcountasc'  => esc_html__( 'Posts count &laquo;123&raquo;', 'searchfiltersort' ),
            'idasc'         => esc_html__( 'Term ID &laquo;123&raquo;', 'searchfiltersort' ),
            'menuasc'       => esc_html__( 'Menu order &laquo;123&raquo;', 'searchfiltersort' ),
            'namedesc'      => esc_html__( 'Term name &laquo;cba&raquo;', 'searchfiltersort' ),
            'postcountdesc' => esc_html__( 'Posts count &laquo;321&raquo;', 'searchfiltersort' ),
            'iddesc'        => esc_html__( 'Term ID &laquo;321&raquo;', 'searchfiltersort' ),
            'menudesc'      => esc_html__( 'Menu order &laquo;321&raquo;', 'searchfiltersort' )
        );

        return $orderBy;
    }

    public function getFieldsByType($type, $configuredFields = [] )
    {
        $selected = [];
        $type = ucfirst($type);

        foreach ( $configuredFields as $key => $field) {
            if ( isset( $field['type'] ) && ( $field['type'] === $type ) ) {
                $selected[$key] = $field;
            }
        }

        return $selected;

    }

    private function getFilterNames( $set_id )
    {
        $filternames = [];

        if( ! $set_id ){
            return $filternames;
        }

        $filters = $this->em->selectOnlySetFilters( $set_id );

        if( ! empty( $filters ) ) {
            if ( count( $filters ) > 1 ) {
                $filternames['-1'] = esc_html__( '— Select Filter —', 'searchfiltersort' ) ;
            } else {
                $filternames['no'] = esc_html__( 'Please, add filters first', 'searchfiltersort' );
            }

            foreach ( $filters as $filter ) {

                if ( in_array( $filter['entity'], [ 'post_meta_num', 'tax_numeric', 'post_date' ] ) ) {
                    continue;
                }

                if ( isset( $filter['ID'] ) ) {
                    $filternames[ $filter['ID'] ] = $filter['label'] .' (' . $filter['e_name'] .')';
                }
            }
        }

        return $filternames;
    }

    public function getEmptyFilterObject( $set_id )
    {
        $defaults = new \stdClass();
        $defaults->ID = self::SFS_NEW_FILTER_ID;
        $defaults->post_parent = '';
        $defaults->menu_order = 0;
        $defaults->post_title = '';
        $defaults->post_content = '';
        $defaults->post_name = '';

        $filter = $this->em->prepareFilter( $defaults );

        return $this->prepareFilterInputsToDisplay( $filter, $this->getFilterNames( $set_id ) );
    }

    public function getFiltersInputs( $set_id )
    {
        $preparedFilters = [];

        if( ! $set_id || empty( $set_id ) ){
            return $preparedFilters;
        }

        $filters = $this->em->selectOnlySetFilters( $set_id );
        foreach ( $filters as $field_key => $filter ){
            $preparedFilters[] = $this->prepareFilterInputsToDisplay( $filter, $this->getFilterNames( $set_id ) );
        }

        return $preparedFilters;
    }

    private function prepareFilterInputsToDisplay( $filter, $filternames )
    {
        $postTypes  = false;
        $terms      = [];
        // Used for filter table class in filter set fields.
        // E.g. post_meta, taxonomy, author.

        $short_entity   = $filter['entity'] ? $filter['entity'] : 'taxonomy';

        if ( $short_entity === 'taxonomy' ) {
            // Add hierarchical class
            if ( is_taxonomy_hierarchical( $filter['e_name'] ) ) {
                $short_entity .= ' taxonomy-hierarchical';
            }

            // Add product attribute class
            if ( strpos( $filter['e_name'], 'pa_' ) === 0 ) {
                $short_entity .= ' taxonomy-product-attribute';
            }

            if ( strpos( $filter['e_name'], 'product_visibility') !== false
                && $filter['view'] === 'rating') {
                $short_entity .= ' selected-and-above-show ';
            }

            if ( sfs_get_experimental_option('use_color_swatches') === 'on' ) {
                $taxonomies = sfs_get_experimental_option( 'color_swatches_taxonomies', [] );

                if ( in_array( $filter['e_name'], $taxonomies ) ) {
                    $short_entity .= ' taxonomy-has-swatches';
                }
            }
        }

        if ( in_array( $filter['e_name'], sfs_brand_filter_entities() ) ) {
            $short_entity .= ' sfswp-filter-has-brands';
        }


        $short_entity .= isset( $filter['view'] ) ? ' sfswp-view-'.$filter['view'] : '';

        $belongs = $this->filterBelongsToPostType( $filter['parent'], $filter['entity'], $filter['e_name'] );

        $postType = '';
        // For post_meta_num entity postType is critical value so we have to set it up
        if ( in_array( $filter['entity'], array( 'post_meta_num', 'post_meta_exists' ) ) ) {
            $fss  = Container::instance()->getSFSSetService();
            $set =  $fss->getSet( $filter['parent'] );
            if( isset( $set['post_type']['value'] ) ){
                $postType = $set['post_type']['value'];
            }
        }

        $entity = $this->em->getEntityByFilter( $filter, $postType );

        if( $entity ){
            $terms = $entity->getTermsForSelect();
            asort( $terms, SORT_NATURAL );
        }

        // This is required only for filter fields inputs
        $filter = $this->fse->combineEntityNameInFilter( $filter );

        foreach( $this->getFieldsMapping() as $fieldKey => $fieldData ){

            if ( isset( $filter[$fieldKey] ) ) {

                if ( $fieldKey === 'parent_filter' ) {
                    // Exclude current filter
                    if ( isset( $filternames[ $filter['ID'] ] ) ) {
                        unset( $filternames[ $filter['ID'] ] );
                    }

                    if ( ! empty( $filternames ) ) {
                        $fieldData['options'] = $filternames;
                    }
                }

                $default_value = isset( $fieldData['default'] ) ? $fieldData['default'] : '';

                $multiple = ( $fieldKey === 'exclude' && ! in_array( $filter['entity'], [ 'post_meta_num', 'tax_numeric', 'post_date' ] ) );

                $fieldData['name']     = $this->generateInputName( $filter['ID'], $fieldKey, $multiple );
                $fieldData['id']       = $this->generateInputID( $filter['ID'], $fieldKey );

                $fieldData['value']    = ( $filter[$fieldKey] ) ? $filter[$fieldKey] : $default_value;

                if( $fieldKey === 'hide_until_parent' ){
                    if( $filter['parent_filter'] > 0 ){
                        $fieldData['additional_class'] = 'sfswp-opened';
                    }
                }

                if ( $fieldKey === 'slug' && $fieldData['value'] ) {
                    $fieldData['readonly'] = 'readonly';
                }

                if ( $fieldKey === 'e_name' && $fieldData['value'] ) {
                    unset( $fieldData['options'] );
                    $fieldData['readonly']  = 'readonly';
                    $fieldData['type']      = 'Text';

                    if ( $filter['entity'] === 'tax_numeric' ) {
                        $fieldData['label']        = esc_html__(  'Taxonomy', 'searchfiltersort' );
                        $fieldData['instructions'] = esc_html__(  'Taxonomy with numeric values you need to filter by', 'searchfiltersort' );
                    }
                }

                if( $fieldKey === 'entity' ) {
                    $fieldData['short_entity'] = $short_entity;

                    if ( $filter['ID'] === self::SFS_NEW_FILTER_ID ) {
                        $fieldData['entity_belongs'] = true;
                    } else {
                        $fieldData['entity_belongs'] = $belongs;
                    }

                    // Add instead-entity field
                    if( $filter['ID'] !== self::SFS_NEW_FILTER_ID ){

                        /**
                         * @feature maybe move this to separate method
                         * @bug slug wasn't saved for new filter
                         */
                        // For existing filters we need to forbid changing entity value
                        // And we need to show input instead select
                        $fieldData['type'] = 'Text';
                        // To make it compatible with Text field
                        unset( $fieldData['options'] );
                        // Forbid to edit field
                        $fieldData['readonly'] = 'readonly';

                        $flatEntities = $this->em->getFlatEntities();

                        $insteadEntityVal = isset( $flatEntities[ $fieldData['value'] ] ) ? $flatEntities[ $fieldData['value'] ] : '';

                        if( $fieldData['value'] === 'post_meta_exists' && ! defined('SFS_FILTERS_PRO') ){
                            $insteadEntityVal = esc_html__('Available in PRO', 'searchfiltersort');
                        }

                        if( $fieldData['value'] === 'tax_numeric' && ! defined('SFS_FILTERS_PRO') ){
                            $insteadEntityVal = esc_html__('Available in PRO', 'searchfiltersort');
                        }

                        // Field, that will be visible instead of entity, that will be hidden.
                        $prepared[ 'instead-entity' ] = $this->getInsteadEntityField( $filter['ID'], $insteadEntityVal );

                    }

                    if ( $filter['ID'] === self::SFS_NEW_FILTER_ID ) {
                        if ( ! defined('SFS_FILTERS_PRO') ) {
                            $fieldData['disabled'] = array( 'post_meta_exists', 'tax_numeric' );
                        }
                    }

                }

                if( $fieldKey === 'exclude' ){
                    // We always add terms even they are empty array to fill Select2 with related terms.
                    $fieldData['options'] = $terms;

                    if( in_array( $filter['entity'], [ 'post_meta_num', 'tax_numeric', 'post_date' ] ) ) {
                        $fieldData['options'] = [];
                    }
                }

                // Set disabled fields for some situations
                if( in_array( $filter['entity'], array( 'author_author', 'post_meta_exists', 'taxonomy_product_visibility' ) ) /* $filter['entity'] === 'author_author'*/ && $fieldKey === 'logic' ){
                    $fieldData['disabled'] = array('and');
                }

                if( in_array( $filter['entity'], [ 'post_meta_num', 'tax_numeric', 'post_date' ] ) && $fieldKey === 'logic' ){
                    $fieldData['disabled'] = array('or');
                }

                if ( $fieldKey === 'view' ) {
                    if ( in_array( $filter['entity'], [ 'post_meta_num', 'tax_numeric' ] ) ) {
                        $fieldData['disabled'] = array(
                            'checkboxes',
                            'dropdown',
                            'radio',
                            'labels',
                            'date',
                            'colors',
                            'image',
                            'rating',
                        );
                    } else if ( in_array( $filter['entity'], [ 'post_date' ] ) ) {
                        $fieldData['disabled'] = array(
                            'checkboxes',
                            'dropdown',
                            'radio',
                            'labels',
                            'range',
                            'colors',
                            'image',
                            'rating',
                        );
                    } else if ( in_array( $filter['entity'], [ 'taxonomy_product_visibility' ] ) ) {
                        $fieldData['disabled'] = array(
                            'date',
                            'range',
                        );
                    }else {
                        $fieldData['disabled'] = array(
                            'range',
                            'date',
                            'rating',
                        );
                    }
                }

                if( $fieldKey === 'orderby' ) {
                    if ( $filter['entity'] !== 'taxonomy_product_cat' && ( mb_strpos( $filter['entity'], 'taxonomy_pa' ) === false ) ){
                        $fieldData['disabled'] = ['menuasc', 'menudesc'];
                    }
                }

                if ( ( $filter['ID'] === self::SFS_NEW_FILTER_ID || $filter['entity'] === 'post_date' ) && $fieldKey === 'date_format' ) {
                    // This can be new filter or existing
                    $date_type         = $filter['date_type'] ? $filter['date_type'] : 'date';
                    $dateFormatOptions = $this->getDateFormatOptions( $date_type );
                    $disabled_custom = true;
                    $value_to_custom = $fieldData['value'];

                    if ( ! in_array( $value_to_custom, self::getPossibleDateFormats( $date_type ) ) ) {
                        $disabled_custom    = false;
                        $fieldData['value'] = 'other';
                    }

                    $customFormatField = '</label><label>' . $this->addCustomDateFormatField( $fieldData['name'], $value_to_custom, $disabled_custom );
                    $dateFormatOptions['other'] = str_replace( '</span>', '</span>' . $customFormatField, $dateFormatOptions['other'] );

                    $fieldData['options'] = $dateFormatOptions;
                }
            }

            $prepared[ $fieldKey ] = $fieldData;
        }

        return $prepared;
    }

    public function sanitizeSFSFields( $filter )
    {
        if( is_array( $filter ) ){
            $sanitizedFilter = [];
            $not_esc_html    = [ 'e_name', 'tooltip' ];

            foreach( $filter as $key => $value ){

                if( in_array( $key, $not_esc_html, true ) ){
                    // Why? because meta_key field can contain any different characters
                    $sanitizedFilter[ $key ] = $value;
                }else{

                    if( is_array( $value ) ){
                        array_map( 'esc_html', $value );
                        $sanitizedFilter[ $key ] = $value;
                    } else {
                        $sanitizedFilter[ $key ] = esc_html( $value );
                    }

                }
            }

            if( isset( $sanitizedFilter['menu_order'] ) ){
                $sanitizedFilter['menu_order'] = sfs_sanitize_int( $sanitizedFilter['menu_order'] );
            }

            if( isset( $sanitizedFilter['label'] ) ){
                $sanitizedFilter['label'] = sanitize_text_field( $sanitizedFilter['label'] );
            }

            if( isset( $sanitizedFilter['slug'] ) ){
                $sanitizedFilter['slug'] = preg_replace( '/[^a-z0-9\-\_]+/', '', mb_strtolower($sanitizedFilter['slug']) );
                $sanitizedFilter['slug'] = trim($sanitizedFilter['slug'], '-');
            }

            if( isset( $sanitizedFilter['step'] ) ){
                $sanitizedFilter['step'] = preg_replace('/[^\d\.]+/', '', $sanitizedFilter['step'] );
                if( ! $sanitizedFilter['step'] ){
                    $sanitizedFilter['step'] = 1;
                }
            }

            if( isset( $sanitizedFilter['tooltip'] ) ){
                $sanitizedFilter['tooltip'] = wp_kses(
                    $sanitizedFilter['tooltip'],
                    array(
                        'br'        => array(),
                        'span'      => array('class' => true, 'id' => true),
                        'em'        => array(),
                        'strong'    => array('class' => true),
                        'i'         => array(),
                        'b'         => array()
                    )
                );
            }

            return $sanitizedFilter;
        }

        return $filter;
    }

    public function validateTheFilter( $filter, $id = false ) {

        $valid          = true;
        $validEntity    = true;
        $validator      = new Validator();

        if( $filter['ID'] === self::SFS_NEW_FILTER_ID ) {
            $filterID  = $id;
        }else{
            $filterID  = $filter['ID'];
        }

        // Check all fields are our fields
        $defaultFields = $this->getFieldsMapping();
        // To make compatible with this field
        $defaultFields['instead-entity'] = true;

        foreach( $filter as $fieldKey => $fieldValue ){
            if( ! isset( $defaultFields[ $fieldKey ] ) ){
                $this->pushError(32); // Invalid fields present
                $valid = false;
            }
        }

        /**
         * Check required field data, that should be not empty
         */
        if( isset( $filter['ID'] ) ){
            // We need to check post type for current ID otherwise we can override any existing post
            if( $filter['ID'] !== self::SFS_NEW_FILTER_ID ){
                $savedFilter = get_post( $filter['ID'] );

                // Filter post doesn't exist
                if( ! $savedFilter ){
                    $this->pushError(33); // Invalid filter ID
                    $valid = false;
                }
                // Other post type
                if( ! isset( $savedFilter->post_type ) || $savedFilter->post_type !== SFS_FILTERS_POST_TYPE ){
                    $this->pushError(33); // Invalid filter ID
                    $valid = false;
                }
            }

        } else {
            $this->pushError(33); // Invalid filter ID
            $valid = false;
        }

        /**
         * Parent field aka Set ID
         */
        if( isset( $filter['parent'] ) ){

            if( $filter['ID'] !== self::SFS_NEW_FILTER_ID ) {
                $savedSet = get_post($filter['parent']);

                // Set post doesn't exist
                if (!$savedSet) {
                    $this->pushError(34); // Invalid Set ID
                    $valid = false;
                }
                // Other post type
                if ( ! isset( $savedSet->post_type ) || $savedSet->post_type !== SFS_FILTERS_SET_POST_TYPE) {
                    $this->pushError(34); // Invalid Set ID
                    $valid = false;
                }
            }

        }else{
            $this->pushError( 34); // Invalid Set ID
            $valid = false;
        }

        /**
         * Entity
         */
        if( isset( $filter['entity'] ) ){
            if( ! $validator->validatePossibleEntity( $filter['entity'] ) ){
                $this->pushError( 35, $filterID, 'entity' ); // Invalid Entity
                $this->pushError( 35, $filterID, 'instead-entity' );
                $validEntity = false;
                $valid = false;
            }

        }else{
            $this->pushError( 35, $filterID, 'entity' ); // Invalid Entity
            $this->pushError( 35, $filterID, 'instead-entity' );
            $validEntity = false;
            $valid = false;
        }

        /**
         * Slug validations
         */
        if( isset( $filter['slug'] ) ){
            $existingSlugs = get_option('sfswp_filter_permalinks', []);
            $prefix = $filter['slug'];

            // Check this only for new filters
            if( $filter['ID'] === self::SFS_NEW_FILTER_ID && $validEntity ) {
                $fs = Container::instance()->getFilterService();
                $newEntityKey = $fs->getEntityKey($filter['entity'], $filter['e_name']);
                // Prohibit using the same slug for another entity
                if (!$validator->validateExistingPrefix($prefix, $existingSlugs, $newEntityKey)) {
                    $this->pushError( 36, $filterID, 'slug' ); // Prefix is already used
                    $valid = false;
                }
            }

            // Ensure, that prefix contains at least one alphabetic character
            // Also this prevents from empty prefix
            if ( ! $validator->validateAlphabCharsExists( $prefix ) ) {
                $this->pushError( 37, $filterID, 'slug' ); // Invalid prefix. Has to contain at least one alphabetic symbol
                $valid = false;
            }

            // Check hyphens problem when cat and cat-x exists
            if ( ! $validator->validatePrefixHyphens( $prefix, $existingSlugs ) ) {
                $this->pushError( 39, $filterID, 'slug' ); // Invalid prefix. Incorrect hyphens.
                $valid = false;
            }

            // Check for slugs, that matches with native WP Entities
            if ( ! $validator->validateAllowedPrefixes( $prefix, $filter ) ) {
                $errorCode = 3991;
                if ( defined( 'SFS_FILTERS_PRO' ) && SFS_FILTERS_PRO ) {
                    $errorCode = 399;
                }
                $this->pushError( $errorCode, $filterID, 'slug' ); // Invalid prefix. Equal to wp entity.
                $valid = false;
            }

        } else {
            $this->pushError( 38, $filterID, 'slug' ); // Invalid prefix. Empty.
            $valid = false;
        }

        /**
         * E_name validations
         */
        if ( isset( $filter['e_name'] ) ) {

            if( isset( $filter['entity'] ) ) {
                if ( in_array( $filter['entity'], array( 'post_meta', 'post_meta_num', 'post_meta_exists', 'tax_numeric' ) ) ) {
                    $e_name = $filter['e_name'];

                    if ( $e_name === '' ) {
                        $this->pushError( 401, $filterID, 'e_name' ); // Select or Enter meta key.
                        $valid = false;
                    } else {

                        if ( $filter['entity'] === 'tax_numeric' ) {
                            // Validate taxonomy
                            $taxonomies      = $this->em->getPossibleTaxonomies();
                            $taxonomy_e_name = $filter[ 'e_name' ];

                            if( mb_strpos( $filter[ 'e_name' ], 'taxonomy_' ) === false ){
                                $taxonomy_e_name = 'taxonomy_' . $filter[ 'e_name' ];
                            }

                            if ( ! isset( $taxonomies[ $taxonomy_e_name ] ) ) {
                                $this->pushError( 402, $filterID, 'e_name' ); // Invalid E_name.
                                $valid = false;
                            }
                        } else {
                            // Should not be empty and should contain at least one alphabetic character
                            if ( ! $validator->validateAlphabCharsExists( $e_name ) ) {
                                $this->pushError( 40, $filterID, 'e_name' ); // Invalid E_name.
                                $valid = false;
                            }

                            // Should not contain characters that escaped by esc_attr
                            if ( ! $validator->validateEscAttrCharacters( $e_name ) ) {
                                $this->pushError( 40, $filterID, 'e_name' ); // Invalid E_name.
                                $valid = false;
                            }

                            // Some meta keys should be prohibited to use
                            $excludedMetaKeys = sfs_get_forbidden_meta_keys();
                            if ( in_array( $e_name, $excludedMetaKeys, true ) ) {
                                $this->pushError(  40, $filterID, 'e_name' ); // Invalid E_name.
                                $valid = false;
                            }
                        }
                    }
                }

                if ( in_array( $filter['e_name'], array( 'post_date', 'post_meta_date' ) ) ) {

                    if ( isset( $filter['date_format'] ) && isset( $filter['date_type'] ) ) {

                        $date_time = sfs_split_date_time( $filter['date_format'] );

                        switch ( $filter['date_type'] ) {
                            case 'date':
                                // check for date
                                if ( ! $date_time['date'] ) {
                                    $this->pushError(  60, $filterID, 'date_format' ); // Invalid date_format.
                                    $valid = false;
                                }
                                break;
                            case 'datetime':
                                if ( ! $date_time['date'] || ! $date_time['time'] ) {
                                    $this->pushError(  62, $filterID, 'date_format' ); // Invalid date_format.
                                    $valid = false;
                                }
                                break;
                            case 'time':
                                if ( ! $date_time['time'] ) {
                                    $this->pushError(  61, $filterID, 'date_format' ); // Invalid date_format.
                                    $valid = false;
                                }
                                break;
                        }

                    }
                }
            }

        }else{
            $this->pushError( 40, $filterID, 'e_name' ); // Invalid E_name.
            $valid = false;
        }

        /**
         * View validations
         */
        if( isset( $filter['view'] ) ){
            $viewOptions = array_keys( $this->getViewOptions() );

            if( ! $validator->validateView( $filter, $viewOptions ) ){
                $this->pushError( 41, $filterID, 'view' ); // Invalid View.
                $valid = false;
            }

        } else {
            $this->pushError( 41, $filterID, 'view' ); // Invalid View.
            $valid = false;
        }

        /**
         * Date Type validations
         */
        if ( isset( $filter['date_type'] ) ) {
            if( ! in_array( $filter['date_type'], array( 'date', 'datetime', 'time' ), true ) ){
                $this->pushError( 411, $filterID, 'date_type' ); // Invalid Data Type
                $valid = false;
            }
        }

        /**
         * Logic validations
         */
        if( isset( $filter['logic'] ) ){

            if( ! in_array( $filter['logic'], array( 'or', 'and' ), true ) ){
                $this->pushError( 42, $filterID, 'logic' ); // Invalid Logic.
                $valid = false;
            }

            // For author and post_meta_exists entities logic can be only OR
            if( in_array( $filter['entity'], array( 'author_author', 'post_meta_exists', 'taxonomy_product_visibility' ) )  ){
                if( $filter['logic'] !== 'or' ){
                    $this->pushError( 45, $filterID, 'logic' ); // Not acceptable logic.
                    $valid = false;
                }
            }

            // For author entity logic can be only OR
            if ( in_array( $filter['entity'], [ 'post_meta_num', 'tax_numeric', 'post_date' ] ) ) {
                if( $filter['logic'] !== 'and' ){
                    $this->pushError( 47, $filterID, 'logic' ); // Not acceptable logic.
                    $valid = false;
                }
            }

        } else {
            $this->pushError( 42, $filterID, 'logic' ); // Invalid Logic.
            $valid = false;
        }

        /**
         * Orderby validations
         */
        if( isset( $filter['orderby'] ) ){
            $orderBy = array_keys( $this->getOrderByOptions() );
            if( ! in_array( $filter['orderby'], $orderBy, true ) ){
                $this->pushError( 43, $filterID, 'orderby' ); // Invalid Orderby.
                $valid = false;
            }
        } else {
            $this->pushError( 43, $filterID, 'orderby' ); // Invalid Orderby.
            $valid = false;
        }

        /**
         * In path and Collapse doesn't require validations
         * because their values are prepended in code
         */

        /**
         * Exclude validations
         */
        if( isset( $filter['exclude'] ) && $validEntity ){
            if( ! $validator->validateExcludeTerms( $filter['exclude'], $filter ) ){
                $this->pushError( 44, $filterID, 'exclude' ); // Invalid Exclude.
                $valid = false;
            }
        }

        if( isset( $filter['include'] ) ){
            if( ! in_array( $filter['include'], ['yes', 'no'] ) ){
                $this->pushError( 441, $filterID, 'exclude' ); // Invalid Exclude.
                $valid = false;
            }
        }

        // In case when checkbox is not checked there is no $_POST['in_path'] parameter
        if( isset( $filter['in_path'] ) ){
            if( in_array( $filter['entity'], [ 'post_meta_num', 'tax_numeric', 'post_date' ] ) && $filter['in_path'] === 'yes' ){
                $this->pushError( 46, $filterID, 'in_path' ); // Invalid In Path for Post meta num.
                $valid = false;
            }
        }

        // Check data types if they are correct
        /**
         * @todo validate range_slider, show_chips, step fields !!! IMPORTANT
         */

        // Check combinations of different data
        /**
         * The unique slug and entity problem
         */
        // The slug and entity pair should be the same between all filters in all sets!
        // Otherwise we will have different slugs and URLs in different categories for the same Post type.
        // If user adding new slug for the same entity, it should be notified, that this new slug
        // will be changed for all same entities.

        // Obligatorily sanitize all data. Maybe separate method for that.
        // If entity is taxonomy, field e_name should be empty.

        // Filter slugs should not be from blacklist. Blacklist - typical Wordpress entities /categories, tags etc

        // Slugs should be checked for non UTF symbols and maybe converted to latin UTF symbols
        // Because user can add slug = категория and also user can add slug 'cheap' where 'c' will be cyrillic or other.

        return $valid;
    }

    public function validateFilters( $filters ) {

        if ( ! $filters ) {
            return false;
        }
        $valid = true;

        // Check permissions
        if ( ! current_user_can( sfs_plugin_user_caps() ) ) {
            // An error occurred. You do not have permission to edit this.
            $this->pushError(202);
            $valid = false;
        }

        // Check for equal filters
        foreach ( (array) $filters as $filter ) {
            if( isset( $filter['entity'] ) && isset( $filter['e_name'] ) ) {
                // To avoid few filters with the same meta key
                if ( in_array( $filter['entity'], array( 'post_meta', 'post_meta_num', 'post_meta_exists' ), true ) ) {
                    $keys[] = 'post_meta' . $filter['e_name'];
                } elseif ( $filter['entity'] === 'tax_numeric' || $filter['entity'] === 'post_date' ) {
                    $keys[] = $filter['entity'] .'_'. $filter['e_name'];
                } else {
                    $keys[] = $filter['entity'] . $filter['e_name'];
                }
            }
        }

        if ( sfs_array_contains_duplicate( $keys ) ) {
            $this->pushError(31); // Equal filters
            $valid = false;
        }

        return apply_filters( 'sfswp_validate_filter_fields', $valid, $this );
    }

    public function ajaxValidateFilters()
    {
        $postData   = Container::instance()->getThePost();

        $data       = isset( $postData['validateData'] ) ? $postData['validateData'] : false;
        $response   = [];

        if( ! $data ){
            $this->pushError(20);
        }

        if( ! isset( $data['_sfs_nonce'] ) || ! wp_verify_nonce( $data['_sfs_nonce'], SFSSet::NONCE_ACTION ) ){
            $this->pushError(20); // Default common error
        }

        // Validate all filters
        // If no one filter it's ok
        if( isset( $data['sfswp_filter_fields'] ) ){

            $this->validateFilters( $data['sfswp_filter_fields'] );

            // Set up checkbox fields if they are empty
            $filterConfiguredFields = $this->getFieldsMapping();

            // Validate each filter separately
            foreach ( (array) $data['sfswp_filter_fields'] as $filterId => $filter ) {
                $filter = $this->prepareFilterCheckboxFields($filter, $this->getFieldsByType('checkbox', $filterConfiguredFields));
                $this->validateTheFilter( $filter, $filterId );
            }

        }


        $this->fillErrorsMessages();
        $errors = $this->getErrors();
        // Send errors if they exist
        if( $errors && ! empty( $errors ) ){
            $response['errors'] = $errors;
            wp_send_json_error($response);
        }

        /**
         * @feature it is better to validate all fields and collect all errors to show them simultaneously.
         */

        wp_send_json_success();
    }

    function saveFilter( $filter ) {
        // May have been posted. Remove slashes.
        $filter = wp_unslash( $filter );

        $filter = apply_filters( 'sfswp_pre_save_filter', $filter );

        unset( $filter['instead-entity'] );
        // Make a backup of field data and remove some args.
        $_filter = $filter;
        sfs_extract_vars( $_filter, array( 'ID', 'label', 'parent', 'menu_order', 'slug' ) );

        $_filter = $this->fse->splitEntityFullNameInFilter( $_filter );

        // Create array of data to save.
        $to_save = array(
            'ID'			=> $filter['ID'],
            'post_status'	=> 'publish',
            'post_type'		=> SFS_FILTERS_POST_TYPE,
            'post_title'	=> $filter['label'],
            'post_content'	=> maybe_serialize( $_filter ),
            'post_parent'	=> $filter['parent'],
            'menu_order'	=> $filter['menu_order'] ? $filter['menu_order'] : 0,
            'post_name'     => $filter['slug'],
            'post_excerpt'  => $filter['entity']
        );

        // Unhook wp_targeted_link_rel() filter from WP 5.1 corrupting serialized data.
        remove_filter( 'content_save_pre', 'wp_targeted_link_rel' );

        do_action( 'sfswp_pre_save_post_filter', $to_save, $filter );

        add_filter( 'pre_wp_unique_post_slug', 'sfs_force_non_unique_slug', 10, 2 );

        // Slash data.
        // WP expects all data to be slashed and will unslash it (fixes '\' character issues).
        $to_save = wp_slash( $to_save );

        // Update or Insert.
        if( $filter['ID'] === self::SFS_NEW_FILTER_ID ){
            $filter['ID'] = wp_insert_post( $to_save );
        }else{
            wp_update_post( $to_save );
        }

        remove_filter( 'pre_wp_unique_post_slug', 'sfs_force_non_unique_slug', 10 );

        // Return field.
        return $filter;
    }

    public function prepareFilterCheckboxFields( $filter, $configuredFields = [] )
    {
        foreach ( $configuredFields as $key => $checkbox ){
            if( ! isset( $filter[ $key ] ) ){
                $filter[ $key ] = 'no';
            }else{
                $filter[ $key ] = 'yes';
            }
        }

        return $filter;
    }

    public function getFieldsMapping()
    {
        return $this->defaultFields;
    }

    public function deleteFilter( $ID, $force = false )
    {
        if( ! $ID ){
            return false;
        }

        if( $force ){
            return wp_delete_post( $ID, true );
        }else{
            return wp_trash_post( $ID );
        }

    }

    public function deleteRelatedFilters( $postid, $post )
    {
        if( $post->post_type !== SFS_FILTERS_SET_POST_TYPE ){
            return $postid;
        }

        $args = array(
            'post_type'         => SFS_FILTERS_POST_TYPE,
            'posts_per_page'    => -1,
            'post_parent'       => $postid,
            'post_status'		=> array('any'),
        );

        $setFilters = get_posts( $args );

        if( ! empty( $setFilters ) ){
            foreach ( $setFilters as $filter ) {
                $this->deleteFilter( $filter->ID, true );
            }
        }

        return $postid;
    }

    public function ajaxDeleteFilter()
    {
        $postData   = Container::instance()->getThePost();
        $filterId   = isset( $postData['fid'] ) ? $postData['fid'] : false;
        if( $filterId === self::SFS_NEW_FILTER_ID ){
            wp_send_json_success();
        }

        $nonce          = isset( $postData['_wpnonce'] ) ? $postData['_wpnonce'] : false;
        $errorResponse  = array(
            'fid' => $filterId,
            'message' => esc_html__('An error occurred. Please, refresh the page and try again.', 'searchfiltersort')
        );

        if( ! wp_verify_nonce( $nonce, SFSSet::NONCE_ACTION ) ){
            wp_send_json_error( $errorResponse );
        }

        if( ! $filterId ){
            wp_send_json_error( $errorResponse );
        }

        $filter = get_post( $filterId );

        if( ! isset( $filter->post_type ) || ( $filter->post_type !== SFS_FILTERS_POST_TYPE ) ){
            wp_send_json_error( $errorResponse );
        }

        if( $filterPost = $this->deleteFilter( $filterId, true ) ){
            $response['fid'] = $filterPost->ID;
            wp_send_json_success( $response );
        }else{
            wp_send_json_error( $errorResponse );
        }
    }

    public static function getPossibleDateFormats( $date_type )
    {
        $possibleFormats = [
            'date' => array(
                'd/m/Y',
                'm/d/Y',
                __('F j, Y'),
            ),
            'datetime' => array(
                'd/m/Y g:i a',
                'm/d/Y g:i a',
                __('F j, Y g:i a'),
            ),
            'time' => array(
                __('g:i a'),
                'H:i:s',
            )
        ];

        if ( isset( $possibleFormats[$date_type] ) ) {
            return $possibleFormats[$date_type];
        }

        return [];
    }

    private function addCustomDateFormatField( $name, $value, $disabled = false )
    {
        $html = '<input type="text" name="'.$name.'" value="'.$value.'"';
        if ( $disabled ) {
            $html .= ' disabled="disabled"';
        }

        $html .= ' class="sfswp-date-custom-format" />';

        return $html;
    }

    public function sendDateFormats()
    {
        $postData   = Container::instance()->getThePost();
        $filterId   = isset( $postData['fid'] ) ? $postData['fid'] : false;
        $setId      = isset( $postData['setId'] ) ? $postData['setId'] : false;
        $dateType   = isset( $postData['dateType'] ) ? $postData['dateType'] : false;

        $errorResponse  = array(
            'fid' => $filterId,
            'message' => esc_html__('An error occurred. Please, refresh the page and try again.', 'searchfiltersort')
        );

        if ( ! $filterId || ! $dateType ) {
            wp_send_json_error( $errorResponse );
        }

        $response = [];

//        $savedValue = $this->em->getFilterBy( 'ID', $filterId, array( array( 'ID' => $setId ) ) );
        $savedValue = $this->em->getFilterById( $filterId );
        $formatoptions = self::getDateFormatOptions( $dateType );

        $field = [
            'type'          => 'Radio',
            'class'         => 'sfswp-date-format',
            'default'       => __( 'F j, Y' ),
        ];

        $default_formats = [
            'date'     => __( 'F j, Y' ),
            'datetime' => __('F j, Y g:i a'),
            'time'     => __('g:i a'),
        ];

        $disabled_custom = true;
        $value_to_custom = ( $savedValue['date_format'] && $dateType === $savedValue['date_type'] ) ? $savedValue['date_format'] : $default_formats[$dateType];
        if ( $value_to_custom && ! in_array( $value_to_custom, self::getPossibleDateFormats( $dateType ) ) ) {
            $disabled_custom = false;
            $field['value']  = 'other';
        }

        $customFormatField = '</label><label>' . $this->addCustomDateFormatField( $this->generateInputName( $filterId, 'date_format' ), $value_to_custom, $disabled_custom );
        $formatoptions['other'] = str_replace( '</span>', '</span>' . $customFormatField, $formatoptions['other'] );

        $field['name']      = $this->generateInputName( $filterId, 'date_format' );
        $field['id']        = $this->generateInputID( $filterId, 'date_format' );
        $field['options']   = $formatoptions;

        ob_start();

        echo sfs_render_input( $field );

        $response['html'] = ob_get_clean();

        wp_send_json_success( $response );
        die();
    }

    public function sendExcludedTerms()
    {
        $container  = Container::instance();
        $postData   = $container->getThePost();
        $validator  = new Validator();
        $filterId   = isset( $postData['fid'] ) ? $postData['fid'] : false;
        $nonce      = isset( $postData['_wpnonce'] ) ? $postData['_wpnonce'] : false;
        $entity     = isset( $postData['entity'] ) ? $postData['entity'] : false;
        $e_name     = isset( $postData['ename'] ) ? $postData['ename'] : false;

        $errorResponse  = array(
            'fid' => $filterId,
            'message' => esc_html__('An error occurred. Please, refresh the page and try again.', 'searchfiltersort')
        );

        if( ! wp_verify_nonce( $nonce, SFSSet::NONCE_ACTION ) ){
            wp_send_json_error( $errorResponse );
        }

        if( ! $validator->validatePossibleEntity( $entity ) ){
            wp_send_json_error( $errorResponse );
        }

        $em = $container->getEntityManager();
        if( $e_name ){
            $filterEntity['entity'] = $entity;
            $filterEntity['e_name'] = $e_name;
        }else{
            $fse = $container->getFilterService();
            $filterEntity = $fse->splitEntityFullNameInFilter( array( 'entity' => $entity ) );
        }

        $entity = $em->getEntityByFilter( $filterEntity );
        $terms  = $entity->getTermsForSelect2();

        if( ! empty( $terms ) ){
            $response['terms']  = $terms;
            $response['fid']    = $filterId;
            wp_send_json_success( $response );
        }else{
            wp_send_json_error( $errorResponse );
        }

    }

    public function addSpinnerToDateFormats( $html, $attributes )
    {
        if ( isset( $attributes['class'] ) && $attributes['class'] === 'sfswp-date-format' ) {
            $spinner        = '<span class="spinner"></span>'."\r\n";
            $openContainer  = '<div class="sfswp-after-spinner-container">'."\r\n";

            $closeContainer = '</div>'."\r\n";

            if ( isset( $attributes['options']['d/m/Y'] ) ) {
                $date_type = 'sfswp-type-date';
            }

            if ( isset( $attributes['options']['d/m/Y g:i a'] ) ) {
                $date_type = 'sfswp-type-datetime';
            }

            if ( isset( $attributes['options']['H:i:s'] ) ) {
                $date_type = 'sfswp-type-time';
            }

            $html = str_replace( 'sfswp-radio-list', 'sfswp-radio-list ' . $date_type, $html );

            $html = $spinner . $openContainer . $html . $closeContainer;
        }
        return $html;
    }

    public function addSpinnerToSelect( $html, $attributes )
    {
        if( isset( $attributes['class'] ) ){
            $requiredIds = array(
                $this->generateInputID( self::SFS_NEW_FILTER_ID, 'exclude' ),
            );

            if( in_array( $attributes['id'], $requiredIds ) ){

                $spinner        = '<span class="spinner"></span>'."\r\n";
                $openContainer  = '<div class="sfswp-after-spinner-container">'."\r\n";

                $closeContainer = '</div>'."\r\n";

                $html = $spinner . $openContainer . $html . $closeContainer;

            }
        }
        return $html;
    }

    private function getInsteadEntityField( $filterId, $insteadEntityVal )
    {
        $insteadEntity = array(
            'type'          => 'Text',
            'label'         => esc_html__( 'Filter by', 'searchfiltersort' ),
            'class'         => 'sfswp-field-instead-entity',
            'name'          => $this->generateInputName( $filterId, 'instead-entity' ),
            'id'            => $this->generateInputID( $filterId, 'instead-entity' ),
            'value'         => $insteadEntityVal,
            'readonly'      => 'readonly',
            'default'       => '',
            'instructions'  => esc_html__( 'A thing by which posts will be filtered', 'searchfiltersort'),
            'tooltip'       => esc_html__( 'An already selected value cannot be changed. But you can always delete the current one and create a new filter if you need.', 'searchfiltersort' ),
            'required'      => true
        );

        return $insteadEntity;
    }


    /**
     * @return true|false
     */
    public function filterBelongsToPostType( $parent, $entity, $e_name )
    {
        if( ! isset( $parent ) ){
            return false;
        }

        $fss  = Container::instance()->getSFSSetService();
        $set =  $fss->getSet( $parent );

        if( ! isset( $set['post_type']['value'] ) ){
            return false;
        }

        $post_type = $set['post_type']['value'];

        return $this->entityBelongsToPostType( $post_type, $entity, $e_name );
    }

    /**
     * @return true|false
     */
    private function entityBelongsToPostType( $post_type, $entity, $e_name )
    {
        if ( empty( $post_type ) ){
            return false;
        }

        if( in_array( $entity, array(
            'author',
            'post_date',
            'post_meta',
            'post_meta_num',
            'post_date',
        ) ) ){
            return true;
        }

        if( in_array( $entity, array( 'post_meta_exists', 'tax_numeric' ) ) && defined( 'SFS_FILTERS_PRO' ) ){
            return true;
        }

        if( $entity === 'taxonomy' || $entity === 'tax_numeric' ){
            if ( $entity === 'tax_numeric' ) {
                if ( ! defined( 'SFS_FILTERS_PRO' ) ) {
                    return false;
                }
                // We have to cut e_name to make correct verification
                if ( mb_strpos( $e_name, 'taxonomy_' ) !== false ) {
                    $e_name = mb_strcut( $e_name, strlen( 'taxonomy_' ) );
                }
            }

            return $this->isTaxonomyBelongsToPostType( $post_type, $e_name );
        }

        return false;
    }

    public function getErrorMessage( $code = 20 )
    {
        $messages = $this->getErrorsList();

        if( isset( $messages[$code] ) ){
            return $messages[$code];
        }

        return $messages[20];
    }

    public function pushError( $errorCode, $filterId = false, $filterKey = '' )
    {
        $to_add = true;
        $error  = array(
            'code' => $errorCode
        );

        if( $filterId && $filterKey ){
            $error['id'] = $this->generateInputID( $filterId, $filterKey );
        }

        if ( ! empty( $this->errors ) ) {
            foreach ( $this->errors as $single_error ) {
                if ( isset( $single_error['code'] ) && $single_error['code'] == $errorCode ) {
                    $to_add = false;
                    break;
                }
            }
        }

        if ( $to_add ) {
            $this->errors[] = $error;
            return $error;
        }

        return false;
    }

    public function getErrors()
    {
        if( ! empty( $this->errors ) ) {
            return $this->errors;
        }

        return false;
    }

    public function fillErrorsMessages()
    {
        if( ! empty( $this->errors ) ){
            $errorsWithMessages = [];
            $messagesList = $this->getErrorsList();

            foreach ( $this->errors as $index => $error ){
                if( isset( $error['code'] ) && isset( $messagesList[ $error['code'] ] ) ){
                    $errorsWithMessages[$index] = $error;
                    $errorsWithMessages[$index]['message'] =  $messagesList[ $error['code'] ];
                }
            }

            $this->errors = $errorsWithMessages;
            return true;
        }

        return false;
    }

    public function getErrorCodes()
    {
        $codes = [];

        if( ! empty( $this->errors ) ){
            foreach( $this->errors as $error ){
                $codes[] = $error['code'];
            }
        }

        return $codes;
    }


    /**
     * @return true|false
     */
    private function isTaxonomyBelongsToPostType( $post_type, $taxonomy = null )
    {
        if ( is_object( $post_type ) )
            $post_type = $post_type->post_type;

        if ( empty( $post_type ) ){
            return false;
        }

        $taxonomies = get_object_taxonomies( $post_type );

        return in_array( $taxonomy, $taxonomies );
    }

    public function generateInputID( $ID, $key )
    {
        return self::FILTER_FIELD_KEY . '-' . $ID . '-' . $key;
    }

    public function generateInputName( $ID, $key, $multiple = false )
    {
        $name = self::FILTER_FIELD_KEY . '['. $ID .']['. $key . ']';
        if( $multiple ){
            $name .= '[]';
        }
        return $name;
    }

    public static function getErrorsList()
    {
        $errors = array(
            // Set errors
            20 => esc_html__('An error occurred. SFS Set fields were not saved, please try again.', 'searchfiltersort'),
            201 => esc_html__('An error occurred. SEO Rule fields were not saved, please try again.', 'searchfiltersort'),
            202 => esc_html__('An error occurred. You do not have permission to edit this.', 'searchfiltersort'),
            211 => esc_html__( 'Error: invalid WP Page type.', 'searchfiltersort' ),
            21 => esc_html__( 'Error: invalid post type.', 'searchfiltersort' ),
            22 => esc_html__( 'Error: invalid value of the Where to filter? field.', 'searchfiltersort' ),
            221 => esc_html__( 'Error: invalid query.', 'searchfiltersort' ),
            23 => esc_html__( 'Error: invalid Hide empty field.', 'searchfiltersort' ),
            24 => esc_html__( 'Error: invalid Show count field.', 'searchfiltersort' ),
            242 => esc_html__( 'Error: invalid «Apply Button» mode field.', 'searchfiltersort' ),
//            241 => esc_html__( 'Error: invalid Order field.', 'searchfiltersort' ),
            // Filters errors
            31 => esc_html__( 'Error: two or more filters with equal Filter By and Meta key values are forbidden in the same Set. Please, remove or change equal filters.', 'searchfiltersort' ),
            32 => esc_html__( 'Error: invalid fields present.', 'searchfiltersort' ),
            33 => esc_html__( 'Error: invalid filter ID.', 'searchfiltersort' ),
            34 => esc_html__( 'Error: invalid set ID.', 'searchfiltersort' ),
            35 => esc_html__( 'Error: invalid filter entity.', 'searchfiltersort' ),
            36 => esc_html__( 'Error: filter prefix is already used for another entity.', 'searchfiltersort' ),
            37 => esc_html__( 'Error: filter prefix should have at least one alphabetic symbol.', 'searchfiltersort' ),
            38 => esc_html__( 'Error: filter prefix should not be empty.', 'searchfiltersort' ),
            39 => esc_html__( 'Error: prefix part before "-" character can not be equal to another existing prefix.', 'searchfiltersort' ),
            399 => esc_html__( 'Error: this prefix is not allowed because it matches a taxonomy or term name already used on your site. Please use a different prefix.', 'searchfiltersort' ),
            3991 => esc_html__( 'Error: var name is not allowed because it matches a taxonomy or term name already used on your site.', 'searchfiltersort' ),
            40 => esc_html__( 'Error: invalid Meta key value', 'searchfiltersort' ),
            401 => esc_html__( 'Error: you must enter Meta Key', 'searchfiltersort' ),
            402 => esc_html__( 'Error: invalid Taxonomy', 'searchfiltersort' ),
            41 => esc_html__( 'Error: invalid View parameter.', 'searchfiltersort' ),
            411 => esc_html__( 'Error: invalid Date Type parameter.', 'searchfiltersort' ),
            42 => esc_html__( 'Error: invalid Logic parameter', 'searchfiltersort' ),
            43 => esc_html__( 'Error: invalid the Sort Terms by parameter', 'searchfiltersort' ),
            44 => esc_html__( 'Error: invalid exclude terms', 'searchfiltersort' ),
            441 => esc_html__( 'Error: invalid include checkbox value', 'searchfiltersort' ),
            45 => esc_html__( 'Error: for filter Post Author logic OR is acceptable only.', 'searchfiltersort' ),
            46 => esc_html__( 'Error: numeric filter can not be in the URL path.', 'searchfiltersort' ),
            47 => esc_html__( 'Error: only AND logic is supported for numeric filters.', 'searchfiltersort' ),
            48 => esc_html__( 'Error: Range slider is acceptable for Post Meta Num filters only.', 'searchfiltersort' ),
            50 => esc_html__( 'Error: SEO Rule must have specified Post Type.', 'searchfiltersort' ),
            51 => esc_html__( 'Error: SEO Rule must contain at least one filter.', 'searchfiltersort' ),
            52 => esc_html__( 'Error: all SEO data fields could not be empty.', 'searchfiltersort' ),
            53 => esc_html__( 'Error: invalid or forbidden filter presents.', 'searchfiltersort' ),
            54 => esc_html__( 'Error: invalid SEO Rule ID.', 'searchfiltersort' ),
            55 => esc_html__( 'Error: SEO rule with selected Filters Combination already exists.', 'searchfiltersort' ),
            60 => esc_html__( 'Error: Invalid date format.', 'searchfiltersort' ),
            61 => esc_html__( 'Error: Invalid time format.', 'searchfiltersort' ),
            62 => esc_html__( 'Error: Invalid date or time format.', 'searchfiltersort' ),
            90 => wp_kses(
                sprintf(
                    __('Error: you can not update settings because the SearchFilterSort Pro plugin is locked. Please, enter your <a href="%1$s" target="_blank">license key</a> to unlock it.', 'searchfiltersort' ),
                    admin_url( 'edit.php?post_type=sfs-set&page=sfs-settings&tab=license' ) ),
                array(
                    'a' => array(
                        'href'=> true,
                        'target' => true
                    )
                )
            ),
            91 => esc_html__('Error: you can not update settings because the SearchFilterSort Pro plugin is locked. Please, ask your site Superadmin to enter the plugin license key to unlock it.', 'searchfiltersort' ),

        );

        return $errors;
    }
}