<?php
/**
 * The main plugin class
 *
 * @package WPApps_Searchable_Fields
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main WPApps_Searchable_Fields Class
 */
class Searchable_Parent_Dropdowns {

    /**
     * Plugin options
     *
     * @var array
     */
    public static $plugin_options = array();

    /**
     * Constructor
     */
    public function __construct() {
        $this->define_constants();
        $this->init_hooks();
        $this->init_plugin_options();
    }

    /**
     * Define constants
     */
    private function define_constants() {
        $this->define('WPAPPS_SF_ABSPATH', dirname(WPAPPS_SF_PLUGIN_DIR) . '/');
        $this->define('WPAPPS_SF_VERSION', '1.0.0');
    }

    /**
     * Define constant if not already set
     *
     * @param string $name  Constant name
     * @param mixed  $value Constant value
     */
    private function define($name, $value) {
        if (!defined($name)) {
            define($name, $value);
        }
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('admin_init', array($this, 'register_settings'));
        
        // Disable block editor for better compatibility
        add_filter('use_block_editor_for_post', '__return_false');
    }

    /**
     * Initialize plugin options
     */
    private function init_plugin_options() {
        self::$plugin_options = array(
            array(
                'selector' => '#parent_id, #inspector-select-control-1, #post_parent',
                'label'    => esc_html__('Parent Page', 'searchable-parent-dropdowns'),
                'id'       => 'parent_page'
            ),
            array(
                'selector' => '#addtag #parent, #edittag #parent',
                'label'    => esc_html__('Parent Category', 'searchable-parent-dropdowns'),
                'id'       => 'parent_category'
            ),
        );
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            esc_html__('Searchable Parent Dropdowns Settings', 'searchable-parent-dropdowns'),
            esc_html__('Searchable Parent Dropdowns', 'searchable-parent-dropdowns'),
            'manage_options',
            'searchable-parent-dropdowns',
            array($this, 'render_settings_page'),
            'dashicons-search',
            100
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting(
            'wpapps_sf_settings_group',
            'wpapps_sf_enabled_for',
            array(
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default'           => ''
            )
        );

        add_settings_section(
            'wpapps_sf_settings_section',
            esc_html__('Searchable Parent Dropdowns Settings', 'searchable-parent-dropdowns'),
            array($this, 'render_settings_section'),
            'searchable-parent-dropdowns'
        );

        add_settings_field(
            'wpapps_sf_enabled_fields',
            esc_html__('Enable Searchable Fields For', 'searchable-parent-dropdowns'),
            array($this, 'render_enabled_fields_setting'),
            'searchable-parent-dropdowns',
            'wpapps_sf_settings_section'
        );
    }

    /**
     * Render settings section
     */
    public function render_settings_section() {
        echo '<p>' . esc_html__('Select which fields should be converted to searchable dropdowns.', 'searchable-parent-dropdowns') . '</p>';
    }

    /**
     * Render enabled fields setting
     */
    public function render_enabled_fields_setting() {
        $enabled_for = get_option('wpapps_sf_enabled_for', '');
        $enabled_fields = !empty($enabled_for) ? explode(', ', $enabled_for) : array();
        
        foreach (self::$plugin_options as $option) {
            $checked = in_array($option['selector'], $enabled_fields) ? 'checked="checked"' : '';
            printf(
                '<p><label><input type="checkbox" name="wpapps_sf_enabled_fields[]" value="%s" %s> %s</label></p>',
                esc_attr($option['selector']),
                checked(in_array($option['selector'], $enabled_fields), true, false),
                esc_html($option['label'])
            );
        }
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }

        // Save settings if form was submitted
        if (isset($_POST['wpapps_sf_settings_nonce']) && wp_verify_nonce(sanitize_key($_POST['wpapps_sf_settings_nonce']), 'wpapps_sf_save_settings')) {
            // Initialize array to store enabled selectors
            $enabled_selectors = array();
            
            // Check which options were submitted
            if (isset($_POST['wpapps_sf_enabled_fields']) && is_array($_POST['wpapps_sf_enabled_fields'])) {
                // Sanitize the submitted values
                $submitted_fields = array_map('sanitize_text_field', wp_unslash($_POST['wpapps_sf_enabled_fields']));
                
                // Match submitted fields with our plugin options
                foreach (self::$plugin_options as $option) {
                    if (in_array($option['selector'], $submitted_fields)) {
                        $enabled_selectors[] = $option['selector'];
                    }
                }
            }
            
            // Save the enabled selectors
            update_option('wpapps_sf_enabled_for', implode(', ', $enabled_selectors));
            
            // Debug log
            error_log('Saved enabled fields: ' . print_r($enabled_selectors, true));
            
            add_settings_error(
                'wpapps_sf_messages',
                'wpapps_sf_message',
                esc_html__('Settings Saved', 'searchable-parent-dropdowns'),
                'updated'
            );
        }

        // Show error/update messages
        settings_errors('wpapps_sf_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <form action="" method="post">
                <?php
                // Output security fields
                wp_nonce_field('wpapps_sf_save_settings', 'wpapps_sf_settings_nonce');
                
                // Output settings sections
                do_settings_sections('searchable-parent-dropdowns');
                
                // Output save settings button
                submit_button(esc_html__('Save Settings', 'searchable-parent-dropdowns'));
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Enqueue admin scripts and styles
     *
     * @param string $hook Current admin page
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our plugin page and post edit screens
        if ($hook === 'toplevel_page_searchable-parent-dropdowns' || in_array($hook, array('post.php', 'post-new.php', 'edit-tags.php'))) {
            $enabled_for = get_option('wpapps_sf_enabled_for');
            
            if (!empty($enabled_for)) {
                // Enqueue Select2
                wp_enqueue_style(
                    'wpapps-sf-select2',
                    WPAPPS_SF_PLUGIN_URL . 'assets/css/select2.min.css',
                    array(),
                    '4.1.0-rc.0',
                    'all'
                );

                wp_enqueue_script(
                    'wpapps-sf-select2',
                    WPAPPS_SF_PLUGIN_URL . 'assets/js/select2.full.min.js',
                    array('jquery'),
                    '4.1.0-rc.0',
                    true
                );

                // Enqueue main script
                wp_enqueue_script(
                    'wpapps-sf-admin',
                    WPAPPS_SF_PLUGIN_URL . 'assets/js/admin.js',
                    array('jquery', 'wpapps-sf-select2'),
                    WPAPPS_SF_VERSION,
                    true
                );

                // Localize script with settings
                wp_localize_script('wpapps-sf-admin', 'wpappsSfSettings', array(
                    'selectors' => $enabled_for,
                    'i18n' => array(
                        'searchPlaceholder' => esc_attr__('Search...', 'searchable-parent-dropdowns'),
                        'noResults' => esc_attr__('No results found', 'searchable-parent-dropdowns'),
                    )
                ));
            }
        }
    }
}

// Initialize the plugin class
function wpapps_searchable_fields() {
    return new WPApps_Searchable_Fields();
}

// Start the plugin
add_action('plugins_loaded', 'wpapps_searchable_fields');
