<?php
/**
 * Taxonomies settings for the Search Anything Anywhere plugin.
 */
class Search_Anything_Anywhere_Taxonomies extends Search_Anything_Anywhere_Options_Base {
	/**
	 * Default taxonomy filters.
	 *
	 * @var array<string,bool>
	 */
	private array $default_taxonomy_filters = array( 'category' => true );

	/**
	 * Default taxonomy priority order.
	 *
	 * @var array<string>
	 */
	private array $default_taxonomy_priority = array( 'category' );

	/**
	 * Constructor.
	 */
	public function __construct() {
		Search_Anything_Anywhere::debug_log( 'init', 'Search_Anything_Anywhere_Taxonomies initialized' );
	}

	/**
	 * Get the default taxonomy filters.
	 *
	 * @return array<string,bool>
	 */
	public function get_default_taxonomy_filters(): array {
		$default_taxonomy_filters = $this->default_taxonomy_filters;
		return $default_taxonomy_filters;
	}

	/**
	 * Get the default taxonomy priority order.
	 *
	 * @return array<string>
	 */
	public function get_default_taxonomy_priority(): array {
		$default_taxonomy_priority = $this->default_taxonomy_priority;
		return $default_taxonomy_priority;
	}

	/**
	 * Register settings for the Taxonomies options.
	 */
	public function register_settings(): void {
		add_settings_section(
			'search_anything_anywhere_taxonomy_filters',
			__( 'Taxonomy Filters', 'search-anything-anywhere' ),
			function () {
				echo '<p class="description">' . esc_html__( 'Select which taxonomies to display as filters in the search form.', 'search-anything-anywhere' ) . '</p>';
			},
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'search_anything_anywhere_taxonomy_filters_list',
			__( 'Available Taxonomies', 'search-anything-anywhere' ),
			array( $this, 'render_taxonomy_filters' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_taxonomy_filters'
		);

		add_settings_section(
			'search_anything_anywhere_taxonomy_priority',
			__( 'Taxonomy Priority Order', 'search-anything-anywhere' ),
			function () {
				echo '<p class="description">' . esc_html__( 'Set the priority order for taxonomies. This order determines which taxonomy is assigned to a term when multiple taxonomies might contain the same term slug, ensuring unambiguous parsing in search queries. This order will be applied only when Friendly URLs are enabled in the General tab.', 'search-anything-anywhere' ) . '</p>';
			},
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'search_anything_anywhere_taxonomy_priority_list',
			__( 'Taxonomy Priority Order', 'search-anything-anywhere' ),
			array( $this, 'render_taxonomy_priority' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_taxonomy_priority'
		);

		wp_enqueue_style(
			'search-anything-anywhere-options-css',
			plugins_url( '../assets/css/options.css', __FILE__ ),
			array(),
			SEARCH_ANYTHING_ANYWHERE_VERSION
		);

		wp_enqueue_script(
			'search-anything-anywhere-taxonomies-js',
			plugins_url( '../assets/js/taxonomies.js', __FILE__ ),
			array( 'jquery', 'jquery-ui-sortable' ),
			SEARCH_ANYTHING_ANYWHERE_VERSION,
			true
		);

		wp_localize_script(
			'search-anything-anywhere-taxonomies-js',
			'searchAnythingAnywhereTaxonomies',
			array(
				'nonce'    => wp_create_nonce( 'search_anything_anywhere_taxonomies_nonce' ),
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'debug'    => ! empty( get_option( 'search_anything_anywhere_miscellaneous_settings_settings', array() )['enable_debug_logging'] ),
			)
		);

		Search_Anything_Anywhere::debug_log( 'settings', 'Taxonomies settings registered' );
	}

	/**
	 * Render the taxonomy filters settings.
	 */
	public function render_taxonomy_filters(): void {
		$options    = get_option( 'search_anything_anywhere_taxonomy_filters', $this->default_taxonomy_filters );
		$taxonomies = get_taxonomies( array( 'public' => true ), 'objects' );
		foreach ( $taxonomies as $taxonomy ) {
			$taxonomy_name = esc_attr( $taxonomy->name );
			$checked       = checked( 1, isset( $options[ $taxonomy->name ] ) ? $options[ $taxonomy->name ] : 0, false );
			$label_name    = esc_html( $taxonomy->labels->name );
			$label_machine = esc_html( $taxonomy->name );
			?>
			<input type="hidden" name="search_anything_anywhere_taxonomy_filters[<?php echo wp_kses_post( $taxonomy_name ); ?>]" value="0">
			<label>
				<input type="checkbox" name="search_anything_anywhere_taxonomy_filters[<?php echo wp_kses_post( $taxonomy_name ); ?>]" value="1" <?php echo wp_kses_post( $checked ); ?>>
				<?php echo wp_kses_post( $label_name ); ?> (<?php echo wp_kses_post( $label_machine ); ?>)
			</label>
			<br>
			<?php
		}
	}

	/**
	 * Render the taxonomy priority order settings.
	 */
	public function render_taxonomy_priority(): void {
		$taxonomy_filters  = get_option( 'search_anything_anywhere_taxonomy_filters', $this->get_default_taxonomy_filters() );
		$taxonomy_priority = get_option( 'search_anything_anywhere_taxonomy_priority', $this->get_default_taxonomy_priority() );
		$priority          = ! empty( $taxonomy_priority ) ? $taxonomy_priority : array_keys( $taxonomy_filters );
		$taxonomies        = get_taxonomies( array( 'public' => true ), 'objects' );

		Search_Anything_Anywhere::debug_log( 'render_priority', 'Taxonomy Filters: ' . json_encode( $taxonomy_filters ) );
		Search_Anything_Anywhere::debug_log( 'render_priority', 'Taxonomy Priority: ' . json_encode( $taxonomy_priority ) );

		if ( ! empty( $taxonomy_filters ) && array_filter( $taxonomy_filters ) ) {
			Search_Anything_Anywhere::debug_log( 'render_priority', 'Rendering taxonomy priority list' );
			echo '<ul id="taxonomy-priority-list">';
			$used_taxonomies = array();
			foreach ( $priority as $taxonomy ) {
				if ( isset( $taxonomy_filters[ $taxonomy ] ) && $taxonomy_filters[ $taxonomy ] && isset( $taxonomies[ $taxonomy ] ) ) {
					echo '<li data-taxonomy="' . esc_attr( $taxonomy ) . '">' . esc_html( $taxonomies[ $taxonomy ]->labels->name ) . ' (' . esc_html( $taxonomy ) . ')</li>';
					$used_taxonomies[] = $taxonomy;
					Search_Anything_Anywhere::debug_log( 'render_priority', "Added to list: $taxonomy" );
				}
			}
			foreach ( $taxonomy_filters as $taxonomy => $enabled ) {
				if ( $enabled && ! in_array( $taxonomy, $used_taxonomies, true ) && isset( $taxonomies[ $taxonomy ] ) ) {
					echo '<li data-taxonomy="' . esc_attr( $taxonomy ) . '">' . esc_html( $taxonomies[ $taxonomy ]->labels->name ) . ' (' . esc_html( $taxonomy ) . ')</li>';
					$used_taxonomies[] = $taxonomy;
					Search_Anything_Anywhere::debug_log( 'render_priority', "Added remaining to list: $taxonomy" );
				}
			}
			echo '</ul>';
			echo '<input type="hidden" name="search_anything_anywhere_taxonomy_priority[taxonomy_priority]" id="taxonomy-priority-field" value="' . esc_attr( json_encode( $priority ) ) . '">';
			echo '<p>' . esc_html__( 'Drag and drop to reorder taxonomies.', 'search-anything-anywhere' ) . '</p>';
		} else {
			Search_Anything_Anywhere::debug_log( 'render_priority', 'No taxonomies enabled' );
			echo '<p>' . esc_html__( 'No taxonomies are enabled. Please select taxonomies above to configure priority.', 'search-anything-anywhere' ) . '</p>';
		}
	}

	/**
	 * Sanitize the taxonomy filter input.
	 *
	 * @param array $input The raw input from the settings form.
	 * @return array Sanitized taxonomy filters.
	 */
	public function sanitize_taxonomy_filters( $input ): array {
		$sanitized  = array();
		$taxonomies = get_taxonomies( array( 'public' => true ) );
		foreach ( $taxonomies as $taxonomy ) {
			$sanitized[ $taxonomy ] = isset( $input[ $taxonomy ] ) && '1' === $input[ $taxonomy ] ? true : false;
		}

		// Sync taxonomy_priority with enabled taxonomies
		$current_priority   = get_option( 'search_anything_anywhere_taxonomy_priority', $this->get_default_taxonomy_priority() );
		$enabled_taxonomies = array_keys( array_filter( $sanitized ) );
		$updated_priority   = array();

		// Explicitly include only enabled taxonomies from current priority
		Search_Anything_Anywhere::debug_log( 'sanitize_filters', 'Current priority: ' . json_encode( $current_priority ) );
		Search_Anything_Anywhere::debug_log( 'sanitize_filters', 'Enabled taxonomies: ' . json_encode( $enabled_taxonomies ) );
		foreach ( $current_priority as $taxonomy ) {
			if ( in_array( $taxonomy, $enabled_taxonomies, true ) ) {
				$updated_priority[] = $taxonomy;
				Search_Anything_Anywhere::debug_log( 'sanitize_filters', "Kept enabled taxonomy: $taxonomy" );
			} else {
				Search_Anything_Anywhere::debug_log( 'sanitize_filters', "Removed disabled taxonomy: $taxonomy" );
			}
		}

		// Add any newly enabled taxonomies not in the current priority
		foreach ( $enabled_taxonomies as $taxonomy ) {
			if ( ! in_array( $taxonomy, $updated_priority, true ) ) {
				$updated_priority[] = $taxonomy;
				Search_Anything_Anywhere::debug_log( 'sanitize_filters', "Added new enabled taxonomy: $taxonomy" );
			}
		}

		// Update search_anything_anywhere_taxonomy_priority with the new priority and force immediate save
		Search_Anything_Anywhere::debug_log( 'sanitize_filters', 'Updating taxonomy_priority to: ' . json_encode( $updated_priority ) );
		update_option( 'search_anything_anywhere_taxonomy_priority', $updated_priority, true ); // Force update

		return $sanitized;
	}

	/**
	 * Sanitize the taxonomy priority input.
	 *
	 * @param array $input The raw input from the settings form.
	 * @return array Sanitized taxonomy priority order.
	 */
	public function sanitize_taxonomy_priority( $input ): array {
		$sanitized = array();

		if ( isset( $input['taxonomy_priority'] ) ) {
			$priority = is_string( $input['taxonomy_priority'] ) ? json_decode( $input['taxonomy_priority'], true ) : $input['taxonomy_priority'];
			Search_Anything_Anywhere::debug_log( 'sanitize', 'Raw submitted taxonomy_priority: ' . json_encode( $priority ) );
			if ( is_array( $priority ) ) {
				$valid_taxonomies   = array_keys( get_taxonomies( array( 'public' => true ) ) );
				$taxonomy_filters   = get_option( 'search_anything_anywhere_taxonomy_filters', array() );
				$enabled_taxonomies = array_keys( array_filter( $taxonomy_filters ) );
				// Preserve the submitted order, only keeping enabled and valid taxonomies
				$sanitized = array_values(
					array_filter(
						$priority,
						function ( $taxonomy ) use ( $enabled_taxonomies, $valid_taxonomies ) {
							return in_array( $taxonomy, $enabled_taxonomies, true ) && in_array( $taxonomy, $valid_taxonomies, true );
						}
					)
				);
				// Ensure all enabled taxonomies are included, maintaining the submitted order where possible
				foreach ( $enabled_taxonomies as $taxonomy ) {
					if ( ! in_array( $taxonomy, $sanitized, true ) && in_array( $taxonomy, $valid_taxonomies, true ) ) {
						$sanitized[] = $taxonomy;
						Search_Anything_Anywhere::debug_log( 'sanitize', "Added missing enabled taxonomy: $taxonomy" );
					}
				}
				Search_Anything_Anywhere::debug_log( 'sanitize', 'Sanitized taxonomy_priority: ' . json_encode( $sanitized ) );
			} else {
				Search_Anything_Anywhere::debug_log( 'sanitize', 'taxonomy_priority decoding failed, set to empty array' );
			}
		}

		return $sanitized;
	}

	/**
	 * Reset the Taxonomies options to their defaults.
	 */
	public function reset_options(): void {
		global $wp_settings_fields;

		Search_Anything_Anywhere::debug_log( 'reset', 'Resetting Taxonomies options - Filters to: ' . json_encode( $this->default_taxonomy_filters ) );
		Search_Anything_Anywhere::debug_log( 'reset', 'Resetting Taxonomies options - Priority to: ' . json_encode( $this->default_taxonomy_priority ) );

		// Temporarily remove sanitization callbacks to prevent interference during reset
		$filters_settings  = $wp_settings_fields['search_anything_anywhere_options']['search_anything_anywhere_taxonomy_filters_list'] ?? array();
		$priority_settings = $wp_settings_fields['search_anything_anywhere_options']['search_anything_anywhere_taxonomy_priority_list'] ?? array();

		$original_filters_sanitize_callback  = $filters_settings['args']['sanitize_callback'] ?? null;
		$original_priority_sanitize_callback = $priority_settings['args']['sanitize_callback'] ?? null;

		unregister_setting( 'search_anything_anywhere_taxonomy_filters', 'search_anything_anywhere_taxonomy_filters' );
		unregister_setting( 'search_anything_anywhere_taxonomy_priority', 'search_anything_anywhere_taxonomy_priority' );

		// Perform the reset without sanitization
		update_option( 'search_anything_anywhere_taxonomy_filters', $this->default_taxonomy_filters );
		update_option( 'search_anything_anywhere_taxonomy_priority', $this->default_taxonomy_priority );

		// Restore the sanitization callbacks
		register_setting(
			'search_anything_anywhere_taxonomy_filters',
			'search_anything_anywhere_taxonomy_filters',
			array( 'sanitize_callback' => $original_filters_sanitize_callback )
		);
		register_setting(
			'search_anything_anywhere_taxonomy_priority',
			'search_anything_anywhere_taxonomy_priority',
			array( 'sanitize_callback' => $original_priority_sanitize_callback )
		);

		Search_Anything_Anywhere::debug_log( 'reset', 'Taxonomies options reset completed - Filters: ' . json_encode( get_option( 'search_anything_anywhere_taxonomy_filters' ) ) );
		Search_Anything_Anywhere::debug_log( 'reset', 'Taxonomies options reset completed - Priority: ' . json_encode( get_option( 'search_anything_anywhere_taxonomy_priority' ) ) );
	}
}
