<?php
/**
 * Main settings class for the Search Anything Anywhere plugin.
 *
 * Handles the admin menu, tab navigation, and settings registration.
 */
class Search_Anything_Anywhere_Options {
	/**
	 * Active tab for the settings page.
	 *
	 * @var string
	 */
	private string $active_tab;

	/**
	 * Tab constants for easy reference.
	 */
	const TAB_GENERAL       = 'general';
	const TAB_DISPLAY       = 'display';
	const TAB_TAXONOMIES    = 'taxonomies';
	const TAB_MISCELLANEOUS = 'miscellaneous';

	/**
	 * Option group names for settings registration.
	 */
	const OPTION_GROUP_GENERAL       = 'search_anything_anywhere_general_options';
	const OPTION_GROUP_DISPLAY       = 'search_anything_anywhere_display_options';
	const OPTION_GROUP_TAXONOMIES    = 'search_anything_anywhere_taxonomies_settings';
	const OPTION_GROUP_MISCELLANEOUS = 'search_anything_anywhere_miscellaneous_options';

	/**
	 * Available tabs for the settings page.
	 *
	 * @var array<string,string>
	 */
	private array $available_tabs = array(
		self::TAB_GENERAL       => 'General',
		self::TAB_DISPLAY       => 'Display',
		self::TAB_TAXONOMIES    => 'Taxonomies',
		self::TAB_MISCELLANEOUS => 'Miscellaneous',
	);

	/**
	 * General, Display, Taxonomies, and Miscellaneous option classes.
	 *
	 * @var Search_Anything_Anywhere_General|null
	 * @var Search_Anything_Anywhere_Display|null
	 * @var Search_Anything_Anywhere_Taxonomies|null
	 * @var Search_Anything_Anywhere_Miscellaneous|null
	 */
	private ?Search_Anything_Anywhere_General $general             = null;
	private ?Search_Anything_Anywhere_Display $display             = null;
	private ?Search_Anything_Anywhere_Taxonomies $taxonomies       = null;
	private ?Search_Anything_Anywhere_Miscellaneous $miscellaneous = null;

	/**
	 * Constructor to initialize the options class.
	 * @return void
	 */
	public function __construct() {
		$required_classes = array(
			'Search_Anything_Anywhere_General',
			'Search_Anything_Anywhere_Display',
			'Search_Anything_Anywhere_Taxonomies',
			'Search_Anything_Anywhere_Miscellaneous',
		);

		foreach ( $required_classes as $class ) {
			if ( ! class_exists( $class ) ) {
				Search_Anything_Anywhere::debug_log( 'class_load', "$class class not found" );
				return;
			}
		}

		$this->general       = new Search_Anything_Anywhere_General();
		$this->display       = new Search_Anything_Anywhere_Display();
		$this->taxonomies    = new Search_Anything_Anywhere_Taxonomies();
		$this->miscellaneous = new Search_Anything_Anywhere_Miscellaneous();

		$this->set_active_tab();

		add_action( 'admin_menu', array( $this, 'add_options_page' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_init', array( $this, 'handle_reset_options' ) );
		Search_Anything_Anywhere::debug_log( 'init', 'Search_Anything_Anywhere_Options initialized' );
	}

	/**
	 * Set the active tab based on the current request.
	 *
	 * Defaults to 'general' if no tab is specified.
	 */
	private function set_active_tab(): void {
		$tab = self::TAB_GENERAL;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for admin tab selection.
		if ( isset( $_GET['tab'] ) && is_string( $_GET['tab'] ) ) {
			$un_slashed_tab = sanitize_text_field( wp_unslash( $_GET['tab'] ) );
			if ( array_key_exists( $un_slashed_tab, $this->available_tabs ) ) {
				$tab = $un_slashed_tab;
			}
		}
		$this->active_tab = $tab;

		Search_Anything_Anywhere::debug_log( 'tab_selection', "Active tab set to: $tab" );
	}

	/**
	 * Add the options page and submenus to the admin menu.
	 */
	public function add_options_page(): void {
		add_options_page(
			__( 'Search Anything Anywhere Settings', 'search-anything-anywhere' ),
			__( 'Search', 'search-anything-anywhere' ),
			'manage_options',
			'search-anything-anywhere-settings',
			array( $this, 'render_options_page' ),
			7
		);

		add_submenu_page(
			'search-anything-anywhere-settings',
			'Search Anything Anywhere General Settings',
			'General',
			'manage_options',
			'search-anything-anywhere-settings',
			array( $this, 'render_options_page' )
		);

		add_submenu_page(
			'search-anything-anywhere-settings',
			'Search Anything Anywhere Display Settings',
			'Display',
			'manage_options',
			'search-anything-anywhere-settings&tab=' . self::TAB_DISPLAY,
			array( $this, 'render_options_page' )
		);

		add_submenu_page(
			'search-anything-anywhere-settings',
			'Search Anything Anywhere Taxonomies Settings',
			'Taxonomies',
			'manage_options',
			'search-anything-anywhere-settings&tab=' . self::TAB_TAXONOMIES,
			array( $this, 'render_options_page' )
		);

		add_submenu_page(
			'search-anything-anywhere-settings',
			'Search Anything Anywhere Miscellaneous Settings',
			'Miscellaneous',
			'manage_options',
			'search-anything-anywhere-settings&tab=' . self::TAB_MISCELLANEOUS,
			array( $this, 'render_options_page' )
		);

		add_action(
			'admin_head',
			function () {
				global $submenu_file;
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for admin submenu highlighting.
				if ( isset( $_GET['page'] ) && 'search-anything-anywhere-settings' === $_GET['page'] ) {
					$submenu_file = isset( $_GET['tab'] )
						? 'search-anything-anywhere-settings&tab=' . sanitize_text_field( wp_unslash( $_GET['tab'] ) )
						: 'search-anything-anywhere-settings';
				}
			}
		);

		Search_Anything_Anywhere::debug_log( 'admin_menu', 'Options page and submenus added' );
	}

	/**
	 * Register settings for the active tab.
	 *
	 * This method registers the settings for the General, Display, Taxonomies, and Miscellaneous tabs.
	 */
	public function register_settings(): void {
		if ( ! $this->general || ! $this->display || ! $this->taxonomies || ! $this->miscellaneous ) {
			Search_Anything_Anywhere::debug_log( 'settings_registration', 'General, Display, Taxonomies or Miscellaneous classes not initialized' );
			return;
		}

		register_setting(
			self::OPTION_GROUP_GENERAL,
			'search_anything_anywhere_general_options',
			array( 'sanitize_callback' => array( $this->general, 'sanitize_options' ) )
		);

		register_setting(
			self::OPTION_GROUP_DISPLAY,
			'search_anything_anywhere_display_options',
			array( 'sanitize_callback' => array( $this->display, 'sanitize_options' ) )
		);

		register_setting(
			self::OPTION_GROUP_TAXONOMIES,
			'search_anything_anywhere_taxonomy_filters',
			array( 'sanitize_callback' => array( $this->taxonomies, 'sanitize_taxonomy_filters' ) )
		);

		register_setting(
			self::OPTION_GROUP_TAXONOMIES,
			'search_anything_anywhere_taxonomy_priority',
			array( 'sanitize_callback' => array( $this->taxonomies, 'sanitize_taxonomy_priority' ) )
		);

		register_setting(
			self::OPTION_GROUP_MISCELLANEOUS,
			'search_anything_anywhere_miscellaneous_options',
			array( 'sanitize_callback' => array( $this->miscellaneous, 'sanitize_options' ) )
		);

		if ( self::TAB_GENERAL === $this->active_tab ) {
			$this->general->register_settings();
		} elseif ( self::TAB_DISPLAY === $this->active_tab ) {
			$this->display->register_settings();
		} elseif ( self::TAB_TAXONOMIES === $this->active_tab ) {
			$this->taxonomies->register_settings();
		} elseif ( self::TAB_MISCELLANEOUS === $this->active_tab ) {
			$this->miscellaneous->register_settings();
		}

		Search_Anything_Anywhere::debug_log( 'settings_registration', "Settings registered for tab: {$this->active_tab}" );
	}

	/**
	 * Handle reset options request.
	 *
	 * This method processes the reset options form submission and resets the settings for the specified tab or all tabs.
	 */
	public function handle_reset_options(): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified below for reset actions.
		$reset_general       = isset( $_POST['search_anything_anywhere_reset_general'] ) ? sanitize_text_field( wp_unslash( $_POST['search_anything_anywhere_reset_general'] ) ) : null;
		$reset_display       = isset( $_POST['search_anything_anywhere_reset_display'] ) ? sanitize_text_field( wp_unslash( $_POST['search_anything_anywhere_reset_display'] ) ) : null;
		$reset_taxonomies    = isset( $_POST['search_anything_anywhere_reset_taxonomies'] ) ? sanitize_text_field( wp_unslash( $_POST['search_anything_anywhere_reset_taxonomies'] ) ) : null;
		$reset_miscellaneous = isset( $_POST['search_anything_anywhere_reset_miscellaneous'] ) ? sanitize_text_field( wp_unslash( $_POST['search_anything_anywhere_reset_miscellaneous'] ) ) : null;
		$reset_all           = isset( $_POST['search_anything_anywhere_reset_all'] ) ? sanitize_text_field( wp_unslash( $_POST['search_anything_anywhere_reset_all'] ) ) : null;

		if ( ! $reset_general && ! $reset_display && ! $reset_taxonomies && ! $reset_miscellaneous && ! $reset_all ) {
			return;
		}

		if ( ! isset( $_POST['_wpnonce'] ) || ! is_string( $_POST['_wpnonce'] ) ) {
			Search_Anything_Anywhere::debug_log( 'reset_options', 'Nonce verification failed' );
			add_settings_error( 'search_anything_anywhere_options', 'search_anything_anywhere_nonce_failed', 'Security check failed. Please try again.', 'error' );
			return;
		}
		$wpnonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) );
		if ( ! wp_verify_nonce( $wpnonce, 'search_anything_anywhere_reset_options_nonce' ) ) {
			Search_Anything_Anywhere::debug_log( 'reset_options', 'Nonce verification failed' );
			add_settings_error( 'search_anything_anywhere_options', 'search_anything_anywhere_nonce_failed', 'Security check failed. Please try again.', 'error' );
			return;
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			Search_Anything_Anywhere::debug_log( 'reset_options', 'User lacks manage_options permission' );
			add_settings_error( 'search_anything_anywhere_options', 'search_anything_anywhere_permission_denied', 'Insufficient permissions to reset settings.', 'error' );
			return;
		}

		$reset_message                = '';
		$before_general_settings      = get_option( 'search_anything_anywhere_general_options', array() );
		$was_enable_friendly_urls_off = empty( $before_general_settings['enable_friendly_urls'] );

		if ( isset( $_POST['search_anything_anywhere_reset_general'] ) && $this->general ) {
			$before_reset = $before_general_settings;
			Search_Anything_Anywhere::debug_log( 'reset_options', 'Before General reset: ' . json_encode( $before_reset ) );
			$this->general->reset_options();
			$after_reset   = get_option( 'search_anything_anywhere_general_options' );
			$reset_message = 'General settings reset to defaults successfully.';
			Search_Anything_Anywhere::debug_log( 'reset_options', 'After General reset: ' . json_encode( $after_reset ) );
		} elseif ( isset( $_POST['search_anything_anywhere_reset_display'] ) && $this->display ) {
			$before_reset = get_option( 'search_anything_anywhere_display_options', array() );
			$this->display->reset_options();
			$after_reset   = get_option( 'search_anything_anywhere_display_options' );
			$reset_message = 'Display settings reset to defaults successfully.';
			Search_Anything_Anywhere::debug_log( 'reset_options', 'Before Display reset: ' . json_encode( $before_reset ) );
			Search_Anything_Anywhere::debug_log( 'reset_options', 'After Display reset: ' . json_encode( $after_reset ) );
		} elseif ( isset( $_POST['search_anything_anywhere_reset_taxonomies'] ) && $this->taxonomies ) {
			$before_reset_filters  = get_option( 'search_anything_anywhere_taxonomy_filters', array() );
			$before_reset_priority = get_option( 'search_anything_anywhere_taxonomy_priority', array() );
			$this->taxonomies->reset_options();
			$after_reset_filters  = get_option( 'search_anything_anywhere_taxonomy_filters' );
			$after_reset_priority = get_option( 'search_anything_anywhere_taxonomy_priority' );
			$reset_message        = 'Taxonomies settings reset to defaults successfully.';
			Search_Anything_Anywhere::debug_log( 'reset_options', 'Before Taxonomies reset - Filters: ' . json_encode( $before_reset_filters ) );
			Search_Anything_Anywhere::debug_log( 'reset_options', 'Before Taxonomies reset - Priority: ' . json_encode( $before_reset_priority ) );
			Search_Anything_Anywhere::debug_log( 'reset_options', 'After Taxonomies reset - Filters: ' . json_encode( $after_reset_filters ) );
			Search_Anything_Anywhere::debug_log( 'reset_options', 'After Taxonomies reset - Priority: ' . json_encode( $after_reset_priority ) );
		} elseif ( isset( $_POST['search_anything_anywhere_reset_miscellaneous'] ) && $this->miscellaneous ) {
			$before_reset = get_option( 'search_anything_anywhere_miscellaneous_options', array() );
			$this->miscellaneous->reset_options();
			$after_reset   = get_option( 'search_anything_anywhere_miscellaneous_options' );
			$reset_message = 'Miscellaneous settings reset to defaults successfully.';
			Search_Anything_Anywhere::debug_log( 'reset_options', 'Before Miscellaneous reset: ' . json_encode( $before_reset ) );
			Search_Anything_Anywhere::debug_log( 'reset_options', 'After Miscellaneous reset: ' . json_encode( $after_reset ) );
		} elseif ( isset( $_POST['search_anything_anywhere_reset_all'] ) ) {
			if ( $this->general ) {
				$before_reset_general = $before_general_settings;
				Search_Anything_Anywhere::debug_log( 'reset_options', 'Before All reset - General: ' . json_encode( $before_reset_general ) );
				$this->general->reset_options();
				$after_reset_general = get_option( 'search_anything_anywhere_general_options' );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'After All reset - General: ' . json_encode( $after_reset_general ) );
			}

			if ( $this->display ) {
				$before_reset_display = get_option( 'search_anything_anywhere_display_options', array() );
				$this->display->reset_options();
				$after_reset_display = get_option( 'search_anything_anywhere_display_options' );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'Before All reset - Display: ' . json_encode( $before_reset_display ) );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'After All reset - Display: ' . json_encode( $after_reset_display ) );
			}

			if ( $this->taxonomies ) {
				$before_reset_filters  = get_option( 'search_anything_anywhere_taxonomy_filters', array() );
				$before_reset_priority = get_option( 'search_anything_anywhere_taxonomy_priority', array() );
				$this->taxonomies->reset_options();
				$after_reset_filters  = get_option( 'search_anything_anywhere_taxonomy_filters' );
				$after_reset_priority = get_option( 'search_anything_anywhere_taxonomy_priority' );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'Before All reset - Filters: ' . json_encode( $before_reset_filters ) );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'Before All reset - Priority: ' . json_encode( $before_reset_priority ) );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'After All reset - Filters: ' . json_encode( $after_reset_filters ) );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'After All reset - Priority: ' . json_encode( $after_reset_priority ) );
			}

			if ( $this->miscellaneous ) {
				$before_reset_miscellaneous = get_option( 'search_anything_anywhere_miscellaneous_options', array() );
				$this->miscellaneous->reset_options();
				$after_reset_miscellaneous = get_option( 'search_anything_anywhere_miscellaneous_options' );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'Before All reset - Miscellaneous: ' . json_encode( $before_reset_miscellaneous ) );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'After All reset - Miscellaneous: ' . json_encode( $after_reset_miscellaneous ) );
			}

			$reset_message = 'All settings reset to defaults successfully.';
		}

		// Check if enable_friendly_urls was off and is now enabled after reset
		if ( ( $reset_general || $reset_all ) && $was_enable_friendly_urls_off ) {
			$after_general_settings = get_option( 'search_anything_anywhere_general_options', array() );
			if ( ! empty( $after_general_settings['enable_friendly_urls'] ) ) {
				set_transient( 'search_anything_anywhere_flush_permalinks', true, DAY_IN_SECONDS );
				Search_Anything_Anywhere::debug_log( 'reset_options', 'Friendly URLs enabled after reset, set flush permalinks transient' );
			}
		}

		if ( $reset_message ) {
			Search_Anything_Anywhere::debug_log( 'reset_options', $reset_message );
			delete_option( 'search_anything_anywhere_general_options_cache' );
			delete_option( 'search_anything_anywhere_display_settings_cache' );
			delete_option( 'search_anything_anywhere_taxonomy_filters_cache' );
			delete_option( 'search_anything_anywhere_taxonomy_priority_cache' );
			delete_option( 'search_anything_anywhere_miscellaneous_settings_cache' );

			// Clear all existing messages to prevent conflicts
			delete_transient( 'search_anything_anywhere_options_message' );

			// Set transient with reset-specific message
			set_transient(
				'search_anything_anywhere_options_message',
				array(
					'code'    => 'search_anything_anywhere_options_reset',
					'message' => $reset_message,
					'type'    => 'updated',
				),
				60
			);

			$current_tab     = isset( $_GET['tab'] ) && is_string( $_GET['tab'] ) ? '&tab=' . sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : '';
			$reset_all_param = $reset_all ? '&reset-all=true' : '';
			wp_redirect( admin_url( 'options-general.php?page=search-anything-anywhere-settings' . $current_tab . '&reset-tab=true' . $reset_all_param ) );
			exit;
		}
	}


	/**
	 * Render the options page with tabs and settings forms.
	 */
	public function render_options_page(): void {
		if ( ! $this->general || ! $this->display || ! $this->taxonomies || ! $this->miscellaneous ) {
			echo '<div class="error"><p>Error: Search Anything Anywhere settings could not be loaded. Please check the plugin configuration.</p></div>';
			return;
		}

		$transient = get_transient( 'search_anything_anywhere_options_message' );
		if ( $transient ) {
			$notice_type = 'notice-warning' === $transient['type'] ? 'notice-warning' : 'updated';
			add_settings_error(
				'search_anything_anywhere_options',
				$transient['code'],
				$transient['message'],
				$notice_type
			);
			delete_transient( 'search_anything_anywhere_options_message' );
		}

		if ( self::TAB_GENERAL === $this->active_tab ) {
			Search_Anything_Anywhere::debug_log( 'render_page', 'Rendering General tab with options: ' . json_encode( get_option( 'search_anything_anywhere_general_options' ) ) );
		} elseif ( self::TAB_DISPLAY === $this->active_tab ) {
			Search_Anything_Anywhere::debug_log( 'render_page', 'Rendering Display tab with options: ' . json_encode( get_option( 'search_anything_anywhere_display_options' ) ) );
		} elseif ( self::TAB_TAXONOMIES === $this->active_tab ) {
			Search_Anything_Anywhere::debug_log( 'render_page', 'Rendering Taxonomies tab - Filters: ' . json_encode( get_option( 'search_anything_anywhere_taxonomy_filters' ) ) );
			Search_Anything_Anywhere::debug_log( 'render_page', 'Rendering Taxonomies tab - Priority: ' . json_encode( get_option( 'search_anything_anywhere_taxonomy_priority' ) ) );
		} elseif ( self::TAB_MISCELLANEOUS === $this->active_tab ) {
			Search_Anything_Anywhere::debug_log( 'render_page', 'Rendering Miscellaneous tab with options: ' . json_encode( get_option( 'search_anything_anywhere_miscellaneous_options' ) ) );
		}

		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Search Anything Anywhere Settings', 'search-anything-anywhere' ); ?></h1>
			<?php settings_errors( 'search_anything_anywhere_options' ); ?>
			<h2 class="nav-tab-wrapper">
				<?php foreach ( $this->available_tabs as $tab => $label ) : ?>
					<a href="?page=search-anything-anywhere-settings&tab=<?php echo esc_attr( $tab ); ?>"
						class="nav-tab <?php echo $tab === $this->active_tab ? 'nav-tab-active' : ''; ?>">
						<?php echo esc_html( $label ); ?>
					</a>
				<?php endforeach; ?>
			</h2>
			<form method="post" action="options.php">
				<?php
				if ( self::TAB_GENERAL === $this->active_tab ) {
					settings_fields( self::OPTION_GROUP_GENERAL );
				} elseif ( self::TAB_DISPLAY === $this->active_tab ) {
					settings_fields( self::OPTION_GROUP_DISPLAY );
				} elseif ( self::TAB_TAXONOMIES === $this->active_tab ) {
					settings_fields( self::OPTION_GROUP_TAXONOMIES );
				} elseif ( self::TAB_MISCELLANEOUS === $this->active_tab ) {
					settings_fields( self::OPTION_GROUP_MISCELLANEOUS );
				}
				do_settings_sections( 'search_anything_anywhere_options' );
				submit_button();
				?>
			</form>
			<div class="reset-options-section" style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #ccc;">
				<h3>Reset Options</h3>
				<p>Reset settings for the current tab or all settings:</p>
				<form method="post" action="">
					<?php wp_nonce_field( 'search_anything_anywhere_reset_options_nonce', '_wpnonce' ); ?>
					<?php
					if ( self::TAB_GENERAL === $this->active_tab ) {
						submit_button( 'Reset General Options', 'secondary', 'search_anything_anywhere_reset_general', false, array( 'onclick' => 'return confirm("Are you sure you want to reset general options?");' ) );
					} elseif ( self::TAB_DISPLAY === $this->active_tab ) {
						submit_button( 'Reset Display Options', 'secondary', 'search_anything_anywhere_reset_display', false, array( 'onclick' => 'return confirm("Are you sure you want to reset display options?");' ) );
					} elseif ( self::TAB_TAXONOMIES === $this->active_tab ) {
						submit_button( 'Reset Taxonomies Options', 'secondary', 'search_anything_anywhere_reset_taxonomies', false, array( 'onclick' => 'return confirm("Are you sure you want to reset taxonomies options?");' ) );
					} elseif ( self::TAB_MISCELLANEOUS === $this->active_tab ) {
						submit_button( 'Reset Miscellaneous Options', 'secondary', 'search_anything_anywhere_reset_miscellaneous', false, array( 'onclick' => 'return confirm("Are you sure you want to reset miscellaneous options?");' ) );
					}

					submit_button(
						'Reset All Options',
						'secondary',
						'search_anything_anywhere_reset_all',
						false,
						array(
							'onclick' => 'return confirm("Are you sure you want to reset all options?");',
							'style'   => 'margin-left: 10px;',
						)
					);
					?>
				</form>
			</div>
		</div>
		<?php
		Search_Anything_Anywhere::debug_log( 'render_page', 'Options page rendered' );
	}
}
