<?php
/**
 * Handles admin notices for the Search Anything Anywhere plugin.
 */
class Search_Anything_Anywhere_Notices {

	/**
	 * Register hooks for admin notices.
	 */
	public function register() {
		add_action( 'admin_notices', array( $this, 'activation_notice' ) );
		add_action( 'admin_notices', array( $this, 'permalink_notice' ) );
		add_action( 'wp_ajax_search_anything_anywhere_dismiss_activation_notice', array( $this, 'dismiss_activation_notice' ) );
		add_action( 'wp_ajax_search_anything_anywhere_dismiss_permalink_notice', array( $this, 'dismiss_permalink_notice' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
		Search_Anything_Anywhere::debug_log( 'init', 'Search_Anything_Anywhere_Notices registered' );
	}

	/**
	 * Display activation notice if the transient is set.
	 */
	public function activation_notice() {
		if ( ! current_user_can( 'manage_options' ) || ! get_transient( 'search_anything_anywhere_activation_notice' ) ) {
			return;
		}

		$message = sprintf(
			/* Translators: %1$s: URL to permalinks settings, %2$s: URL to Search Anything Anywhere settings */
			__( 'Search Anything Anywhere: Thank you for activating the plugin! To ensure proper functionality, please <a href="%1$s">flush permalinks</a>. Additionally, clear any site, browser, or CDN cache to ensure the latest changes are reflected. Customize your settings in the <a href="%2$s">Search Anything Anywhere settings page</a>.', 'search-anything-anywhere' ),
			admin_url( 'options-permalink.php' ),
			admin_url( 'admin.php?page=search-anything-anywhere-settings' )
		);
		?>
		<div id="search-anything-anywhere-activation-notice" class="notice notice-info is-dismissible">
			<p><?php echo wp_kses_post( $message ); ?></p>
		</div>
		<?php
		Search_Anything_Anywhere::debug_log( 'activation_notice', 'Displayed activation notice' );
	}

	/**
	 * Display permalink flush notice if the transient is set.
	 */
	public function permalink_notice() {
		if ( ! current_user_can( 'manage_options' ) || ! get_transient( 'search_anything_anywhere_flush_permalinks' ) || get_transient( 'search_anything_anywhere_activation_notice' ) ) {
			return;
		}

		$message = sprintf(
			/* Translators: %s: URL to the permalinks settings page. */
			__( 'Search Anything Anywhere: Friendly URLs have been enabled. Please <a href="%s">flush permalinks</a> to activate them. Ensure your .htaccess file is writable.', 'search-anything-anywhere' ),
			admin_url( 'options-permalink.php' )
		);
		?>
		<div id="search-anything-anywhere-permalink-notice" class="notice notice-warning is-dismissible">
			<p><?php echo wp_kses_post( $message ); ?></p>
		</div>
		<?php
		Search_Anything_Anywhere::debug_log( 'permalink_notice', 'Displayed permalink flush notice' );
	}

	/**
	 * Handle AJAX request to dismiss the activation notice.
	 */
	public function dismiss_activation_notice() {
		check_ajax_referer( 'search_anything_anywhere_dismiss_activation_notice', 'nonce' );
		if ( current_user_can( 'manage_options' ) ) {
			delete_transient( 'search_anything_anywhere_activation_notice' );
			// Also dismiss permalink notice if it exists.
			$this->dismiss_permalink_notice();
			Search_Anything_Anywhere::debug_log( 'activation_notice', 'Activation notice dismissed' );
			wp_send_json_success();
		} else {
			Search_Anything_Anywhere::debug_log( 'activation_notice', 'User lacks permission to dismiss activation notice' );
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'search-anything-anywhere' ) ) );
		}
	}

	/**
	 * Handle AJAX request to dismiss the permalink notice.
	 */
	public function dismiss_permalink_notice() {
		check_ajax_referer( 'search_anything_anywhere_dismiss_permalink_notice', 'nonce' );
		if ( ! current_user_can( 'manage_options' ) ) {
			Search_Anything_Anywhere::debug_log( 'permalink_notice', 'User lacks permission to dismiss permalink notice' );
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'search-anything-anywhere' ) ) );
		}

		// Check .htaccess writable.
		$htaccess_file   = ABSPATH . '.htaccess';
		$htaccess_status = '';
		global $wp_filesystem;
		if ( ! function_exists( 'request_filesystem_credentials' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}
		if ( ! $wp_filesystem ) {
			WP_Filesystem();
		}

		if ( ! $wp_filesystem->exists( $htaccess_file ) ) {
			$htaccess_status = 'missing';
		} elseif ( ! $wp_filesystem->is_writable( $htaccess_file ) ) {
			$htaccess_status = 'not_writable';
		} else {
			$htaccess_content = $wp_filesystem->get_contents( $htaccess_file );
			if ( strpos( $htaccess_content, '# BEGIN WordPress' ) === false ) {
				$htaccess_status = 'no_wp_rules';
			}
		}

		if ( $htaccess_status ) {
			$error_message = __( 'Please ensure your .htaccess file is configured correctly: ', 'search-anything-anywhere' );
			switch ( $htaccess_status ) {
				case 'missing':
					$error_message .= __( '.htaccess file is missing.', 'search-anything-anywhere' );
					break;
				case 'not_writable':
					$error_message .= __( '.htaccess file is not writable.', 'search-anything-anywhere' );
					break;
				case 'no_wp_rules':
					$error_message .= __( '.htaccess file lacks WordPress rewrite rules.', 'search-anything-anywhere' );
					break;
			}
			Search_Anything_Anywhere::debug_log( 'permalink_notice', "Dismissal failed: $error_message" );
			wp_send_json_error( array( 'message' => $error_message ) );
		}

		delete_transient( 'search_anything_anywhere_flush_permalinks' );
		Search_Anything_Anywhere::debug_log( 'permalink_notice', 'Permalink notice dismissed' );
		wp_send_json_success();
	}

	/**
	 * Enqueue admin scripts for handling notice dismissal.
	 */
	public function enqueue_admin_scripts() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$needs_scripts = get_transient( 'search_anything_anywhere_activation_notice' ) || get_transient( 'search_anything_anywhere_flush_permalinks' );
		if ( ! $needs_scripts ) {
			return;
		}

		wp_enqueue_script(
			'search-anything-anywhere-notice',
			plugins_url( '../assets/js/notice.js', __FILE__ ),
			array( 'jquery' ),
			SEARCH_ANYTHING_ANYWHERE_VERSION,
			true
		);
		wp_localize_script(
			'search-anything-anywhere-notice',
			'searchAnythingAnywhereNotice',
			array(
				'ajax_url'         => admin_url( 'admin-ajax.php' ),
				'activation_nonce' => wp_create_nonce( 'search_anything_anywhere_dismiss_activation_notice' ),
				'permalink_nonce'  => wp_create_nonce( 'search_anything_anywhere_dismiss_permalink_notice' ),
				'debug'            => ! empty( get_option( 'search_anything_anywhere_miscellaneous_settings_settings', array() )['enable_debug_logging'] ),
			)
		);
		Search_Anything_Anywhere::debug_log( 'admin_scripts', 'Enqueued admin scripts for notice dismissal' );
	}
}
