<?php
/**
 * General settings for the Search Anything Anywhere plugin.
 */
class Search_Anything_Anywhere_Miscellaneous extends Search_Anything_Anywhere_Options_Base {
	/**
	 * Default settings for the miscellaneous options.
	 *
	 * @var array<string,mixed>
	 */
	private array $default_options = array(
		'homepage_post_types'       => array( 'post' ),
		'enable_cache_invalidation' => true,
		'cache_duration'            => HOUR_IN_SECONDS,
		'enable_debug_logging'      => false,
	);

	/**
	 * Constructor to initialize the class.
	 */
	public function __construct() {
		Search_Anything_Anywhere::debug_log( 'init', 'Search_Anything_Anywhere_Miscellaneous initialized' );
	}

	/**
	 * Get the default general options.
	 *
	 * @return array<string,mixed>
	 */
	public function get_default_options(): array {
		return $this->default_options;
	}

	/**
	 * Register the settings and fields for the miscellaneous options.
	 */
	public function register_settings(): void {
		add_settings_section(
			'search_anything_anywhere_miscellaneous_options',
			__( 'Miscellaneous Settings', 'search-anything-anywhere' ),
			function () {
				echo '<p class="description">' . esc_html__( 'Configure miscellaneous options.', 'search-anything-anywhere' ) . '</p>';
			},
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'search_anything_anywhere_homepage_post_types',
			__( 'Homepage Post Types', 'search-anything-anywhere' ),
			array( $this, 'render_homepage_post_types' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_miscellaneous_options'
		);

		add_settings_section(
			'search_anything_anywhere_cache_settings',
			__( 'Cache Settings', 'search-anything-anywhere' ),
			array( $this, 'render_cache_section_description' ),
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'enable_cache_invalidation',
			__( 'Enable Cache Invalidation', 'search-anything-anywhere' ),
			array( $this, 'render_cache_invalidation' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_cache_settings'
		);

		add_settings_field(
			'cache_duration',
			__( 'Cache Duration', 'search-anything-anywhere' ),
			array( $this, 'render_cache_duration' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_cache_settings'
		);

		add_settings_field(
			'clear_term_cache',
			__( 'Clear Term Cache', 'search-anything-anywhere' ),
			array( $this, 'render_clear_term_cache' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_cache_settings'
		);

		add_settings_section(
			'search_anything_anywhere_debug_settings',
			__( 'Debug Settings', 'search-anything-anywhere' ),
			function () {
				echo '<p class="description">' . esc_html__( 'Enable debug logging to help diagnose issues with the plugin.', 'search-anything-anywhere' ) . '</p>';
			},
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'search_anything_anywhere_enable_debug_logging',
			__( 'Enable Debug Logging', 'search-anything-anywhere' ),
			array( $this, 'render_debug_logging' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_debug_settings'
		);

		wp_enqueue_style(
			'search-anything-anywhere-options-css',
			plugins_url( '../assets/css/options.css', __FILE__ ),
			array(),
			SEARCH_ANYTHING_ANYWHERE_VERSION
		);

		wp_enqueue_script(
			'search-anything-anywhere-options-js',
			plugins_url( '../assets/js/options.js', __FILE__ ),
			array( 'jquery' ),
			SEARCH_ANYTHING_ANYWHERE_VERSION,
			true
		);

		wp_localize_script(
			'search-anything-anywhere-options-js',
			'searchAnythingAnywhereOptions',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'debug'    => ! empty( get_option( 'search_anything_anywhere_miscellaneous_options', array() )['enable_debug_logging'] ),
			)
		);

		Search_Anything_Anywhere::debug_log( 'settings', 'General settings registered' );
	}

	/**
	 * Render the homepage post types selection.
	 */
	public function render_homepage_post_types(): void {
		$options = get_option( 'search_anything_anywhere_miscellaneous_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'homepage_post_types option value: ' . json_encode( $options['homepage_post_types'] ) );
		$selected   = ! empty( $options['homepage_post_types'] ) ? $options['homepage_post_types'] : array();
		$post_types = get_post_types( array( 'public' => true ), 'objects' );
		echo '<select name="search_anything_anywhere_miscellaneous_options[homepage_post_types][]" multiple style="width: 300px;">';
		foreach ( $post_types as $post_type ) {
			$selected_attr = selected( in_array( $post_type->name, $selected, true ), true, false );
			?>
			<option value="<?php echo esc_attr( $post_type->name ); ?>" <?php echo wp_kses_post( $selected_attr ); ?>>
				<?php echo esc_html( $post_type->labels->name ); ?>
			</option>
			<?php
		}
		echo '</select>';
		echo '<p class="description">' . esc_html__( 'Select which post types to include in the homepage search results. This allows you to customize the content shown on your homepage search results.', 'search-anything-anywhere' ) . '</p>';
	}

	/**
	 * Render the cache section description.
	 */
	public function render_cache_section_description(): void {
		?>
		<p class="description"><?php esc_html_e( 'Cache settings control how term counts (e.g., number of posts per category or tag) are stored for search form filters. Term counts are cached to improve performance. When posts or terms (e.g., new tags, updated posts) are modified, the cache can be automatically cleared to refresh filters, or you can manually clear it below. The cache duration determines how long counts are stored before refreshing.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the cache invalidation option.
	 */
	public function render_cache_invalidation(): void {
		$options = get_option( 'search_anything_anywhere_miscellaneous_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'enable_cache_invalidation option value: ' . json_encode( $options['enable_cache_invalidation'] ) );
		$checked = checked( 1, $options['enable_cache_invalidation'], false );
		$name    = 'search_anything_anywhere_miscellaneous_options[enable_cache_invalidation]';
		$label   = esc_html__( 'Automatically clear term counts cache when posts or terms are updated.', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, the term counts cache will be automatically cleared whenever posts or taxonomy terms are created, updated, or deleted, ensuring that filters always show accurate counts.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the cache duration selection.
	 */
	public function render_cache_duration(): void {
		$options = get_option( 'search_anything_anywhere_miscellaneous_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'cache_duration option value: ' . json_encode( $options['cache_duration'] ) );
		$duration = ! empty( $options['cache_duration'] ) ? (int) $options['cache_duration'] : HOUR_IN_SECONDS;
		?>
		<select name="search_anything_anywhere_miscellaneous_options[cache_duration]">
			<option value="<?php echo esc_attr( HOUR_IN_SECONDS ); ?>" <?php selected( $duration, HOUR_IN_SECONDS ); ?>><?php esc_html_e( '1 Hour', 'search-anything-anywhere' ); ?></option>
			<option value="<?php echo esc_attr( 12 * HOUR_IN_SECONDS ); ?>" <?php selected( $duration, 12 * HOUR_IN_SECONDS ); ?>><?php esc_html_e( '12 Hours', 'search-anything-anywhere' ); ?></option>
			<option value="<?php echo esc_attr( DAY_IN_SECONDS ); ?>" <?php selected( $duration, DAY_IN_SECONDS ); ?>><?php esc_html_e( '1 Day', 'search-anything-anywhere' ); ?></option>
			<option value="<?php echo esc_attr( WEEK_IN_SECONDS ); ?>" <?php selected( $duration, WEEK_IN_SECONDS ); ?>><?php esc_html_e( '1 Week', 'search-anything-anywhere' ); ?></option>
		</select>
		<p class="description"><?php esc_html_e( 'Set how long term counts are cached before being refreshed. Shorter durations ensure more up-to-date counts but may impact performance.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the button to clear the term cache.
	 */
	public function render_clear_term_cache() {
		$nonce = wp_create_nonce( 'search_anything_anywhere_clear_term_cache' );
		?>
		<button id="search-anything-anywhere-clear-term-cache" class="button" data-nonce="<?php echo esc_attr( $nonce ); ?>"><?php esc_html_e( 'Clear Cache Now', 'search-anything-anywhere' ); ?></button>
		<p id="search-anything-anywhere-clear-term-cache-status"></p>
		<p class="description"><?php esc_html_e( 'Manually clear the term counts cache. Useful if you have disabled automatic cache invalidation.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the debug logging option.
	 */
	public function render_debug_logging(): void {
		$options = get_option( 'search_anything_anywhere_miscellaneous_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'enable_debug_logging option value: ' . json_encode( $options['enable_debug_logging'] ) );
		$checked = checked( 1, $options['enable_debug_logging'], false );
		$name    = 'search_anything_anywhere_miscellaneous_options[enable_debug_logging]';
		$label   = esc_html__( 'Enable Debug Logging', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Enables detailed debug logging to help diagnose issues with the plugin. Logs are written to wp-content/debug.log. Requires WP_DEBUG and WP_DEBUG_LOG to be enabled in wp-config.php.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Sanitize and validate options before saving.
	 *
	 * @param array<string,mixed> $input The input options to sanitize.
	 * @return array<string,mixed> The sanitized options.
	 */
	public function sanitize_options( $input ): array {
		$current_options = get_option( 'search_anything_anywhere_miscellaneous_options', $this->default_options );
		$sanitized       = $current_options;

		if ( is_array( $input ) ) {
			foreach ( array( 'enable_cache_invalidation', 'enable_debug_logging' ) as $key ) {
				$sanitized[ $key ] = isset( $input[ $key ] ) && '1' === $input[ $key ] ? true : false;
			}

			if ( isset( $input['homepage_post_types'] ) && is_array( $input['homepage_post_types'] ) ) {
				$valid_post_types                 = get_post_types( array( 'public' => true ) );
				$sanitized['homepage_post_types'] = array_filter(
					$input['homepage_post_types'],
					function ( $post_type ) use ( $valid_post_types ) {
						return in_array( $post_type, $valid_post_types, true );
					}
				);
			}

			if ( isset( $input['cache_duration'] ) && is_numeric( $input['cache_duration'] ) && $input['cache_duration'] > 0 ) {
				$sanitized['cache_duration'] = (int) $input['cache_duration'];
			} else {
				$sanitized['cache_duration'] = HOUR_IN_SECONDS;
			}
		}

		Search_Anything_Anywhere::debug_log( 'sanitize', 'Sanitized options: ' . json_encode( $sanitized ) );
		return $sanitized;
	}

	/**
	 * Reset options to their default values.
	 */
	public function reset_options(): void {
		global $wp_settings_fields;

		Search_Anything_Anywhere::debug_log( 'reset', 'Resetting Miscellaneous options to: ' . json_encode( $this->default_options ) );

		// Temporarily remove sanitization callback to prevent it from interfering with reset
		$current_settings           = $wp_settings_fields['search_anything_anywhere_options']['search_anything_anywhere_miscellaneous_options'] ?? array();
		$original_sanitize_callback = $current_settings['args']['sanitize_callback'] ?? null;
		unregister_setting( 'search_anything_anywhere_miscellaneous_options', 'search_anything_anywhere_miscellaneous_options' );

		// Perform the reset without sanitization
		update_option( 'search_anything_anywhere_miscellaneous_options', $this->default_options );

		// Restore the sanitization callback
		register_setting(
			'search_anything_anywhere_miscellaneous_options',
			'search_anything_anywhere_miscellaneous_options',
			array( 'sanitize_callback' => $original_sanitize_callback )
		);

		Search_Anything_Anywhere::debug_log( 'reset', 'Miscellaneous options reset completed, current value: ' . json_encode( get_option( 'search_anything_anywhere_miscellaneous_options' ) ) );
	}
}
