<?php
/**
 * General settings for the Search Anything Anywhere plugin.
 */
class Search_Anything_Anywhere_General extends Search_Anything_Anywhere_Options_Base {

	/**
	 * Default settings.
	 *
	 * @var array<string,mixed>
	 */
	private array $default_options = array(
		'search_acf'            => false,
		'search_taxonomies'     => false,
		'show_taxonomy_filters' => true,
		'enable_friendly_urls'  => true,
		'replace_core_search'   => true,
	);

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'update_option_search_anything_anywhere_general_options', array( $this, 'handle_enable_friendly_urls_toggle' ), 10, 2 );
		Search_Anything_Anywhere::debug_log( 'init', 'Search_Anything_Anywhere_General initialized' );
	}

	/**
	 * Get the default general options.
	 *
	 * @return array<string,mixed>
	 */
	public function get_default_options(): array {
		return $this->default_options;
	}

	/**
	 * Register settings, sections, and fields.
	 */
	public function register_settings(): void {
		add_settings_section(
			'search_anything_anywhere_search_settings',
			__( 'Search Settings', 'search-anything-anywhere' ),
			function () {
				echo '<p class="description">' . esc_html__( 'Configure how the search functionality behaves.', 'search-anything-anywhere' ) . '</p>';
			},
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'search_anything_anywhere_search_acf',
			__( 'Advanced Custom Fields', 'search-anything-anywhere' ),
			array( $this, 'render_search_acf' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_search_settings'
		);

		add_settings_field(
			'search_anything_anywhere_search_taxonomies',
			__( 'Taxonomies', 'search-anything-anywhere' ),
			array( $this, 'render_search_taxonomies' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_search_settings'
		);

		add_settings_field(
			'search_anything_anywhere_show_taxonomy_filters',
			__( 'Show Taxonomy Filters', 'search-anything-anywhere' ),
			array( $this, 'render_show_taxonomy_filters' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_search_settings'
		);

		add_settings_section(
			'search_anything_anywhere_general_options',
			__( 'General Settings', 'search-anything-anywhere' ),
			function () {
				echo '<p class="description">' . esc_html__( 'Configure general settings.', 'search-anything-anywhere' ) . '</p>';
			},
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'search_anything_anywhere_enable_friendly_urls',
			__( 'Enable Friendly URLs', 'search-anything-anywhere' ),
			array( $this, 'render_enable_friendly_urls' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_general_options'
		);

		add_settings_field(
			'search_anything_anywhere_replace_core_search',
			__( 'WordPress Search Form', 'search-anything-anywhere' ),
			array( $this, 'render_replace_core_search' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_general_options'
		);

		wp_enqueue_style(
			'search-anything-anywhere-options-css',
			plugins_url( '../assets/css/options.css', __FILE__ ),
			array(),
			SEARCH_ANYTHING_ANYWHERE_VERSION
		);

		Search_Anything_Anywhere::debug_log( 'settings', 'General settings registered' );
	}

	/**
	 * Render the "Search inside Advanced Custom Fields" checkbox.
	 */
	public function render_search_acf(): void {
		$options = get_option( 'search_anything_anywhere_general_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'search_acf option value: ' . json_encode( $options['search_acf'] ) );
		$checked = checked( 1, $options['search_acf'], false );
		$name    = 'search_anything_anywhere_general_options[search_acf]';
		$label   = esc_html__( 'Search inside Advanced Custom Fields', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Includes posts in results if the search term matches inside ACF fields.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the "Search inside Taxonomies" checkbox.
	 */
	public function render_search_taxonomies(): void {
		$options = get_option( 'search_anything_anywhere_general_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'search_taxonomies option value: ' . json_encode( $options['search_taxonomies'] ) );
		$checked = checked( 1, $options['search_taxonomies'], false );
		$name    = 'search_anything_anywhere_general_options[search_taxonomies]';
		$label   = esc_html__( 'Search inside Taxonomies', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Includes posts in results if the search term matches their taxonomy terms (e.g., categories or tags).', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the "Show Taxonomy Filters" checkbox.
	 */
	public function render_show_taxonomy_filters(): void {
		$options = get_option( 'search_anything_anywhere_general_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'show_taxonomy_filters option value: ' . json_encode( $options['show_taxonomy_filters'] ) );
		$checked = checked( 1, $options['show_taxonomy_filters'], false );
		$name    = 'search_anything_anywhere_general_options[show_taxonomy_filters]';
		$label   = esc_html__( 'Show taxonomy filters', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Displays taxonomy filters to allow users to refine their search by categories, tags, or other taxonomies. Customize the list of taxonomies in the "Taxonomies" tab.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the "Enable Friendly URLs" checkbox.
	 */
	public function render_enable_friendly_urls(): void {
		$options = get_option( 'search_anything_anywhere_general_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'enable_friendly_urls option value: ' . json_encode( $options['enable_friendly_urls'] ) );
		$checked = checked( 1, $options['enable_friendly_urls'], false );
		$name    = 'search_anything_anywhere_general_options[enable_friendly_urls]';
		$label   = esc_html__( 'Enable friendly URLs (e.g., /search/term/action)', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Use clean, SEO-friendly search URLs. After enabling, go to Settings > Permalinks and click Save to refresh permalinks.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the "Replace Core Search Form" checkbox.
	 */
	public function render_replace_core_search(): void {
		$options = get_option( 'search_anything_anywhere_general_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'replace_core_search option value: ' . json_encode( $options['replace_core_search'] ) );
		$checked = checked( 1, $options['replace_core_search'], false );
		$name    = 'search_anything_anywhere_general_options[replace_core_search]';
		$label   = esc_html__( 'Replace WordPress Search Form with Search Anything Anywhere Form', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Automatically replaces the core WordPress Search block with the Search Anything Anywhere block.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Handle toggling of the "Enable Friendly URLs" option.
	 *
	 * @param array<string,mixed> $old_value The old option values.
	 * @param array<string,mixed> $new_value The new option values.
	 */
	public function handle_enable_friendly_urls_toggle( $old_value, $new_value ): void {
		if ( ! isset( $old_value['enable_friendly_urls'] ) || ! isset( $new_value['enable_friendly_urls'] ) ) {
			return;
		}
		if ( $new_value['enable_friendly_urls'] && ! $old_value['enable_friendly_urls'] ) {
			set_transient( 'search_anything_anywhere_flush_permalinks', true, DAY_IN_SECONDS );
		} elseif ( ! $new_value['enable_friendly_urls'] && $old_value['enable_friendly_urls'] ) {
			delete_transient( 'search_anything_anywhere_flush_permalinks' );
		}
	}

	/**
	 * Sanitize and validate options before saving.
	 *
	 * @param array<string,mixed> $input The input options to sanitize.
	 * @return array<string,mixed> The sanitized options.
	 */
	public function sanitize_options( $input ): array {
		$current_options = get_option( 'search_anything_anywhere_general_options', $this->default_options );
		$sanitized       = $current_options;

		if ( is_array( $input ) ) {
			foreach ( array( 'search_acf', 'search_taxonomies', 'show_taxonomy_filters', 'enable_friendly_urls', 'replace_core_search' ) as $key ) {
				$sanitized[ $key ] = isset( $input[ $key ] ) && '1' === $input[ $key ] ? true : false;
			}
		}

		Search_Anything_Anywhere::debug_log( 'sanitize', 'Sanitized options: ' . json_encode( $sanitized ) );
		return $sanitized;
	}

	/**
	 * Reset options to their default values.
	 */
	public function reset_options(): void {
		global $wp_settings_fields;

		Search_Anything_Anywhere::debug_log( 'reset', 'Resetting General options to: ' . json_encode( $this->get_default_options() ) );

		// Temporarily remove sanitization callback to prevent it from interfering with reset
		$current_settings           = $wp_settings_fields['search_anything_anywhere_options']['search_anything_anywhere_general_options'] ?? array();
		$original_sanitize_callback = $current_settings['args']['sanitize_callback'] ?? null;
		unregister_setting( 'search_anything_anywhere_general_options', 'search_anything_anywhere_general_options' );

		// Perform the reset without sanitization
		update_option( 'search_anything_anywhere_general_options', $this->get_default_options() );

		// Restore the sanitization callback
		register_setting(
			'search_anything_anywhere_general_options',
			'search_anything_anywhere_general_options',
			array( 'sanitize_callback' => $original_sanitize_callback )
		);

		Search_Anything_Anywhere::debug_log( 'reset', 'General options reset completed, current value: ' . json_encode( get_option( 'search_anything_anywhere_general_options' ) ) );
	}
}
