<?php
/**
 * General settings for the Search Anything Anywhere plugin.
 */
class Search_Anything_Anywhere_Display extends Search_Anything_Anywhere_Options_Base {

	/**
	 * Default settings for the display options.
	 *
	 * @var array<string,mixed>
	 */
	private array $default_options = array(
		'show_term_counts'          => true,
		'context_aware_placeholder' => true,
		'exclude_no_featured_image' => false,
		'taxonomy_filter_sort'      => 'count',
		'search_result_sort'        => 'relevance',
	);

	/**
	 * Constructor.
	 */
	public function __construct() {
		Search_Anything_Anywhere::debug_log( 'init', 'Search_Anything_Anywhere_Display initialized' );
	}

	/**
	 * Get the default general options.
	 *
	 * @return array<string,mixed>
	 */
	public function get_default_options(): array {
		return $this->default_options;
	}

	/**
	 * Register the settings and fields for the display options.
	 */
	public function register_settings(): void {
		add_settings_section(
			'search_anything_anywhere_display_options',
			__( 'Display Settings', 'search-anything-anywhere' ),
			function () {
				echo '<p class="description">' . esc_html__( 'Customize how the search form and results are displayed.', 'search-anything-anywhere' ) . '</p>';
			},
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'search_anything_anywhere_show_term_counts',
			__( 'Show Term Counts', 'search-anything-anywhere' ),
			array( $this, 'render_show_term_counts' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_display_options'
		);

		add_settings_field(
			'search_anything_anywhere_context_aware_placeholder',
			__( 'Context-Aware Placeholder', 'search-anything-anywhere' ),
			array( $this, 'render_context_aware_placeholder' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_display_options'
		);

		add_settings_field(
			'search_anything_anywhere_exclude_no_featured_image',
			__( 'Exclude Posts Without Featured Images', 'search-anything-anywhere' ),
			array( $this, 'render_exclude_no_featured_image' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_display_options'
		);

		add_settings_section(
			'search_anything_anywhere_sorting_settings',
			__( 'Sorting Settings', 'search-anything-anywhere' ),
			function () {
				echo '<p class="description">' . esc_html__( 'Configure how taxonomy filters and search results are sorted.', 'search-anything-anywhere' ) . '</p>';
			},
			'search_anything_anywhere_options'
		);

		add_settings_field(
			'search_anything_anywhere_taxonomy_filter_sort',
			__( 'Taxonomy Filter Sorting', 'search-anything-anywhere' ),
			array( $this, 'render_taxonomy_filter_sort' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_sorting_settings'
		);

		add_settings_field(
			'search_anything_anywhere_search_result_sort',
			__( 'Search Result Sorting', 'search-anything-anywhere' ),
			array( $this, 'render_search_result_sort' ),
			'search_anything_anywhere_options',
			'search_anything_anywhere_sorting_settings'
		);

		wp_enqueue_style(
			'search-anything-anywhere-options-css',
			plugins_url( '../assets/css/options.css', __FILE__ ),
			array(),
			SEARCH_ANYTHING_ANYWHERE_VERSION
		);

		Search_Anything_Anywhere::debug_log( 'settings', 'General settings registered' );
	}

	/**
	 * Render the show term counts checkbox.
	 */
	public function render_show_term_counts(): void {
		$options = get_option( 'search_anything_anywhere_display_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'show_term_counts option value: ' . json_encode( $options['show_term_counts'] ) );
		$checked = checked( 1, $options['show_term_counts'], false );
		$name    = 'search_anything_anywhere_display_options[show_term_counts]';
		$label   = esc_html__( 'Show term counts in taxonomy filters', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Displays the number of posts associated with each term in taxonomy filters.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

		/**
	 * Render the context-aware placeholder checkbox.
	 */
	public function render_context_aware_placeholder(): void {
		$options = get_option( 'search_anything_anywhere_display_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'context_aware_placeholder option value: ' . json_encode( $options['context_aware_placeholder'] ) );
		$checked = checked( 1, $options['context_aware_placeholder'], false );
		$name    = 'search_anything_anywhere_display_options[context_aware_placeholder]';
		$label   = esc_html__( 'Use context-aware search placeholder', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Dynamically displays the search input placeholder text based on the current page context (e.g., "Search Products" on product pages).', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the exclude posts without featured images checkbox.
	 */
	public function render_exclude_no_featured_image(): void {
		$options = get_option( 'search_anything_anywhere_display_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'exclude_no_featured_image option value: ' . json_encode( $options['exclude_no_featured_image'] ) );
		$checked = checked( 1, $options['exclude_no_featured_image'], false );
		$name    = 'search_anything_anywhere_display_options[exclude_no_featured_image]';
		$label   = esc_html__( 'Exclude posts without featured images', 'search-anything-anywhere' );
		?>
		<input type='hidden' name='<?php echo wp_kses_post( $name ); ?>' value='0'>
		<label>
			<input type='checkbox' name='<?php echo wp_kses_post( $name ); ?>' value='1' <?php echo wp_kses_post( $checked ); ?>>
			<?php echo wp_kses_post( $label ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'Excludes posts that do not have a featured image from search results.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the taxonomy filter sort options.
	 */
	public function render_taxonomy_filter_sort(): void {
		$options = get_option( 'search_anything_anywhere_display_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'taxonomy_filter_sort option value: ' . json_encode( $options['taxonomy_filter_sort'] ) );
		$sort = $options['taxonomy_filter_sort'] ?? 'name';
		?>
		<p>
			<label><input type="radio" name="search_anything_anywhere_display_options[taxonomy_filter_sort]" value="name" <?php checked( $sort, 'name' ); ?>> <?php esc_html_e( 'Alphabetical', 'search-anything-anywhere' ); ?></label><br>
			<label><input type="radio" name="search_anything_anywhere_display_options[taxonomy_filter_sort]" value="count" <?php checked( $sort, 'count' ); ?>> <?php esc_html_e( 'By Count (Highest First)', 'search-anything-anywhere' ); ?></label>
		</p>
		<p class="description"><?php esc_html_e( 'Choose how to sort terms in taxonomy filters.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Render the search result sort options.
	 */
	public function render_search_result_sort(): void {
		$options = get_option( 'search_anything_anywhere_display_options', $this->default_options );
		Search_Anything_Anywhere::debug_log( 'render', 'search_result_sort option value: ' . json_encode( $options['search_result_sort'] ) );
		$sort = $options['search_result_sort'] ?? 'relevance';
		?>
		<select name="search_anything_anywhere_display_options[search_result_sort]">
			<option value="relevance" <?php selected( $sort, 'relevance' ); ?>><?php esc_html_e( 'Relevance (Default)', 'search-anything-anywhere' ); ?></option>
			<option value="date_newest" <?php selected( $sort, 'date_newest' ); ?>><?php esc_html_e( 'Date (Newest First)', 'search-anything-anywhere' ); ?></option>
			<option value="date_oldest" <?php selected( $sort, 'date_oldest' ); ?>><?php esc_html_e( 'Date (Oldest First)', 'search-anything-anywhere' ); ?></option>
			<option value="title_asc" <?php selected( $sort, 'title_asc' ); ?>><?php esc_html_e( 'Title (A-Z)', 'search-anything-anywhere' ); ?></option>
			<option value="title_desc" <?php selected( $sort, 'title_desc' ); ?>><?php esc_html_e( 'Title (Z-A)', 'search-anything-anywhere' ); ?></option>
		</select>
		<p class="description"><?php esc_html_e( 'Select the sorting order for search results.', 'search-anything-anywhere' ); ?></p>
		<?php
	}

	/**
	 * Sanitize and validate options before saving.
	 *
	 * @param array<string,mixed> $input The input options to sanitize.
	 * @return array<string,mixed> The sanitized options.
	 */
	public function sanitize_options( $input ): array {
		$current_options = get_option( 'search_anything_anywhere_display_options', $this->default_options );
		$sanitized       = $current_options;

		if ( is_array( $input ) ) {
			foreach ( array( 'show_term_counts', 'context_aware_placeholder', 'exclude_no_featured_image' ) as $key ) {
				$sanitized[ $key ] = isset( $input[ $key ] ) && '1' === $input[ $key ] ? true : false;
			}

			if ( isset( $input['taxonomy_filter_sort'] ) && in_array( $input['taxonomy_filter_sort'], array( 'name', 'count' ), true ) ) {
				$sanitized['taxonomy_filter_sort'] = $input['taxonomy_filter_sort'];
			}

			if ( isset( $input['search_result_sort'] ) && in_array( $input['search_result_sort'], array( 'relevance', 'date_newest', 'date_oldest', 'title_asc', 'title_desc' ), true ) ) {
				$sanitized['search_result_sort'] = $input['search_result_sort'];
			}
		}

		Search_Anything_Anywhere::debug_log( 'sanitize', 'Sanitized options: ' . json_encode( $sanitized ) );
		return $sanitized;
	}

	/**
	 * Reset options to their default values.
	 */
	public function reset_options(): void {
		global $wp_settings_fields;

		Search_Anything_Anywhere::debug_log( 'reset', 'Resetting Display options to: ' . json_encode( $this->default_options ) );

		// Temporarily remove sanitization callback to prevent it from interfering with reset
		$current_settings           = $wp_settings_fields['search_anything_anywhere_options']['search_anything_anywhere_display_options'] ?? array();
		$original_sanitize_callback = $current_settings['args']['sanitize_callback'] ?? null;
		unregister_setting( 'search_anything_anywhere_display_options', 'search_anything_anywhere_display_options' );

		// Perform the reset without sanitization
		update_option( 'search_anything_anywhere_display_options', $this->default_options );

		// Restore the sanitization callback
		register_setting(
			'search_anything_anywhere_display_options',
			'search_anything_anywhere_display_options',
			array( 'sanitize_callback' => $original_sanitize_callback )
		);

		Search_Anything_Anywhere::debug_log( 'reset', 'Display options reset completed, current value: ' . json_encode( get_option( 'search_anything_anywhere_display_options' ) ) );
	}
}
