<?php

/**
 * Extend WordPress search to handle search results
 *
 * @package Search_Anything_Anywhere
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Search_Anything_Anywhere_Search_Results {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'pre_get_posts', array( $this, 'initialize_search_query' ) );
		Search_Anything_Anywhere::debug_log( 'init', 'Search_Anything_Anywhere_Search_Results initialized' );
	}

	/**
	 * Initialize the search query with post type, taxonomies, and sorting.
	 */
	public function initialize_search_query( $query ) {
		if ( is_admin() || ! $query->is_main_query() ) {
			return;
		}

		$this->set_search_post_type( $query );
		$this->apply_taxonomy_filters( $query );
		$this->apply_search_result_sort( $query );
		$this->handle_pagination( $query );
	}

	/**
	 * Set the post type for the search query.
	 */
	private function set_search_post_type( $query ) {
		if ( get_query_var( 'search_anything_anywhere_search' ) ) {
			if ( $query->get( 's' ) === 'any' ) {
				$query->set( 's', '' );
			}

			$post_type = get_query_var( 'post_type' );
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only search query parameter, no state change.
			if ( ! empty( $post_type ) && post_type_exists( $post_type ) ) {
				$query->set( 'post_type', $post_type );
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only search query parameter, no state change.
			} elseif ( ! empty( $_GET['post_type'] ) && post_type_exists( sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) ) ) {
				$query->set( 'post_type', sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) );
				Search_Anything_Anywhere::debug_log( 'search_post_type', 'Set post type to: ' . sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) );
			}
		}
	}

	/**
	 * Apply taxonomy filters to the search query.
	 */
	private function apply_taxonomy_filters( $query ) {
		if ( ! $query->is_search() ) {
			return;
		}

		$general_options  = get_option(
			'search_anything_anywhere_general_options',
			( new Search_Anything_Anywhere_General() )->get_default_options()
		);
		$taxonomy_filters = get_option( 'search_anything_anywhere_taxonomy_filters', array() );
		$tax_query        = array();

		// Handle friendly URL taxonomies
		if ( get_query_var( 'search_anything_anywhere_search' ) ) {
			$taxonomies = get_query_var( 'search_anything_anywhere_taxonomies' );
			if ( $taxonomies ) {
				$taxonomy_priority = get_option(
					'search_anything_anywhere_taxonomy_priority',
					( new Search_Anything_Anywhere_Taxonomies() )->get_default_taxonomy_priority()
				);
				$priority          = ! empty( $taxonomy_priority ) ? $taxonomy_priority : array_keys( $taxonomy_filters );
				$terms_by_taxonomy = array();
				$assigned_terms    = array();

				$term_slugs = explode( '+', $taxonomies );
				foreach ( $term_slugs as $term_slug ) {
					if ( in_array( $term_slug, $assigned_terms, true ) ) {
						continue;
					}
					foreach ( $priority as $taxonomy ) {
						if ( isset( $taxonomy_filters[ $taxonomy ] ) && $taxonomy_filters[ $taxonomy ] && term_exists( $term_slug, $taxonomy ) ) {
							$terms_by_taxonomy[ $taxonomy ][] = $term_slug;
							$assigned_terms[]                 = $term_slug;
							Search_Anything_Anywhere::debug_log( 'taxonomy_filters', "Assigned term '$term_slug' to taxonomy '$taxonomy'" );
							break;
						}
					}
				}

				$tax_query = array_merge( $tax_query, $this->build_tax_query( $terms_by_taxonomy ) );
			}
		}

		// Handle dropdown taxonomy filters
		if ( ! empty( $general_options['show_taxonomy_filters'] ) ) {
			$dropdown_terms = array();
			foreach ( $taxonomy_filters as $taxonomy => $enabled ) {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only search query parameter, no state change.
				if ( $enabled && isset( $_GET[ 'taxonomy_' . $taxonomy ] ) && ! empty( $_GET[ 'taxonomy_' . $taxonomy ] ) ) {
                    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only search query parameter, no state change.
					$terms                       = is_array( $_GET[ 'taxonomy_' . $taxonomy ] )
						? array_map( 'sanitize_text_field', wp_unslash( $_GET[ 'taxonomy_' . $taxonomy ] ) )
						: array( sanitize_text_field( wp_unslash( $_GET[ 'taxonomy_' . $taxonomy ] ) ) );
					$dropdown_terms[ $taxonomy ] = $terms;
					Search_Anything_Anywhere::debug_log( 'taxonomy_filters', "Applied dropdown filter for taxonomy '$taxonomy': " . json_encode( $terms ) );
				}
			}
			if ( ! empty( $dropdown_terms ) ) {
				$tax_query = array_merge( $tax_query, $this->build_tax_query( $dropdown_terms ) );
			}
		}

		if ( ! empty( $tax_query ) ) {
			$tax_query['relation'] = 'AND';
			$query->set( 'tax_query', $tax_query );
			Search_Anything_Anywhere::debug_log( 'taxonomy_filters', 'Applied tax query: ' . json_encode( $tax_query ) );
		}
	}

	/**
	 * Build tax query array from terms by taxonomy.
	 */
	private function build_tax_query( $terms_by_taxonomy ) {
		$tax_query = array();
		foreach ( $terms_by_taxonomy as $taxonomy => $terms ) {
			if ( ! empty( $terms ) ) {
				$tax_query[] = array(
					'taxonomy' => $taxonomy,
					'field'    => 'slug',
					'terms'    => $terms,
					'operator' => 'IN',
				);
			}
		}
		return $tax_query;
	}

	/**
	 * Apply sorting to search results.
	 */
	private function apply_search_result_sort( $query ) {
		$display_options = get_option( 'search_anything_anywhere_display_options', array( 'search_result_sort' => 'relevance' ) );

		switch ( $display_options['search_result_sort'] ) {
			case 'date_newest':
				$query->set( 'orderby', 'date' );
				$query->set( 'order', 'DESC' );
				Search_Anything_Anywhere::debug_log( 'search_sort', 'Sorting by date (newest first)' );
				break;
			case 'date_oldest':
				$query->set( 'orderby', 'date' );
				$query->set( 'order', 'ASC' );
				Search_Anything_Anywhere::debug_log( 'search_sort', 'Sorting by date (oldest first)' );
				break;
			case 'title_asc':
				$query->set( 'orderby', 'title' );
				$query->set( 'order', 'ASC' );
				Search_Anything_Anywhere::debug_log( 'search_sort', 'Sorting by title (A-Z)' );
				break;
			case 'title_desc':
				$query->set( 'orderby', 'title' );
				$query->set( 'order', 'DESC' );
				Search_Anything_Anywhere::debug_log( 'search_sort', 'Sorting by title (Z-A)' );
				break;
			default:
				Search_Anything_Anywhere::debug_log( 'search_sort', 'Sorting by relevance (default)' );
				break;
		}
	}

	/**
	 * Handle pagination for search results.
	 */
	private function handle_pagination( $query ) {
		if ( ! $query->is_search() ) {
			return;
		}

		$paged = max( 1, get_query_var( 'paged', 1 ) );

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only pagination parameter
		if ( empty( $paged ) && ! empty( $_GET['paged'] ) && is_numeric( $_GET['paged'] ) ) {
			$paged = max( 1, absint( $_GET['paged'] ) );
		}

		Search_Anything_Anywhere::debug_log( 'pagination', 'URL: ' . ( isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : 'unknown' ) . ', Received paged: ' . $paged . ', Query vars: ' . json_encode( get_query_var( 'paged' ) ) );
		$query->set( 'paged', $paged );
		Search_Anything_Anywhere::debug_log( 'pagination', 'Set paged to: ' . $paged );
	}
}
