<?php

/**
 * Register rewrite rules for friendly search URLs
 *
 * @package Search_Anything_Anywhere
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Search_Anything_Anywhere_Rewrite_Rules {

	/**
	 * Constructor.
	 */
	public function register() {
		add_action( 'init', array( $this, 'add_rewrite_rules' ) );
		add_filter( 'query_vars', array( $this, 'register_query_vars' ) );
		add_action( 'permalink_structure_changed', array( $this, 'clear_permalink_transient' ) );
		add_action( 'update_option_rewrite_rules', array( $this, 'clear_permalink_transient' ) );
		add_action( 'admin_init', array( $this, 'handle_permalink_flush' ) );
		Search_Anything_Anywhere::debug_log( 'init', 'Search_Anything_Anywhere_Rewrite_Rules registered' );
	}

	/**
	 * Adds rewrite rules for friendly search URLs.
	 */
	public function add_rewrite_rules() {
		$settings = get_option( 'search_anything_anywhere_general_options', array( 'enable_friendly_urls' => false ) );
		if ( empty( $settings['enable_friendly_urls'] ) ) {
			Search_Anything_Anywhere::debug_log( 'rewrite_rules', 'Friendly URLs disabled, skipping rewrite rules' );
			return;
		}

		// Register rewrite tags for query vars
		add_rewrite_tag( '%search_anything_anywhere_search%', '([^/]*)' );
		add_rewrite_tag( '%search_anything_anywhere_taxonomies%', '(.+?)' );
		add_rewrite_tag( '%post_type%', '([^/]+)' );
		add_rewrite_tag( '%paged%', '([0-9]+)' );

		// This rule matches URLs with a search term, post type, taxonomies, and pagination.
		add_rewrite_rule(
			'^search/([^/]+)/([^/]+)/(.+?)/page/([0-9]+)/?$',
			'index.php?s=$matches[1]&search_anything_anywhere_search=1&post_type=$matches[2]&search_anything_anywhere_taxonomies=$matches[3]&paged=$matches[4]',
			'top'
		);

		// This rule matches URLs with a search term, post type, and taxonomies without pagination.
		add_rewrite_rule(
			'^search/([^/]+)/([^/]+)/(.+?)/?$',
			'index.php?s=$matches[1]&search_anything_anywhere_search=1&post_type=$matches[2]&search_anything_anywhere_taxonomies=$matches[3]',
			'top'
		);

		// This rule matches URLs with a search term, post type, and pagination without taxonomies.
		add_rewrite_rule(
			'^search/([^/]+)/([^/]+)/page/([0-9]+)/?$',
			'index.php?s=$matches[1]&search_anything_anywhere_search=1&post_type=$matches[2]&paged=$matches[3]',
			'top'
		);

		// This rule matches URLs with a search term and post type without taxonomies or pagination.
		add_rewrite_rule(
			'^search/([^/]+)/([^/]+)/?$',
			'index.php?s=$matches[1]&search_anything_anywhere_search=1&post_type=$matches[2]',
			'top'
		);

		Search_Anything_Anywhere::debug_log( 'rewrite_rules', 'Added rewrite rules for friendly search URLs' );
	}

	/**
	 * Register custom query variables for friendly search URLs.
	 *
	 * @param array $vars Existing query variables.
	 * @return array Modified query variables.
	 */
	public function register_query_vars( $vars ) {
		$vars[] = 'search_anything_anywhere_search';
		$vars[] = 'search_anything_anywhere_taxonomies';
		$vars[] = 'post_type';
		$vars[] = 'paged';
		Search_Anything_Anywhere::debug_log( 'query_vars', 'Registered query vars: search_anything_anywhere_search, search_anything_anywhere_taxonomies, post_type, paged' );
		return $vars;
	}

	/**
	 * Clear the permalink transient when rewrite rules change.
	 */
	public function clear_permalink_transient() {
		delete_transient( 'search_anything_anywhere_flush_permalinks' );
		Search_Anything_Anywhere::debug_log( 'permalink_transient', 'Cleared search_anything_anywhere_flush_permalinks transient' );
	}

	/**
	 * Handle permalink flush on admin init.
	 *
	 * @return void
	 */
	public function handle_permalink_flush() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin action, checked with manage_options capability.
		if ( ! current_user_can( 'manage_options' ) || ! isset( $_GET['permalink_structure'] ) || ! isset( $_GET['submit'] ) ) {
			return;
		}

		if ( get_transient( 'search_anything_anywhere_flush_permalinks' ) ) {
			delete_transient( 'search_anything_anywhere_flush_permalinks' );
			Search_Anything_Anywhere::debug_log( 'permalink_flush', 'Handled permalink flush, transient cleared' );
		}
	}
}
