<?php

/**
 * Cache management for Search Anything Anywhere
 *
 * @package Search_Anything_Anywhere
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Search_Anything_Anywhere_Cache {

	/**
	 * Register cache management hooks.
	 */
	public function register() {
		$miscellaneous_options = get_option( 'search_anything_anywhere_miscellaneous_options', ( new Search_Anything_Anywhere_Miscellaneous() )->get_default_options() );
		if ( ! empty( $miscellaneous_options['enable_cache_invalidation'] ) ) {
			add_action( 'save_post', array( $this, 'clear_term_counts_transients' ), 10, 3 );
			add_action( 'edit_term', array( $this, 'clear_term_counts_transients' ), 10, 3 );
			add_action( 'delete_post', array( $this, 'clear_term_counts_transients' ), 10, 1 );
			add_action( 'delete_term', array( $this, 'clear_term_counts_transients' ), 10, 3 );
			Search_Anything_Anywhere::debug_log( 'term_counts', 'Cache invalidation enabled, transients will be cleared on post/term changes' );
		} else {
			Search_Anything_Anywhere::debug_log( 'term_counts', 'Cache invalidation disabled, no automatic transient clearing' );
		}
	}

	/**
	 * Get the cache duration in seconds.
	 *
	 * @return int Cache duration in seconds.
	 */
	public function get_cache_duration() {
		$miscellaneous_options = get_option( 'search_anything_anywhere_miscellaneous_options', ( new Search_Anything_Anywhere_Miscellaneous() )->get_default_options() );
		$duration              = ! empty( $miscellaneous_options['cache_duration'] ) ? (int) $miscellaneous_options['cache_duration'] : HOUR_IN_SECONDS;
		return max( MINUTE_IN_SECONDS, $duration ); // Ensure minimum 1 minute
	}

	/**
	 * Clear term counts transients when posts or terms are saved or deleted.
	 *
	 * @param int $post_id Post ID.
	 * @param WP_Post|null $post Post object, if available.
	 * @param int $term_id Term ID, if applicable.
	 */
	public function clear_term_counts_transients( $post_id = 0, $post = null, $term_id = 0 ) {
		if ( wp_using_ext_object_cache() ) {
			// Object cache (e.g., Redis, Memcached)
			$transient_keys = get_option( 'search_anything_anywhere_transient_keys', array() );
			if ( ! empty( $transient_keys ) ) {
				foreach ( $transient_keys as $key ) {
					// Deletes both _transient_ and _transient_timeout_
					delete_transient( $key );
				}
				update_option( 'search_anything_anywhere_transient_keys', array() );
				Search_Anything_Anywhere::debug_log( 'term_counts', 'Cleared object cache transients: ' . implode( ', ', $transient_keys ) );
			}
			// Clear cache group
			wp_cache_delete( 'search_anything_anywhere_term_counts', 'search_anything_anywhere_term_counts' );
		} else {
			// Database transients
			global $wpdb;
			// Clear _transient_ entries
			$prefix = '_transient_search_anything_anywhere_term_counts_%';
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Necessary for transient wildcard deletion
			$wpdb->query(
				$wpdb->prepare(
					"DELETE FROM $wpdb->options WHERE option_name LIKE %s",
					$prefix
				)
			);
			// Clear _transient_timeout_ entries
			$timeout_prefix = '_transient_timeout_search_anything_anywhere_term_counts_%';
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Necessary for transient wildcard deletion
			$wpdb->query(
				$wpdb->prepare(
					"DELETE FROM $wpdb->options WHERE option_name LIKE %s",
					$timeout_prefix
				)
			);
			Search_Anything_Anywhere::debug_log( 'term_counts', 'Cleared database transients and timeouts with prefixes: search_anything_anywhere_term_counts_, transient_timeout_search_anything_anywhere_term_counts_' );
		}
	}
}
