<?php
/**
 * Search Anything Anywhere Block
 *
 * @package Search_Anything_Anywhere
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class Search_Anything_Anywhere_Block {

	/**
	 * Register the block and its render callback.
	 */
	public function register() {
		add_action( 'init', array( $this, 'register_block' ) );
		add_filter( 'render_block_core/search', array( $this, 'render_core_search_block' ), 10, 2 );
	}

	/**
	 * Register the Search Anything Anywhere block.
	 *
	 * This block replaces the core search block with a custom search form.
	 */
	public function register_block() {
		// Register block script
		wp_register_script(
			'search-anything-anywhere-search-block',
			plugins_url( 'build/block-editor.js', dirname( __DIR__, 1 ) ),
			array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-components' ),
			SEARCH_ANYTHING_ANYWHERE_VERSION,
			true
		);

		// Localize block script
		wp_localize_script(
			'search-anything-anywhere-search-block',
			'searchAnythingAnywhereSettings',
			array(
				'replaceCoreSearch' => ! empty( get_option( 'search_anything_anywhere_general_options', ( new Search_Anything_Anywhere_General() )->get_default_options() )['replace_core_search'] ),
			)
		);

		// Register block style
		wp_register_style(
			'search-anything-anywhere-search-block-style',
			SEARCH_ANYTHING_ANYWHERE_URI . 'assets/css/style.css',
			array(),
			SEARCH_ANYTHING_ANYWHERE_VERSION
		);

		// Register the block type
		register_block_type(
			SEARCH_ANYTHING_ANYWHERE_PATH . 'blocks/search-anything-anywhere/block.json',
			array(
				'render_callback' => array( $this, 'render_search_anything_anywhere_search_block' ),
			)
		);
	}

	/**
	 * Get term counts for a specific taxonomy and post types.
	 *
	 * @param string $taxonomy Taxonomy name.
	 * @param array $post_types Array of post type names.
	 * @return array Associative array of term slugs and their counts.
	 */
	private function get_term_counts( $taxonomy, $post_types ) {
		$cache_key = 'search_anything_anywhere_term_counts_' . md5( $taxonomy . serialize( $post_types ) );
		$counts    = get_transient( $cache_key );

		if ( false !== $counts ) {
			Search_Anything_Anywhere::debug_log( 'term_counts', "Retrieved term counts for '$taxonomy' from cache" );
			return $counts;
		}

		$counts = array();

		if ( array( 'any' ) === $post_types ) {
			$terms = get_terms(
				array(
					'taxonomy'   => $taxonomy,
					'hide_empty' => true,
				)
			);

			if ( ! is_wp_error( $terms ) ) {
				foreach ( $terms as $term ) {
					$counts[ $term->slug ] = $term->count;
				}
			}
		} else {
			// Get all terms first
			$terms = get_terms(
				array(
					'taxonomy'   => $taxonomy,
					'hide_empty' => true,
					'fields'     => 'all',
				)
			);

			if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
				// Single WP_Query to get posts with any of the terms
				$query = new WP_Query(
					array(
						'post_type'      => $post_types,
						'post_status'    => 'publish',
						'tax_query'      => array(
							array(
								'taxonomy' => $taxonomy,
								'field'    => 'slug',
								'terms'    => wp_list_pluck( $terms, 'slug' ), // All term slugs
							),
						),
						'posts_per_page' => -1, // Get all posts to ensure all terms are covered
						'fields'         => 'ids', // Only get post IDs
						'no_found_rows'  => true, // Optimize by skipping total count
					)
				);

				if ( $query->have_posts() ) {
					// Collect all term slugs from the posts
					$valid_term_slugs = array();
					foreach ( $query->posts as $post_id ) {
						$post_terms = wp_get_post_terms( $post_id, $taxonomy, array( 'fields' => 'slugs' ) );
						if ( ! is_wp_error( $post_terms ) ) {
							$valid_term_slugs = array_merge( $valid_term_slugs, $post_terms );
						}
					}
					$valid_term_slugs = array_unique( $valid_term_slugs );

					// Use $term->count for terms that have posts
					foreach ( $terms as $term ) {
						if ( in_array( $term->slug, $valid_term_slugs, true ) ) {
							$counts[ $term->slug ] = $term->count;
						}
					}
				}
			}
		}

		// Cache the results
		set_transient( $cache_key, $counts, ( new Search_Anything_Anywhere_Cache() )->get_cache_duration() );

		// Track transient key for object cache
		if ( wp_using_ext_object_cache() ) {
			$transient_keys = get_option( 'search_anything_anywhere_transient_keys', array() );
			if ( ! in_array( $cache_key, $transient_keys, true ) ) {
				$transient_keys[] = $cache_key;
				update_option( 'search_anything_anywhere_transient_keys', $transient_keys );
			}
		}

		// Add to cache group
		wp_cache_set( $cache_key, $counts, 'search_anything_anywhere_term_counts', ( new Search_Anything_Anywhere_Cache() )->get_cache_duration() );
		Search_Anything_Anywhere::debug_log( 'term_counts', "Cached term counts for '$taxonomy'" );

		return $counts;
	}

	/**
	 * Render the Search Anything Anywhere search block.
	 *
	 * @param array $attributes Block attributes.
	 * @return string Rendered block HTML.
	 */
	public function render_core_search_block( $block_content, $block ): string {
		$options = get_option(
			'search_anything_anywhere_general_options',
			( new Search_Anything_Anywhere_General() )->get_default_options()
		);
		if ( empty( $options['replace_core_search'] ) ) {
			return $block_content;
		}

		Search_Anything_Anywhere::debug_log( 'render_block', 'Replacing core/search block with search-anything-anywhere/search-form' );

		$attributes = $block['attrs'] ?? array();
		return $this->render_search_anything_anywhere_search_block(
			array(
				'placeholder' => $attributes['placeholder'] ?? '',
			)
		);
	}

	/**
	 * Render the Search Anything Anywhere search block.
	 *
	 * @param array $attributes Block attributes.
	 * @return string Rendered block HTML.
	 */
	public function render_search_anything_anywhere_search_block( $attributes ) {
		$general_options       = get_option(
			'search_anything_anywhere_general_options',
			( new Search_Anything_Anywhere_General() )->get_default_options()
		);
		$display_options       = get_option(
			'search_anything_anywhere_display_options',
			( new Search_Anything_Anywhere_Display() )->get_default_options()
		);
		$taxonomy_filters      = get_option(
			'search_anything_anywhere_taxonomy_filters',
			( new Search_Anything_Anywhere_Taxonomies() )->get_default_taxonomy_filters()
		);
		$miscellaneous_options = get_option(
			'search_anything_anywhere_miscellaneous_options',
			( new Search_Anything_Anywhere_Miscellaneous() )->get_default_options()
		);

		// Determine current post type(s)
		$current_post_types = array( 'any' );
		if ( is_post_type_archive() ) {
			$current_post_types = array( get_queried_object()->name );
		} elseif ( is_home() ) {
			$current_post_types = ! empty( $miscellaneous_options['homepage_post_types'] ) ? $miscellaneous_options['homepage_post_types'] : array( 'post' );
		} elseif ( is_tax() || is_category() || is_tag() ) {
			$taxonomy           = get_queried_object()->taxonomy;
			$tax_obj            = get_taxonomy( $taxonomy );
			$current_post_types = $tax_obj->object_type;
		} elseif ( is_search() && ! empty( get_query_var( 'post_type' ) ) && post_type_exists( get_query_var( 'post_type' ) ) ) {
			$current_post_types = array( get_query_var( 'post_type' ) );
		}

		// Filter taxonomies by post type
		$allowed_taxonomies = array();
		if ( array( 'any' ) !== $current_post_types ) {
			$object_taxonomies = array();
			foreach ( $current_post_types as $post_type ) {
				$object_taxonomies = array_merge( $object_taxonomies, get_object_taxonomies( $post_type, 'names' ) );
			}
			$object_taxonomies = array_unique( $object_taxonomies );
			foreach ( $taxonomy_filters as $taxonomy => $enabled ) {
				if ( $enabled && in_array( $taxonomy, $object_taxonomies, true ) ) {
					$allowed_taxonomies[ $taxonomy ] = $enabled;
				}
			}
		} else {
			$allowed_taxonomies = $taxonomy_filters;
		}

		// Get current taxonomy term if on taxonomy page
		$current_term     = is_tax() || is_category() || is_tag() ? get_queried_object() : null;
		$current_taxonomy = $current_term ? $current_term->taxonomy : '';

		// Set placeholder
		$placeholder = ! empty( $attributes['placeholder'] ) ? $attributes['placeholder'] : 'Search...';
		if ( ! empty( $display_options['context_aware_placeholder'] ) && 1 === count( $current_post_types ) && 'any' !== $current_post_types[0] ) {
			$post_type_obj = get_post_type_object( $current_post_types[0] );
			// translators: %s is the post type name
			$placeholder = sprintf( __( 'Search %s...', 'search-anything-anywhere' ), strtolower( $post_type_obj->labels->name ) );
		}

		// Get context-aware term counts
		$term_counts = array();
		foreach ( $allowed_taxonomies as $taxonomy => $enabled ) {
			if ( $enabled ) {
				$term_counts[ $taxonomy ] = $this->get_term_counts( $taxonomy, $current_post_types );
			}
		}

		// Parse selected terms for friendly URLs
		$selected_terms = array();
		if ( ! empty( $general_options['enable_friendly_urls'] ) && get_query_var( 'search_anything_anywhere_taxonomies' ) ) {
			$taxonomies        = get_query_var( 'search_anything_anywhere_taxonomies' );
			$taxonomy_priority = get_option(
				'search_anything_anywhere_taxonomy_priority',
				( new Search_Anything_Anywhere_Taxonomies() )->get_default_taxonomy_priority()
			);
			$priority          = ! empty( $taxonomy_priority ) ? $taxonomy_priority : array_keys( $taxonomy_filters );
			$term_slugs        = explode( '+', $taxonomies );
			foreach ( $term_slugs as $term_slug ) {
				foreach ( $priority as $taxonomy ) {
					if ( isset( $allowed_taxonomies[ $taxonomy ] ) && term_exists( $term_slug, $taxonomy ) ) {
						$selected_terms[ $taxonomy ][] = $term_slug;
						break;
					}
				}
			}
		}

		// Set form action based on friendly URLs
		$form_action = home_url( '/' );
		if ( ! empty( $general_options['enable_friendly_urls'] ) ) {
			$form_action = home_url( '/search/' );
		}

		try {
			ob_start();
			$template_path = SEARCH_ANYTHING_ANYWHERE_PATH . 'blocks/template-search-block.php';
			if ( ! file_exists( $template_path ) ) {
				throw new Exception( "Template file not found: $template_path" );
			}
			include $template_path;
			$content = ob_get_clean();
			return false !== $content ? $content : '';
		} catch ( Exception $e ) {
			Search_Anything_Anywhere::debug_log( 'render_search_block', 'Template rendering failed: ' . $e->getMessage() );
			return '';
		}
	}
}
