<?php
/*
Plugin Name: Search Anything Anywhere
Plugin URI: https://blog.senthilprabu.in/portfolio/search-anything-anywhere
Description: Search and filter anything, anywhere — posts, fields, taxonomies, and more.
Version: 1.0.1
Author: Senthilprabu Ponnusamy
Author URI: https://www.senthilprabu.in
License: GPLv3 or later
License URI: https://www.gnu.org/licenses/gpl-3.0.html
*/

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Define URL & PATH Constants.
 */
define( 'SEARCH_ANYTHING_ANYWHERE_VERSION', '1.0.1' );
define( 'SEARCH_ANYTHING_ANYWHERE_URI', plugins_url( '/src/', __FILE__ ) );
define( 'SEARCH_ANYTHING_ANYWHERE_PATH', plugin_dir_path( __FILE__ ) . 'src/' );

/**
 * Load options classes.
 */
spl_autoload_register(
	function ( $class_name ) {
		if ( strpos( $class_name, 'Search_Anything_Anywhere_' ) === 0 ) {
			$file = SEARCH_ANYTHING_ANYWHERE_PATH . 'options/classes/class-' . strtolower( str_replace( '_', '-', $class_name ) ) . '.php';
			if ( file_exists( $file ) ) {
				require_once $file;
			}
		}
	}
);

/**
 * Load features classes.
 */
spl_autoload_register(
	function ( $class_name ) {
		if ( strpos( $class_name, 'Search_Anything_Anywhere_' ) === 0 ) {
			$file = SEARCH_ANYTHING_ANYWHERE_PATH . 'features/class-' . strtolower( str_replace( '_', '-', $class_name ) ) . '.php';
			if ( file_exists( $file ) ) {
				require_once $file;
			}
		}
	}
);

/**
 * Load blocks classes.
 */
spl_autoload_register(
	function ( $class_name ) {

		if ( strpos( $class_name, 'Search_Anything_Anywhere_' ) === 0 ) {

			$file = SEARCH_ANYTHING_ANYWHERE_PATH . 'blocks/class-' . strtolower( str_replace( '_', '-', $class_name ) ) . '.php';
			if ( file_exists( $file ) ) {
				require_once $file;
			}
		}
	}
);

/**
 * Load the main Search Anything Anywhere plugin class.
 */
require_once SEARCH_ANYTHING_ANYWHERE_PATH . 'class-search-anything-anywhere.php';


/**
 * Load the Search Anything Anywhere CLI class.
 */
require_once SEARCH_ANYTHING_ANYWHERE_PATH . 'class-search-anything-anywhere-cli.php';

/**
 * Set activation notice transient.
 */
register_activation_hook(
	__FILE__,
	function () {
		set_transient( 'search_anything_anywhere_activation_notice', true, 60 * 60 * 24 );
		Search_Anything_Anywhere::debug_log( 'activation', 'Plugin activated, set activation notice transient' );
	}
);

/**
 * Conditionally load the plugin functionalities based on settings.
 */
add_action(
	'plugins_loaded',
	function () {
		// Log plugin initialization
		Search_Anything_Anywhere::debug_log( 'init', 'Search Anything Anywhere plugin loaded' );

		/**
		 * Load the Search Anything Anywhere General Options.
		 * This is used to manage general settings for the plugin.
		 */
		new Search_Anything_Anywhere_Options();

		/**
		 * Load the Search Anything Anywhere Actions.
		 */
		if ( true ) {
			$notices = new Search_Anything_Anywhere_Actions();
			$notices->register();
		}

		/**
		 * Enqueue assets for the plugin.
		 */
		if ( true ) {
			$assets = new Search_Anything_Anywhere_Assets();
			$assets->enqueue();
		}

		/**
		 * Register the Search Anything Anywhere Search Block.
		 */
		if ( true ) {
			$block = new Search_Anything_Anywhere_Block();
			$block->register();
		}

		/**
		 * Load the Search Anything Anywhere Cache.
		 */
		if ( true ) {
			$cache = new Search_Anything_Anywhere_Cache();
			$cache->register();
		}

		/**
		 * Load the Search Anything Anywhere Notices.
		 */
		if ( true ) {
			$notices = new Search_Anything_Anywhere_Notices();
			$notices->register();
		}

		/**
		 * Initialize the Search Anything Anywhere Search Results.
		 */
		if ( true ) {
			new Search_Anything_Anywhere_Search_Results();
		}

		/**
		 * Add Settings action link on Plugins page.
		 */
		add_filter(
			'plugin_action_links_' . plugin_basename( __FILE__ ),
			function ( $links ) {
				$settings_url  = admin_url( 'admin.php?page=search-anything-anywhere-settings' );
				$settings_link = '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'search-anything-anywhere' ) . '</a>';
				array_unshift( $links, $settings_link );
				return $links;
			}
		);

		/** @var array<string,mixed> */
		$general_options = get_option(
			'search_anything_anywhere_general_options',
			( new Search_Anything_Anywhere_General() )->get_default_options()
		);

		/** @var array<string,mixed> */
		$display_options = get_option(
			'search_anything_anywhere_display_options',
			( new Search_Anything_Anywhere_Display() )->get_default_options()
		);

		/** @var array<string,mixed> */
		$miscellaneous_options = get_option(
			'search_anything_anywhere_miscellaneous_options',
			( new Search_Anything_Anywhere_Miscellaneous() )->get_default_options()
		);

		/**
		 * Initialize the Search Anything Anywhere Search Enhancements for ACF.
		 */
		if ( ! empty( $general_options['search_acf'] ) ) {
			new Search_Anything_Anywhere_ACF_Search();
		}

		/**
		 * Initialize the Search Anything Anywhere Taxonomy Search.
		 */
		if ( ! empty( $general_options['search_taxonomies'] ) ) {
			new Search_Anything_Anywhere_Taxonomy_Search();
		}

		/**
		 * Initialize the Search Anything Anywhere Rewrite Rules.
		 */
		if ( ! empty( $general_options['enable_friendly_urls'] ) ) {
			$rewrite_rules = new Search_Anything_Anywhere_Rewrite_Rules();
			$rewrite_rules->register();
		}

		/**
		 * Initialize the Search Anything Anywhere results filter.
		 */
		if ( ! empty( $display_options['exclude_no_featured_image'] ) ) {
			new Search_Anything_Anywhere_Results_Filter();
		}
	}
);
