<?php
/**
 * Plugin Name: Scholarships Q&A Search
 * Plugin URI: https://www.scholarshipsqanda.com/plugins/scholarships-qanda-search/
 * Description: A WordPress plugin that allows users to search scholarshipsqanda.com content from other scholarship websites and links back to the original source.
 * Version: 1.0.0
 * Author: Scholarships Q&A
 * Author URI: https://www.scholarshipsqanda.com/
 * Requires at least: 5.0
 * Text Domain: scholarships-qa-search
 * Domain Path: /languages
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 *
 * @package SCHOQASE_Search
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

// Define plugin constants
define( 'SCHOQASE_VERSION', '1.0.0' );
define( 'SCHOQASE_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SCHOQASE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'SCHOQASE_BASENAME', plugin_basename( __FILE__ ) );
define( 'SCHOQASE_DOMAIN', 'https://www.scholarshipsqanda.com' );

/**
 * The core plugin class
 */
class SCHOQASE_Plugin {

    /**
     * Instance of this class.
     *
     * @var object
     */
    protected static $instance = null;

    /**
     * Initialize the plugin.
     */
    private function __construct() {
        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     */
    private function load_dependencies() {
        // Include the core classes
        require_once SCHOQASE_PLUGIN_DIR . 'includes/class-schoqase-search.php';
        require_once SCHOQASE_PLUGIN_DIR . 'includes/class-schoqase-admin.php';
        require_once SCHOQASE_PLUGIN_DIR . 'includes/class-schoqase-widget.php';
        require_once SCHOQASE_PLUGIN_DIR . 'includes/class-schoqase-api.php';
        require_once SCHOQASE_PLUGIN_DIR . 'includes/class-schoqase-tracker.php';
    }

    /**
     * Set the locale for internationalization.
     */
    private function set_locale() {
        add_action( 'plugins_loaded', array( $this, 'load_plugin_textdomain' ) );
    }

    /**
     * Load the plugin text domain for translation.
     */
    public function load_plugin_textdomain() {
        load_plugin_textdomain(
            'scholarships-qa-search',
            false,
            dirname( plugin_basename( __FILE__ ) ) . '/languages/'
        );
    }

    /**
     * Register all of the hooks related to the admin area functionality.
     */
    private function define_admin_hooks() {
        $admin = new SCHOQASE_Admin();
        
        // Admin menu and settings
        add_action( 'admin_menu', array( $admin, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $admin, 'register_settings' ) );
        
        // Add settings link to plugins page
        add_filter( 'plugin_action_links_' . SCHOQASE_BASENAME, array( $admin, 'add_settings_link' ) );
    }

    /**
     * Register all of the hooks related to the public-facing functionality.
     */
    private function define_public_hooks() {
        // Enqueue scripts and styles
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_styles' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        
        // Register widget
        add_action( 'widgets_init', array( $this, 'register_widget' ) );
        
        // Register shortcode
        add_shortcode( 'schoqase_search', array( $this, 'search_shortcode' ) );
        
        // Register AJAX handler
        add_action( 'wp_ajax_schoqase_search_results', array( $this, 'get_search_results' ) );
        add_action( 'wp_ajax_nopriv_schoqase_search_results', array( $this, 'get_search_results' ) );
        
        // Register tracking endpoint
        add_action( 'wp_ajax_schoqase_track', array( $this, 'track_click' ) );
        add_action( 'wp_ajax_nopriv_schoqase_track', array( $this, 'track_click' ) );
    }

    /**
     * Enqueue styles.
     */
    public function enqueue_styles() {
        wp_enqueue_style(
            'scholarships-qa-search',
            SCHOQASE_PLUGIN_URL . 'assets/css/schoqase-search.css',
            array(),
            SCHOQASE_VERSION,
            'all'
        );
    }

    /**
     * Enqueue scripts.
     */
    public function enqueue_scripts() {
        wp_enqueue_script(
            'scholarships-qa-search',
            SCHOQASE_PLUGIN_URL . 'assets/js/schoqase-search.js',
            array( 'jquery' ),
            SCHOQASE_VERSION,
            true
        );
        
        wp_localize_script(
            'scholarships-qa-search',
            'schoqaseSearchParams',
            array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'schoqase-search-nonce' ),
            )
        );
    }

    /**
     * Register the widget.
     */
    public function register_widget() {
        register_widget( 'SCHOQASE_Widget' );
    }

    /**
     * Search shortcode callback.
     *
     * @param array $atts Shortcode attributes.
     * @return string Shortcode output.
     */
    public function search_shortcode( $atts ) {
        $atts = shortcode_atts(
            array(
                'title'        => __( 'Search Scholarships Q&A', 'scholarships-qa-search' ),
                'placeholder'  => __( 'Ask a scholarship question...', 'scholarships-qa-search' ),
                'button_text'  => __( 'Search', 'scholarships-qa-search' ),
                'results_num'  => 5,
            ),
            $atts,
            'schoqase_search'
        );
        
        ob_start();
        
        include SCHOQASE_PLUGIN_DIR . 'templates/search-form.php';
        
        return ob_get_clean();
    }

    /**
     * AJAX handler for search results.
     */
    public function get_search_results() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'schoqase-search-nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed', 'scholarships-qa-search' ) ) );
        }
        
        // Validate query
        if ( ! isset( $_POST['query'] ) || empty( $_POST['query'] ) ) {
            wp_send_json_error( array( 'message' => __( 'Please enter a search query', 'scholarships-qa-search' ) ) );
        }
        
        $query = sanitize_text_field( wp_unslash( $_POST['query'] ) );
        $limit = isset( $_POST['limit'] ) ? absint( $_POST['limit'] ) : 5;
        
        // Get search results
        $api = new SCHOQASE_API();
        $results = $api->search( $query, $limit );
        
        if ( is_wp_error( $results ) ) {
            wp_send_json_error( array( 'message' => $results->get_error_message() ) );
        }
        
        ob_start();
        include SCHOQASE_PLUGIN_DIR . 'templates/search-results.php';
        $html = ob_get_clean();
        
        wp_send_json_success( array(
            'html'    => $html,
            'results' => $results,
            'query'   => $query, // Pass the query back for highlighting
        ) );
    }

    /**
     * AJAX handler for tracking clicks.
     */
    public function track_click() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'schoqase-search-nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed', 'scholarships-qa-search' ) ) );
        }
        
        // Validate URL
        if ( ! isset( $_POST['url'] ) || empty( $_POST['url'] ) ) {
            wp_send_json_error( array( 'message' => __( 'Missing URL parameter', 'scholarships-qa-search' ) ) );
        }
        
        $url = esc_url_raw( wp_unslash( $_POST['url'] ) );
        $query = isset( $_POST['query'] ) ? sanitize_text_field( wp_unslash( $_POST['query'] ) ) : '';
        
        // Track click
        $tracker = new SCHOQASE_Tracker();
        $result = $tracker->track_click( $url, $query );
        
        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array( 'message' => $result->get_error_message() ) );
        }
        
        wp_send_json_success( array( 'message' => __( 'Click tracked successfully', 'scholarships-qa-search' ) ) );
    }

    /**
     * Return an instance of this class.
     *
     * @return object A single instance of this class.
     */
    public static function get_instance() {
        if ( null == self::$instance ) {
            self::$instance = new self;
        }
        
        return self::$instance;
    }
}

// Initialize the plugin
function schoqase_initialize_plugin() {
    return SCHOQASE_Plugin::get_instance();
}

schoqase_initialize_plugin();
