<?php
/**
 * Tracking functionality for analytics.
 *
 * @package SQandA_Search
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

/**
 * Tracker class.
 */
class SCHOQASE_Tracker {

    /**
     * The plugin options.
     *
     * @var array
     */
    private $options;

    /**
     * Initialize the class.
     */
    public function __construct() {
        $this->options = get_option( 'schoqase_search_options', array() );
    }

    /**
     * Track a click on a search result.
     *
     * @param string $url   The URL being clicked.
     * @param string $query The search query that led to this result.
     * @return bool|WP_Error True on success, WP_Error on failure.
     */
    public function track_click( $url, $query = '' ) {
        // Check if tracking is enabled
        if ( ! isset( $this->options['enable_tracking'] ) || ! $this->options['enable_tracking'] ) {
            return true; // Tracking disabled, pretend it succeeded
        }

        // Validate URL
        if ( empty( $url ) || ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
            return new WP_Error( 'invalid_url', __( 'Invalid URL', 'scholarships-qa-search' ) );
        }

        // Build tracking data
        $data = array(
            'url'         => $url,
            'query'       => $query,
            'timestamp'   => current_time( 'mysql' ),
            'referrer'    => isset( $_SERVER['HTTP_REFERER'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_REFERER'] ) ) : '',
            'user_agent'  => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',
            'ip_address'  => $this->get_anonymized_ip(),
            'site_url'    => home_url(),
            'site_name'   => get_bloginfo( 'name' ),
            'tracking_id' => isset( $this->options['tracking_id'] ) ? $this->options['tracking_id'] : '',
        );

        // Store tracking data in site database (for local analytics)
        $this->store_tracking_data( $data );

        // Send tracking data to scholarshipsqanda.com if enabled
        $remote_result = $this->send_tracking_data( $data );

        return is_wp_error( $remote_result ) ? $remote_result : true;
    }

    /**
     * Store tracking data in local database.
     *
     * @param array $data Tracking data.
     */
    private function store_tracking_data( $data ) {
        // Get existing tracking data
        $tracking_data = get_option( 'schoqase_tracking_data', array() );

        // Add new entry
        $tracking_data[] = $data;

        // Limit to last 500 entries
        if ( count( $tracking_data ) > 500 ) {
            $tracking_data = array_slice( $tracking_data, -500 );
        }

        // Save
        update_option( 'schoqase_tracking_data', $tracking_data, false );
    }

    /**
     * Send tracking data to scholarshipsqanda.com.
     *
     * @param array $data Tracking data.
     * @return bool|WP_Error True on success, WP_Error on failure.
     */
    private function send_tracking_data( $data ) {
        // Get base URL from options
        $base_url = isset( $this->options['base_url'] ) ? $this->options['base_url'] : SCHOQASE_DOMAIN;
        $tracking_endpoint = trailingslashit( $base_url ) . 'wp-json/schoqase/v1/track';

        // Set up the request
        $args = array(
            'body'      => json_encode( $data ),
            'headers'   => array(
                'Content-Type' => 'application/json',
                'User-Agent'   => 'SCHOQASE-Search-Plugin/' . SCHOQASE_VERSION . '; ' . home_url(),
            ),
            'timeout'   => 5,
            'blocking'  => false, // Non-blocking request
        );

        // Send the request
        $response = wp_remote_post( $tracking_endpoint, $args );

        // We're using non-blocking, so we don't check the result
        return true;
    }

    /**
     * Get anonymized IP address.
     *
     * @return string Anonymized IP address.
     */
    private function get_anonymized_ip() {
        $ip = '0.0.0.0';

        if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
            $ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
            
            // Anonymize IP by removing last octet
            if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) ) {
                $ip = preg_replace( '/\.\d+$/', '.0', $ip );
            } elseif ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6 ) ) {
                // For IPv6, keep only the first 4 hextets
                $ip = preg_replace( '/:[0-9a-f]{0,4}(:[0-9a-f]{0,4}){0,3}$/i', ':0:0:0:0', $ip );
            }
        }

        return $ip;
    }

    /**
     * Get stored tracking data.
     *
     * @param int $limit Maximum number of entries to return.
     * @return array Tracking data.
     */
    public function get_tracking_data( $limit = 100 ) {
        $tracking_data = get_option( 'schoqase_tracking_data', array() );
        
        // Return the most recent entries
        if ( count( $tracking_data ) > $limit ) {
            $tracking_data = array_slice( $tracking_data, -$limit );
        }
        
        return $tracking_data;
    }

    /**
     * Clear tracking data.
     */
    public function clear_tracking_data() {
        delete_option( 'schoqase_tracking_data' );
    }
}
