<?php
/**
 * The core search functionality of the plugin.
 *
 * @package SQandA_Search
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

/**
 * Core search class.
 */
class SCHOQASE_Search_Engine {

    /**
     * The search options.
     *
     * @var array
     */
    private $options;

    /**
     * Initialize the class.
     */
    public function __construct() {
        $this->options = get_option( 'schoqase_search_options', array() );
    }

    /**
     * Generate a unique search ID.
     *
     * @return string Unique search ID.
     */
    public function generate_search_id() {
        return 'schoqase-search-' . uniqid();
    }

    /**
     * Format search results for display.
     *
     * @param array $results Raw search results.
     * @return array Formatted results.
     */
    public function format_results( $results ) {
        $formatted = array();
        
        if ( empty( $results ) || ! is_array( $results ) ) {
            return $formatted;
        }
        
        foreach ( $results as $result ) {
            if ( ! isset( $result['title'], $result['url'], $result['excerpt'] ) ) {
                continue;
            }
            
            $formatted[] = array(
                'title'   => sanitize_text_field( $result['title'] ),
                'url'     => esc_url( $result['url'] ),
                'excerpt' => wp_kses_post( $result['excerpt'] ),
                'score'   => isset( $result['score'] ) ? floatval( $result['score'] ) : 0,
            );
        }
        
        // Sort by relevance score if available
        if ( ! empty( $formatted ) ) {
            usort( $formatted, function( $a, $b ) {
                return $b['score'] <=> $a['score'];
            });
        }
        
        return $formatted;
    }

    /**
     * Process a search query.
     *
     * @param string $query The search query.
     * @param int    $limit Maximum number of results to return.
     * @return array|WP_Error Search results or error.
     */
    public function process_query( $query, $limit = 5 ) {
        if ( empty( $query ) ) {
            return new WP_Error( 'empty_query', __( 'Search query cannot be empty', 'scholarships-qa-search' ) );
        }
        
        $api = new SCHOQASE_API();
        $results = $api->search( $query, $limit );
        
        if ( is_wp_error( $results ) ) {
            return $results;
        }
        
        return $this->format_results( $results );
    }

    /**
     * Get the base URL for scholarshipsqanda.com.
     *
     * @return string The base URL.
     */
    public function get_base_url() {
        return isset( $this->options['base_url'] ) ? esc_url( $this->options['base_url'] ) : SCHOQASE_DOMAIN;
    }

    /**
     * Get search options.
     *
     * @return array Search options.
     */
    public function get_options() {
        $defaults = array(
            'base_url'        => SCHOQASE_DOMAIN,
            'tracking_id'     => '',
            'results_num'     => 5,
            'title'           => __( 'Search Scholarships Q&A', 'scholarships-qa-search' ),
            'placeholder'     => __( 'Ask a scholarship question...', 'scholarships-qa-search' ),
            'button_text'     => __( 'Search', 'scholarships-qa-search' ),
            'results_heading' => __( 'Search Results from ScholarshipsQandA.com', 'scholarships-qa-search' ),
            'no_results_text' => __( 'No results found. Please try a different search query.', 'scholarships-qa-search' ),
            'footer_text'     => '',
            'enable_tracking' => true,
        );
        
        return wp_parse_args( $this->options, $defaults );
    }

    /**
     * Get a specific option value.
     *
     * @param string $key     The option key.
     * @param mixed  $default Default value if option is not set.
     * @return mixed The option value.
     */
    public function get_option( $key, $default = '' ) {
        $options = $this->get_options();
        
        return isset( $options[ $key ] ) ? $options[ $key ] : $default;
    }
}
