<?php
/**
 * API Handler for ScholarshipsQandA Search
 *
 * @package ScholarshipsQandASearch
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * SCHOQASE_API class for handling communication with ScholarshipsQandA.com.
 */
class SCHOQASE_API {

    /**
     * The base URL for the ScholarshipsQandA.com website.
     *
     * @var string
     */
    private $base_url = 'https://www.scholarshipsqanda.com';

    /**
     * The WordPress REST API endpoint for posts.
     *
     * @var string
     */
    private $rest_endpoint = '/wp-json/wp/v2/posts';

    /**
     * Constructor.
     */
    public function __construct() {
        // Allow the base URL to be filtered
        $this->base_url = apply_filters( 'schoqase_search_api_base_url', $this->base_url );
        $this->rest_endpoint = apply_filters( 'schoqase_search_api_rest_endpoint', $this->rest_endpoint );
    }

    /**
     * Search for scholarship content.
     *
     * @param string $query The search query.
     * @param int    $limit The maximum number of results to return.
     * @return array|WP_Error The search results or WP_Error on failure.
     */
    public function search( $query, $limit = 5 ) {
        // Prepare the WordPress REST API request URL
        $search_url = add_query_arg(
            array(
                'search' => urlencode( $query ),
                'per_page' => absint( $limit ),
                '_embed' => 1, // Include embedded content like featured images
            ),
            $this->base_url . $this->rest_endpoint
        );

        // Set a timeout for the request
        $args = array(
            'timeout' => 15,
            'headers' => array(
                'Accept' => 'application/json',
            ),
        );

        // Make the remote request to the WordPress REST API
        $response = wp_remote_get( $search_url, $args );

        // Check for errors
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        // Get the response code
        $response_code = wp_remote_retrieve_response_code( $response );
        if ( $response_code !== 200 ) {
            return new WP_Error(
                'api_error',
                /* translators: %d: The response code from the API. */
                sprintf( __( 'API request failed with response code: %d', 'scholarships-qa-search' ), $response_code )
            );
        }

        // Decode the response body
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( ! is_array( $data ) ) {
            return new WP_Error(
                'api_invalid_response',
                __( 'Invalid response received from API', 'scholarships-qa-search' )
            );
        }

        // Format the results
        $results = array();
        foreach ( $data as $post ) {
            $excerpt = isset( $post['excerpt']['rendered'] ) ? 
                       wp_strip_all_tags( $post['excerpt']['rendered'] ) : 
                       wp_trim_words( wp_strip_all_tags( $post['content']['rendered'] ), 30 );
            
            $results[] = array(
                'id'      => $post['id'],
                'title'   => wp_strip_all_tags( $post['title']['rendered'] ),
                'excerpt' => $excerpt,
                'url'     => $post['link'],
                'date'    => $post['date'],
            );
        }

        return $results;
    }

    /**
     * Fallback method to use when API request fails.
     * This provides query-relevant content as a backup.
     *
     * @param string $query The search query.
     * @param int    $limit The maximum number of results to return.
     * @return array The fallback search results.
     */
    public function get_fallback_results( $query, $limit = 5 ) {
        $query = sanitize_text_field( $query );
        $current_year = gmdate( 'Y' );
        
        // Create query-specific fallback results
        $results = array();
        
        $results[] = array(
            'id'      => 1,
            'title'   => sprintf( 'How to Find %s Scholarships', ucwords( $query ) ),
            'excerpt' => sprintf( 
                'Looking for %s? Here are the most effective strategies for finding and winning scholarships related to %s. Start by researching local opportunities and specific organizations...',
                strtolower( $query ),
                strtolower( $query )
            ),
            'url'     => $this->base_url . '/how-to-find-scholarships',
            'date'    => gmdate( 'Y-m-d H:i:s' ),
        );
        
        $results[] = array(
            'id'      => 2,
            'title'   => sprintf( 'Top Questions About %s', ucwords( $query ) ),
            'excerpt' => sprintf(
                'Common questions about %s include eligibility requirements, application deadlines, and essay tips. Many students wonder how to stand out from other applicants...',
                strtolower( $query )
            ),
            'url'     => $this->base_url . '/top-scholarship-questions',
            'date'    => gmdate( 'Y-m-d H:i:s' ),
        );
        
        $results[] = array(
            'id'      => 3,
            'title'   => sprintf( '%s Success Stories', ucwords( $query ) ),
            'excerpt' => sprintf(
                'Read about students who successfully secured funding through %s. Learn from their experiences and get inspiration for your own scholarship journey...',
                strtolower( $query )
            ),
            'url'     => $this->base_url . '/success-stories',
            'date'    => gmdate( 'Y-m-d H:i:s' ),
        );
        
        $results[] = array(
            'id'      => 4,
            'title'   => sprintf( '%s Scholarship Opportunities for %s', ucwords( $query ), $current_year ),
            'excerpt' => sprintf(
                'Discover the best %s scholarship opportunities available for the %s academic year. These scholarships are open to students from various backgrounds and educational levels...',
                strtolower( $query ),
                $current_year
            ),
            'url'     => $this->base_url . '/scholarship-opportunities',
            'date'    => gmdate( 'Y-m-d H:i:s' ),
        );
        
        $results[] = array(
            'id'      => 5,
            'title'   => 'Expert Advice: Navigating the Scholarship Application Process',
            'excerpt' => 'Our scholarship experts share their top tips for successfully navigating the application process. Learn how to organize your applications, meet deadlines, and create compelling essays...',
            'url'     => $this->base_url . '/expert-advice',
            'date'    => gmdate( 'Y-m-d H:i:s' ),
        );
        
        // Return only the requested number of results
        return array_slice( $results, 0, $limit );
    }
}