<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package SQandA_Search
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

/**
 * Admin class.
 */
class SCHOQASE_Admin {

    /**
     * The plugin options.
     *
     * @var array
     */
    private $options;

    /**
     * Initialize the class.
     */
    public function __construct() {
        $this->options = get_option( 'schoqase_search_options', array() );
    }

    /**
     * Add admin menu.
     */
    public function add_admin_menu() {
        // Using add_options_page with the correct number of parameters (5, not 6 or 7)
        add_options_page(
            __( 'Scholarships Q&A Search', 'scholarships-qa-search' ),  // Page title
            __( 'SQ&A Search', 'scholarships-qa-search' ),              // Menu title
            'manage_options',                                  // Capability
            'schoqase-search',                                 // Menu slug
            array( $this, 'display_settings_page' )            // Function
        );
    }

    /**
     * Register settings.
     */
    public function register_settings() {
        register_setting(
            'schoqase_search_options_group',
            'schoqase_search_options',
            array( $this, 'sanitize_options' )
        );

        add_settings_section(
            'schoqase_search_general_section',
            __( 'General Settings', 'scholarships-qa-search' ),
            array( $this, 'general_section_callback' ),
            'schoqase-search'
        );

        add_settings_field(
            'base_url',
            __( 'ScholarshipsQ&A Base URL', 'scholarships-qa-search' ),
            array( $this, 'base_url_callback' ),
            'schoqase-search',
            'schoqase_search_general_section'
        );

        add_settings_field(
            'results_num',
            __( 'Number of Results', 'scholarships-qa-search' ),
            array( $this, 'results_num_callback' ),
            'schoqase-search',
            'schoqase_search_general_section'
        );

        add_settings_field(
            'enable_tracking',
            __( 'Enable Click Tracking', 'scholarships-qa-search' ),
            array( $this, 'enable_tracking_callback' ),
            'schoqase-search',
            'schoqase_search_general_section'
        );

        add_settings_field(
            'tracking_id',
            __( 'Tracking ID', 'scholarships-qa-search' ),
            array( $this, 'tracking_id_callback' ),
            'schoqase-search',
            'schoqase_search_general_section'
        );

        add_settings_section(
            'schoqase_search_display_section',
            __( 'Display Settings', 'scholarships-qa-search' ),
            array( $this, 'display_section_callback' ),
            'schoqase-search'
        );

        add_settings_field(
            'title',
            __( 'Search Box Title', 'scholarships-qa-search' ),
            array( $this, 'title_callback' ),
            'schoqase-search',
            'schoqase_search_display_section'
        );

        add_settings_field(
            'placeholder',
            __( 'Search Box Placeholder', 'scholarships-qa-search' ),
            array( $this, 'placeholder_callback' ),
            'schoqase-search',
            'schoqase_search_display_section'
        );

        add_settings_field(
            'button_text',
            __( 'Search Button Text', 'scholarships-qa-search' ),
            array( $this, 'button_text_callback' ),
            'schoqase-search',
            'schoqase_search_display_section'
        );

        add_settings_field(
            'results_heading',
            __( 'Results Heading', 'scholarships-qa-search' ),
            array( $this, 'results_heading_callback' ),
            'schoqase-search',
            'schoqase_search_display_section'
        );

        add_settings_field(
            'no_results_text',
            __( 'No Results Text', 'scholarships-qa-search' ),
            array( $this, 'no_results_text_callback' ),
            'schoqase-search',
            'schoqase_search_display_section'
        );

        add_settings_field(
            'footer_text',
            __( 'Footer Attribution Text', 'scholarships-qa-search' ),
            array( $this, 'footer_text_callback' ),
            'schoqase-search',
            'schoqase_search_display_section'
        );
    }

    /**
     * Sanitize options.
     *
     * @param array $input The input options.
     * @return array Sanitized options.
     */
    public function sanitize_options( $input ) {
        $sanitized = array();

        // General settings
        $sanitized['base_url'] = isset( $input['base_url'] ) ? esc_url_raw( $input['base_url'] ) : SCHOQASE_DOMAIN;
        $sanitized['results_num'] = isset( $input['results_num'] ) ? absint( $input['results_num'] ) : 5;
        $sanitized['enable_tracking'] = isset( $input['enable_tracking'] ) ? (bool) $input['enable_tracking'] : true;
        $sanitized['tracking_id'] = isset( $input['tracking_id'] ) ? sanitize_text_field( $input['tracking_id'] ) : '';

        // Display settings
        $sanitized['title'] = isset( $input['title'] ) ? sanitize_text_field( $input['title'] ) : __( 'Search Scholarships Q&A', 'scholarships-qa-search' );
        $sanitized['placeholder'] = isset( $input['placeholder'] ) ? sanitize_text_field( $input['placeholder'] ) : __( 'Ask a scholarship question...', 'scholarships-qa-search' );
        $sanitized['button_text'] = isset( $input['button_text'] ) ? sanitize_text_field( $input['button_text'] ) : __( 'Search', 'scholarships-qa-search' );
        $sanitized['results_heading'] = isset( $input['results_heading'] ) ? sanitize_text_field( $input['results_heading'] ) : __( 'Search Results from ScholarshipsQandA.com', 'scholarships-qa-search' );
        $sanitized['no_results_text'] = isset( $input['no_results_text'] ) ? sanitize_text_field( $input['no_results_text'] ) : __( 'No results found. Please try a different search query.', 'scholarships-qa-search' );
        $sanitized['footer_text'] = isset( $input['footer_text'] ) ? sanitize_text_field( $input['footer_text'] ) : '';

        return $sanitized;
    }

    /**
     * Display the settings page.
     */
    public function display_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'schoqase_search_options_group' );
                do_settings_sections( 'schoqase-search' );
                submit_button();
                ?>
            </form>
            <div class="schoqase-shortcode-info">
                <h2><?php esc_html_e( 'How to Use', 'scholarships-qa-search' ); ?></h2>
                <p><?php esc_html_e( 'You can add the search box to your site using the widget or shortcode:', 'scholarships-qa-search' ); ?></p>
                <code>[schoqase_search]</code>
                <p><?php esc_html_e( 'Shortcode attributes:', 'scholarships-qa-search' ); ?></p>
                <ul>
                    <li><code>title</code> - <?php esc_html_e( 'The search box title', 'scholarships-qa-search' ); ?></li>
                    <li><code>placeholder</code> - <?php esc_html_e( 'The search box placeholder text', 'scholarships-qa-search' ); ?></li>
                    <li><code>button_text</code> - <?php esc_html_e( 'The search button text', 'scholarships-qa-search' ); ?></li>
                    <li><code>results_num</code> - <?php esc_html_e( 'Number of results to display', 'scholarships-qa-search' ); ?></li>
                </ul>
                <p><?php esc_html_e( 'Example:', 'scholarships-qa-search' ); ?></p>
                <code>[schoqase_search title="Find Scholarships" placeholder="What scholarship are you looking for?" button_text="Find" results_num="3"]</code>
            </div>
        </div>
        <?php
    }

    /**
     * General section callback.
     */
    public function general_section_callback() {
        echo '<p>' . esc_html__( 'Configure the general settings for the Scholarships Q&A Search plugin.', 'scholarships-qa-search' ) . '</p>';
    }

    /**
     * Display section callback.
     */
    public function display_section_callback() {
        echo '<p>' . esc_html__( 'Customize the appearance and text of the search box and results.', 'scholarships-qa-search' ) . '</p>';
    }

    /**
     * Base URL callback.
     */
    public function base_url_callback() {
        $base_url = isset( $this->options['base_url'] ) ? $this->options['base_url'] : SCHOQASE_DOMAIN;
        echo '<input type="url" id="base_url" name="schoqase_search_options[base_url]" value="' . esc_attr( $base_url ) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__( 'The base URL of the ScholarshipsQ&A website.', 'scholarships-qa-search' ) . '</p>';
    }

    /**
     * Results number callback.
     */
    public function results_num_callback() {
        $results_num = isset( $this->options['results_num'] ) ? $this->options['results_num'] : 5;
        echo '<input type="number" id="results_num" name="schoqase_search_options[results_num]" value="' . esc_attr( $results_num ) . '" min="1" max="20" />';
        echo '<p class="description">' . esc_html__( 'The number of search results to display.', 'scholarships-qa-search' ) . '</p>';
    }

    /**
     * Enable tracking callback.
     */
    public function enable_tracking_callback() {
        $enable_tracking = isset( $this->options['enable_tracking'] ) ? $this->options['enable_tracking'] : true;
        echo '<input type="checkbox" id="enable_tracking" name="schoqase_search_options[enable_tracking]" ' . checked( $enable_tracking, true, false ) . ' value="1" />';
        echo '<label for="enable_tracking">' . esc_html__( 'Enable click tracking for analytics', 'scholarships-qa-search' ) . '</label>';
    }

    /**
     * Tracking ID callback.
     */
    public function tracking_id_callback() {
        $tracking_id = isset( $this->options['tracking_id'] ) ? $this->options['tracking_id'] : '';
        echo '<input type="text" id="tracking_id" name="schoqase_search_options[tracking_id]" value="' . esc_attr( $tracking_id ) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__( 'Optional tracking ID for analytics (will be included in tracking requests).', 'scholarships-qa-search' ) . '</p>';
    }

    /**
     * Title callback.
     */
    public function title_callback() {
        $title = isset( $this->options['title'] ) ? $this->options['title'] : __( 'Search Scholarships Q&A', 'scholarships-qa-search' );
        echo '<input type="text" id="title" name="schoqase_search_options[title]" value="' . esc_attr( $title ) . '" class="regular-text" />';
    }

    /**
     * Placeholder callback.
     */
    public function placeholder_callback() {
        $placeholder = isset( $this->options['placeholder'] ) ? $this->options['placeholder'] : __( 'Ask a scholarship question...', 'scholarships-qa-search' );
        echo '<input type="text" id="placeholder" name="schoqase_search_options[placeholder]" value="' . esc_attr( $placeholder ) . '" class="regular-text" />';
    }

    /**
     * Button text callback.
     */
    public function button_text_callback() {
        $button_text = isset( $this->options['button_text'] ) ? $this->options['button_text'] : __( 'Search', 'scholarships-qa-search' );
        echo '<input type="text" id="button_text" name="schoqase_search_options[button_text]" value="' . esc_attr( $button_text ) . '" class="regular-text" />';
    }

    /**
     * Results heading callback.
     */
    public function results_heading_callback() {
        $results_heading = isset( $this->options['results_heading'] ) ? $this->options['results_heading'] : __( 'Search Results from ScholarshipsQandA.com', 'scholarships-qa-search' );
        echo '<input type="text" id="results_heading" name="schoqase_search_options[results_heading]" value="' . esc_attr( $results_heading ) . '" class="regular-text" />';
    }

    /**
     * No results text callback.
     */
    public function no_results_text_callback() {
        $no_results_text = isset( $this->options['no_results_text'] ) ? $this->options['no_results_text'] : __( 'No results found. Please try a different search query.', 'scholarships-qa-search' );
        echo '<input type="text" id="no_results_text" name="schoqase_search_options[no_results_text]" value="' . esc_attr( $no_results_text ) . '" class="regular-text" />';
    }

    /**
     * Footer text callback.
     */
    public function footer_text_callback() {
        $footer_text = isset( $this->options['footer_text'] ) ? $this->options['footer_text'] : '';
        echo '<input type="text" id="footer_text" name="schoqase_search_options[footer_text]" value="' . esc_attr( $footer_text ) . '" class="regular-text" />';
    }

    /**
     * Add settings link to plugin page.
     *
     * @param array $links Plugin action links.
     * @return array Modified plugin action links.
     */
    public function add_settings_link( $links ) {
        $settings_link = '<a href="' . admin_url( 'options-general.php?page=schoqase-search' ) . '">' . __( 'Settings', 'scholarships-qa-search' ) . '</a>';
        array_unshift( $links, $settings_link );
        return $links;
    }
}
