/**
 * Scholarships Q&A Search JavaScript
 * 
 * Handles the search functionality for the plugin.
 */
(function($) {
    'use strict';

    // Store search queries and results for caching
    const searchCache = {};

    /**
     * Initialize the search functionality
     */
    function initSearch() {
        // Handle search form submission
        $('.schoqase-search-form').on('submit', function(e) {
            e.preventDefault();
            const form = $(this);
            const query = form.find('.schoqase-search-input').val().trim();
            const resultsContainer = form.siblings('.schoqase-search-results');
            const limit = form.data('results-num') || 5;
            
            if (query.length === 0) {
                return;
            }
            
            performSearch(query, resultsContainer, limit);
        });

        // Handle click tracking
        $(document).on('click', '.schoqase-result-link', function(e) {
            const link = $(this);
            const url = link.attr('href');
            const query = link.data('query') || '';
            
            // Track the click
            trackClick(url, query);
        });
    }

    /**
     * Perform the search and display results
     * 
     * @param {string} query            The search query
     * @param {jQuery} resultsContainer The container to display results in
     * @param {number} limit            Maximum number of results to show
     */
    function performSearch(query, resultsContainer, limit) {
        // Show loading indicator
        resultsContainer.html('<div class="schoqase-loading"><span>Searching...</span></div>').show();
        
        // Check cache first
        const cacheKey = query + '-' + limit;
        if (searchCache[cacheKey]) {
            displayResults(searchCache[cacheKey], resultsContainer);
            return;
        }
        
        // Perform AJAX search
        $.ajax({
            url: schoqaseSearchParams.ajaxUrl,
            type: 'POST',
            data: {
                action: 'schoqase_search_results',
                nonce: schoqaseSearchParams.nonce,
                schoqase_search_nonce: $('input[name="schoqase_search_nonce"]').val(),
                query: query,
                limit: limit
            },
            success: function(response) {
                if (response.success) {
                    // Cache the results
                    searchCache[cacheKey] = response.data;
                    
                    // Display the results
                    displayResults(response.data, resultsContainer);
                } else {
                    // Show error message
                    resultsContainer.html('<div class="schoqase-error">' + (response.data.message || 'An error occurred.') + '</div>');
                }
            },
            error: function() {
                resultsContainer.html('<div class="schoqase-error">Error connecting to server. Please try again.</div>');
            }
        });
    }

    /**
     * Display search results
     * 
     * @param {object} data             The search results data
     * @param {jQuery} resultsContainer The container to display results in
     */
    function displayResults(data, resultsContainer) {
        // Display the HTML results
        resultsContainer.html(data.html);
        
        // Add query parameter to result links for tracking
        if (data.results && data.results.length) {
            const query = data.query || $('.schoqase-search-input').val().trim();
            
            $('.schoqase-result-link').each(function() {
                $(this).data('query', query);
            });
            
            // Apply search term highlighting to the excerpts
            highlightSearchTerms(query, resultsContainer);
        }
        
        // Scroll to results if they're not visible
        if (!isElementInViewport(resultsContainer[0])) {
            $('html, body').animate({
                scrollTop: resultsContainer.offset().top - 50
            }, 500);
        }
    }
    
    /**
     * Highlight search terms in the results
     * 
     * @param {string} query            The search query
     * @param {jQuery} resultsContainer The container with results
     */
    function highlightSearchTerms(query, resultsContainer) {
        console.log('Highlighting search terms for query:', query);
        
        // If query is empty, do nothing
        if (!query || query.trim() === '') {
            console.log('Query is empty, skipping highlight');
            return;
        }
        
        // Break query into terms
        const terms = query.split(' ');
        console.log('Search terms:', terms);
        
        // Process excerpts only
        const excerpts = resultsContainer.find('.schoqase-result-excerpt');
        console.log('Found', excerpts.length, 'excerpts to highlight');
        
        excerpts.each(function(index) {
            // Get the original content
            let content = $(this).html();
            let originalContent = content;
            console.log('Processing excerpt', index + 1);
            
            // Apply highlighting to each term
            terms.forEach(function(term) {
                if (term.length > 2) { // Only highlight terms with 3+ characters
                    console.log('Highlighting term:', term);
                    
                    // Create a case-insensitive global regex for the term
                    const regex = new RegExp('(' + escapeRegExp(term) + ')', 'ig');
                    
                    // Replace all occurrences with our highlighted span
                    content = content.replace(regex, '<span class="schoqase-highlight">$1</span>');
                }
            });
            
            // Log if content was modified
            if (content !== originalContent) {
                console.log('Highlighted content in excerpt', index + 1);
            } else {
                console.log('No terms found in excerpt', index + 1);
            }
            
            // Update the content with highlights
            $(this).html(content);
        });
        
        // Fallback method: Direct text highlighting if no excerpts were modified
        if (excerpts.length === 0) {
            console.log('No excerpts found, trying direct text highlighting');
            resultsContainer.find('*').contents().each(function() {
                if (this.nodeType === 3) { // Text node
                    const text = this.nodeValue;
                    let newHtml = text;
                    
                    terms.forEach(function(term) {
                        if (term.length > 2) {
                            const regex = new RegExp('(' + escapeRegExp(term) + ')', 'ig');
                            if (regex.test(newHtml)) {
                                console.log('Found term in text node:', term);
                                newHtml = newHtml.replace(regex, '<span class="schoqase-highlight">$1</span>');
                            }
                        }
                    });
                    
                    if (newHtml !== text) {
                        const $newElement = $('<span>' + newHtml + '</span>');
                        $(this).replaceWith($newElement);
                    }
                }
            });
        }
        
        // Add a class to the results container to indicate highlighting is done
        resultsContainer.addClass('schoqase-highlighted');
        console.log('Highlighting complete');
    }
    
    /**
     * Escape string for safe use in regex
     * 
     * @param {string} string The string to escape
     * @return {string} Escaped string
     */
    function escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }

    /**
     * Track link clicks
     * 
     * @param {string} url   The URL being clicked
     * @param {string} query The search query that led to this result
     */
    function trackClick(url, query) {
        $.ajax({
            url: schoqaseSearchParams.ajaxUrl,
            type: 'POST',
            data: {
                action: 'schoqase_track',
                nonce: schoqaseSearchParams.nonce,
                url: url,
                query: query
            },
            // We don't need to wait for the response, this is just for tracking
            async: true
        });
    }

    /**
     * Check if element is visible in the viewport
     * 
     * @param {Element} el The element to check
     * @return {boolean} Whether the element is in the viewport
     */
    function isElementInViewport(el) {
        if (!el) return false;
        
        const rect = el.getBoundingClientRect();
        
        return (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
            rect.right <= (window.innerWidth || document.documentElement.clientWidth)
        );
    }

    // Initialize when document is ready
    $(document).ready(function() {
        initSearch();
    });

})(jQuery);