<?php
/**
 * Plugin Name:       Scheduled Posts Showcase
 * Plugin URI:        https://servicios.ayudawp.com
 * Description:       Display your scheduled posts on the frontend without generating 404 links. Show visitors what's coming next with a beautiful, customizable widget, block, or shortcode.
 * Version:           1.0.0
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * Author:            Fernando Tellado
 * Author URI:        https://ayudawp.com
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       scheduled-posts-showcase
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plugin constants.
 */
define( 'SPSCASE_VERSION', '1.0.0' );
define( 'SPSCASE_PLUGIN_FILE', __FILE__ );
define( 'SPSCASE_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SPSCASE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'SPSCASE_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Get default plugin settings.
 *
 * @return array Default settings array.
 */
function spscase_get_default_settings() {
	return array(
		// Display options.
		'count'           => 5,
		'post_type'       => 'post',
		'order'           => 'ASC',
		'show_image'      => true,
		'image_size'      => 'thumbnail',
		'show_date'       => true,
		'show_excerpt'    => false,
		'excerpt_length'  => 25,
		'show_categories' => false,
		// Appearance.
		'heading'         => __( 'Upcoming posts', 'scheduled-posts-showcase' ),
		'heading_tag'     => 'h3',
		'list_style'      => 'theme',
		'icon'            => 'dashicons-calendar-alt',
		'container_style' => 'card',
		'accent_color'    => '#0073aa',
		// Visibility.
		'visibility'      => 'public',
		// Footer content.
		'footer'          => '',
		// Empty state.
		'no_posts_message' => __( 'No scheduled posts at this time.', 'scheduled-posts-showcase' ),
	);
}

/**
 * Get plugin settings with defaults.
 *
 * @return array Plugin settings merged with defaults.
 */
function spscase_get_settings() {
	$defaults = spscase_get_default_settings();
	$settings = get_option( 'spscase_settings', array() );

	return wp_parse_args( $settings, $defaults );
}

/**
 * Plugin activation hook.
 *
 * Sets up default options.
 *
 * @return void
 */
function spscase_activate() {
	// Create default options if they don't exist.
	if ( false === get_option( 'spscase_settings' ) ) {
		update_option( 'spscase_settings', spscase_get_default_settings() );
	}

	// Set flag to show activation notice.
	set_transient( 'spscase_activation_notice', true, 30 );
}
register_activation_hook( __FILE__, 'spscase_activate' );

/**
 * Include required files.
 *
 * @return void
 */
function spscase_includes() {
	// Core classes.
	require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-cache.php';
	require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-query.php';
	require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-render.php';

	// Output methods.
	require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-shortcode.php';
	require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-widget.php';
	require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-block.php';
	require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-rest-api.php';

	// Admin.
	if ( is_admin() ) {
		require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-settings.php';
		require_once SPSCASE_PLUGIN_DIR . 'includes/class-spscase-promo-banner.php';
	}
}
add_action( 'plugins_loaded', 'spscase_includes' );

/**
 * Initialize plugin components.
 *
 * @return void
 */
function spscase_init() {
	// Initialize cache (hooks for invalidation).
	SPSCASE_Cache::init();

	// Initialize shortcode.
	SPSCASE_Shortcode::init();

	// Initialize REST API.
	SPSCASE_Rest_API::init();

	// Initialize block.
	SPSCASE_Block::init();
}
add_action( 'init', 'spscase_init' );

/**
 * Register the widget.
 *
 * @return void
 */
function spscase_register_widget() {
	register_widget( 'SPSCASE_Widget' );
}
add_action( 'widgets_init', 'spscase_register_widget' );

/**
 * Initialize admin components.
 *
 * Runs on admin_menu to ensure menu pages are registered properly.
 *
 * @return void
 */
function spscase_admin_menu() {
	if ( ! is_admin() ) {
		return;
	}

	// Initialize settings page (adds menu).
	SPSCASE_Settings::add_menu_page();
}
add_action( 'admin_menu', 'spscase_admin_menu' );

/**
 * Initialize admin settings registration.
 *
 * @return void
 */
function spscase_admin_init() {
	if ( ! is_admin() ) {
		return;
	}

	// Register settings.
	SPSCASE_Settings::register_settings();

	// Add settings link to plugins page.
	add_filter( 'plugin_action_links_' . SPSCASE_PLUGIN_BASENAME, 'spscase_add_settings_link' );
}
add_action( 'admin_init', 'spscase_admin_init' );

/**
 * Add settings link to plugin actions.
 *
 * @param array $links Existing plugin action links.
 * @return array Modified plugin action links.
 */
function spscase_add_settings_link( $links ) {
	$settings_link = sprintf(
		'<a href="%s">%s</a>',
		esc_url( admin_url( 'tools.php?page=scheduled-posts-showcase' ) ),
		esc_html__( 'Settings', 'scheduled-posts-showcase' )
	);

	array_unshift( $links, $settings_link );

	return $links;
}

/**
 * Display activation notice.
 *
 * @return void
 */
function spscase_activation_notice() {
	// Check if we should show the notice.
	if ( ! get_transient( 'spscase_activation_notice' ) ) {
		return;
	}

	// Only show to users who can manage options.
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	// Delete the transient so notice only shows once.
	delete_transient( 'spscase_activation_notice' );

	$settings_url = admin_url( 'tools.php?page=scheduled-posts-showcase' );
	?>
	<div class="notice notice-success is-dismissible">
		<p>
			<strong><?php esc_html_e( 'Scheduled Posts Showcase is ready!', 'scheduled-posts-showcase' ); ?></strong>
			<?php esc_html_e( 'Configure the display settings, then add the Gutenberg block or classic widget to your sidebars, or use the', 'scheduled-posts-showcase' ); ?>
			<code>[scheduled-posts-showcase]</code>
			<?php esc_html_e( 'shortcode anywhere.', 'scheduled-posts-showcase' ); ?>
		</p>
		<p>
			<a href="<?php echo esc_url( $settings_url ); ?>" class="button button-primary">
				<?php esc_html_e( 'Go to settings', 'scheduled-posts-showcase' ); ?>
			</a>
		</p>
	</div>
	<?php
}
add_action( 'admin_notices', 'spscase_activation_notice' );

/**
 * Enqueue frontend assets.
 *
 * @return void
 */
function spscase_enqueue_frontend_assets() {
	$settings = spscase_get_settings();

	// Build dependencies array - only include dashicons if needed.
	$dependencies = array();
	if ( 'dashicon' === $settings['list_style'] ) {
		// Ensure dashicons is available on frontend when needed.
		wp_enqueue_style( 'dashicons' );
		$dependencies[] = 'dashicons';
	}

	wp_register_style(
		'spscase-frontend',
		SPSCASE_PLUGIN_URL . 'assets/css/frontend.css',
		$dependencies,
		SPSCASE_VERSION
	);

	// Add inline CSS for accent color.
	$accent_color = sanitize_hex_color( $settings['accent_color'] );

	if ( $accent_color ) {
		$inline_css = sprintf(
			':root { --sps-accent-color: %s; }',
			$accent_color
		);
		wp_add_inline_style( 'spscase-frontend', $inline_css );
	}
}
add_action( 'wp_enqueue_scripts', 'spscase_enqueue_frontend_assets' );

/**
 * Enqueue admin assets.
 *
 * @param string $hook_suffix Current admin page hook suffix.
 * @return void
 */
function spscase_enqueue_admin_assets( $hook_suffix ) {
	// Load widget styles on widgets page and customizer.
	if ( 'widgets.php' === $hook_suffix || 'customize.php' === $hook_suffix ) {
		wp_enqueue_style(
			'spscase-admin-widgets',
			SPSCASE_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			SPSCASE_VERSION
		);
		return;
	}

	// Only load full admin assets on plugin settings page.
	if ( 'tools_page_scheduled-posts-showcase' !== $hook_suffix ) {
		return;
	}

	// Color picker.
	wp_enqueue_style( 'wp-color-picker' );

	// Thickbox for plugin installation modals.
	add_thickbox();

	// Admin styles.
	wp_enqueue_style(
		'spscase-admin',
		SPSCASE_PLUGIN_URL . 'assets/css/admin.css',
		array( 'wp-color-picker' ),
		SPSCASE_VERSION
	);

	// Admin scripts.
	wp_enqueue_script(
		'spscase-admin',
		SPSCASE_PLUGIN_URL . 'assets/js/admin.js',
		array( 'jquery', 'wp-color-picker' ),
		SPSCASE_VERSION,
		true
	);

	// Localize script for AJAX.
	wp_localize_script(
		'spscase-admin',
		'spscaseAdmin',
		array(
			'ajaxUrl' => admin_url( 'admin-ajax.php' ),
			'nonce'   => wp_create_nonce( 'spscase_admin_nonce' ),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'spscase_enqueue_admin_assets' );

/**
 * Check if current user can view scheduled posts based on visibility setting.
 *
 * @return bool True if user can view, false otherwise.
 */
function spscase_user_can_view() {
	$settings   = spscase_get_settings();
	$visibility = $settings['visibility'];

	switch ( $visibility ) {
		case 'logged_in':
			return is_user_logged_in();

		case 'edit_posts':
			return current_user_can( 'edit_posts' );

		case 'public':
		default:
			return true;
	}
}

/**
 * Get allowed HTML tags for footer content.
 *
 * @return array Allowed HTML tags for wp_kses().
 */
function spscase_get_allowed_footer_html() {
	$allowed = array(
		'a'      => array(
			'href'   => array(),
			'title'  => array(),
			'target' => array(),
			'rel'    => array(),
			'class'  => array(),
		),
		'strong' => array(
			'class' => array(),
		),
		'em'     => array(
			'class' => array(),
		),
		'br'     => array(),
		'span'   => array(
			'class' => array(),
			'style' => array(),
		),
		'p'      => array(
			'class' => array(),
		),
	);

	/**
	 * Filter allowed HTML tags for footer content.
	 *
	 * @param array $allowed Allowed HTML tags array for wp_kses().
	 */
	return apply_filters( 'spscase_allowed_footer_html', $allowed );
}

/**
 * Get curated list of dashicons for icon selector.
 *
 * @return array Array of dashicon class names.
 */
function spscase_get_available_icons() {
	return array(
		'dashicons-calendar-alt',
		'dashicons-calendar',
		'dashicons-clock',
		'dashicons-star-filled',
		'dashicons-star-empty',
		'dashicons-arrow-right-alt',
		'dashicons-arrow-right-alt2',
		'dashicons-marker',
		'dashicons-yes',
		'dashicons-megaphone',
		'dashicons-edit',
		'dashicons-admin-post',
		'dashicons-media-text',
		'dashicons-format-aside',
		'dashicons-visibility',
	);
}

/**
 * Get available heading tags.
 *
 * @return array Array of heading tag options.
 */
function spscase_get_heading_tags() {
	return array(
		'h2' => 'H2',
		'h3' => 'H3',
		'h4' => 'H4',
		'h5' => 'H5',
		'h6' => 'H6',
	);
}