<?php
/**
 * Widget class.
 *
 * Classic WordPress widget for displaying scheduled posts.
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPSCASE_Widget class.
 *
 * Extends WP_Widget for classic widget functionality.
 */
class SPSCASE_Widget extends WP_Widget {

	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct(
			'spscase_scheduled_posts',
			__( 'Scheduled Posts Showcase', 'scheduled-posts-showcase' ),
			array(
				'description'                 => __( 'Display your scheduled posts without 404 links.', 'scheduled-posts-showcase' ),
				'customize_selective_refresh' => true,
				'show_instance_in_rest'       => true,
			)
		);
	}

	/**
	 * Get default instance values.
	 *
	 * @return array Default values.
	 */
	private function get_defaults() {
		$settings = spscase_get_settings();

		return array(
			'title'            => '',
			'count'            => $settings['count'],
			'post_type'        => $settings['post_type'],
			'order'            => $settings['order'],
			'show_image'       => $settings['show_image'],
			'image_size'       => $settings['image_size'],
			'show_date'        => $settings['show_date'],
			'show_excerpt'     => $settings['show_excerpt'],
			'show_categories'  => $settings['show_categories'],
			'heading'          => $settings['heading'],
			'heading_tag'      => $settings['heading_tag'],
			'list_style'       => $settings['list_style'],
			'icon'             => $settings['icon'],
			'container_style'  => $settings['container_style'],
			'footer'           => $settings['footer'],
			'no_posts_message' => $settings['no_posts_message'],
		);
	}

	/**
	 * Widget frontend output.
	 *
	 * @param array $args     Widget arguments.
	 * @param array $instance Widget instance settings.
	 * @return void
	 */
	public function widget( $args, $instance ) {
		// Check visibility.
		if ( ! spscase_user_can_view() ) {
			return;
		}

		$instance = wp_parse_args( $instance, $this->get_defaults() );
		$settings = spscase_get_settings();

		// Build settings for query and render.
		$display_settings = array(
			'count'            => absint( $instance['count'] ),
			'post_type'        => $instance['post_type'],
			'order'            => $instance['order'],
			'show_image'       => ! empty( $instance['show_image'] ),
			'image_size'       => $instance['image_size'],
			'show_date'        => ! empty( $instance['show_date'] ),
			'show_excerpt'     => ! empty( $instance['show_excerpt'] ),
			'show_categories'  => ! empty( $instance['show_categories'] ),
			'heading'          => $instance['heading'],
			'heading_tag'      => $instance['heading_tag'],
			'list_style'       => $instance['list_style'],
			'icon'             => $instance['icon'],
			'container_style'  => $instance['container_style'],
			'footer'           => $instance['footer'],
			'no_posts_message' => $instance['no_posts_message'],
			'excerpt_length'   => $settings['excerpt_length'],
			'accent_color'     => $settings['accent_color'],
		);

		// Get posts.
		$posts = SPSCASE_Query::get_posts( $display_settings );

		// Don't output anything if no posts (unless in Customizer).
		if ( empty( $posts ) ) {
			if ( is_customize_preview() ) {
				echo wp_kses_post( $args['before_widget'] );
				if ( ! empty( $instance['title'] ) ) {
					echo wp_kses_post( $args['before_title'] ) . esc_html( $instance['title'] ) . wp_kses_post( $args['after_title'] );
				}
				echo '<div class="sps-widget-container">';
				echo wp_kses_post( SPSCASE_Render::admin_placeholder( $display_settings ) );
				echo '</div>';
				echo wp_kses_post( $args['after_widget'] );
			}
			return;
		}

		// Output widget.
		echo wp_kses_post( $args['before_widget'] );

		if ( ! empty( $instance['title'] ) ) {
			echo wp_kses_post( $args['before_title'] ) . esc_html( $instance['title'] ) . wp_kses_post( $args['after_title'] );
		}

		echo '<div class="sps-widget-container">';
		echo wp_kses_post( SPSCASE_Render::output( $posts, $display_settings ) );
		echo '</div>';

		echo wp_kses_post( $args['after_widget'] );
	}

	/**
	 * Widget form in admin.
	 *
	 * @param array $instance Current settings.
	 * @return void
	 */
	public function form( $instance ) {
		$instance     = wp_parse_args( $instance, $this->get_defaults() );
		$post_types   = $this->get_post_types();
		$heading_tags = spscase_get_heading_tags();
		$icons        = spscase_get_available_icons();
		?>

		<!-- Title -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>">
				<?php esc_html_e( 'Title:', 'scheduled-posts-showcase' ); ?>
			</label>
			<input
				type="text"
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>"
				value="<?php echo esc_attr( $instance['title'] ); ?>"
			>
		</p>

		<hr>
		<p><strong><?php esc_html_e( 'Display options', 'scheduled-posts-showcase' ); ?></strong></p>

		<!-- Count -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'count' ) ); ?>">
				<?php esc_html_e( 'Number of posts:', 'scheduled-posts-showcase' ); ?>
			</label>
			<input
				type="number"
				class="tiny-text"
				id="<?php echo esc_attr( $this->get_field_id( 'count' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'count' ) ); ?>"
				value="<?php echo esc_attr( $instance['count'] ); ?>"
				min="1"
				max="20"
			>
		</p>

		<!-- Post Type -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'post_type' ) ); ?>">
				<?php esc_html_e( 'Post type:', 'scheduled-posts-showcase' ); ?>
			</label>
			<select
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'post_type' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'post_type' ) ); ?>"
			>
				<?php foreach ( $post_types as $slug => $label ) : ?>
					<option value="<?php echo esc_attr( $slug ); ?>" <?php selected( $instance['post_type'], $slug ); ?>>
						<?php echo esc_html( $label ); ?>
					</option>
				<?php endforeach; ?>
			</select>
		</p>

		<!-- Order -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'order' ) ); ?>">
				<?php esc_html_e( 'Order:', 'scheduled-posts-showcase' ); ?>
			</label>
			<select
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'order' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'order' ) ); ?>"
			>
				<option value="ASC" <?php selected( $instance['order'], 'ASC' ); ?>>
					<?php esc_html_e( 'Ascending (next first)', 'scheduled-posts-showcase' ); ?>
				</option>
				<option value="DESC" <?php selected( $instance['order'], 'DESC' ); ?>>
					<?php esc_html_e( 'Descending (furthest first)', 'scheduled-posts-showcase' ); ?>
				</option>
			</select>
		</p>

		<!-- Show Image -->
		<p>
			<input
				type="checkbox"
				class="checkbox"
				id="<?php echo esc_attr( $this->get_field_id( 'show_image' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'show_image' ) ); ?>"
				value="1"
				<?php checked( $instance['show_image'] ); ?>
			>
			<label for="<?php echo esc_attr( $this->get_field_id( 'show_image' ) ); ?>">
				<?php esc_html_e( 'Show featured image', 'scheduled-posts-showcase' ); ?>
			</label>
		</p>

		<!-- Image Size -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'image_size' ) ); ?>">
				<?php esc_html_e( 'Image size:', 'scheduled-posts-showcase' ); ?>
			</label>
			<select
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'image_size' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'image_size' ) ); ?>"
			>
				<option value="thumbnail" <?php selected( $instance['image_size'], 'thumbnail' ); ?>>
					<?php esc_html_e( 'Thumbnail', 'scheduled-posts-showcase' ); ?>
				</option>
				<option value="medium" <?php selected( $instance['image_size'], 'medium' ); ?>>
					<?php esc_html_e( 'Medium', 'scheduled-posts-showcase' ); ?>
				</option>
			</select>
		</p>

		<!-- Show Date -->
		<p>
			<input
				type="checkbox"
				class="checkbox"
				id="<?php echo esc_attr( $this->get_field_id( 'show_date' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'show_date' ) ); ?>"
				value="1"
				<?php checked( $instance['show_date'] ); ?>
			>
			<label for="<?php echo esc_attr( $this->get_field_id( 'show_date' ) ); ?>">
				<?php esc_html_e( 'Show date', 'scheduled-posts-showcase' ); ?>
			</label>
		</p>

		<!-- Show Excerpt -->
		<p>
			<input
				type="checkbox"
				class="checkbox"
				id="<?php echo esc_attr( $this->get_field_id( 'show_excerpt' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'show_excerpt' ) ); ?>"
				value="1"
				<?php checked( $instance['show_excerpt'] ); ?>
			>
			<label for="<?php echo esc_attr( $this->get_field_id( 'show_excerpt' ) ); ?>">
				<?php esc_html_e( 'Show excerpt', 'scheduled-posts-showcase' ); ?>
			</label>
		</p>

		<!-- Show Categories -->
		<p>
			<input
				type="checkbox"
				class="checkbox"
				id="<?php echo esc_attr( $this->get_field_id( 'show_categories' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'show_categories' ) ); ?>"
				value="1"
				<?php checked( $instance['show_categories'] ); ?>
			>
			<label for="<?php echo esc_attr( $this->get_field_id( 'show_categories' ) ); ?>">
				<?php esc_html_e( 'Show categories', 'scheduled-posts-showcase' ); ?>
			</label>
		</p>

		<hr>
		<p><strong><?php esc_html_e( 'Appearance', 'scheduled-posts-showcase' ); ?></strong></p>

		<!-- Heading -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'heading' ) ); ?>">
				<?php esc_html_e( 'Heading text:', 'scheduled-posts-showcase' ); ?>
			</label>
			<input
				type="text"
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'heading' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'heading' ) ); ?>"
				value="<?php echo esc_attr( $instance['heading'] ); ?>"
			>
		</p>

		<!-- Heading Tag -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'heading_tag' ) ); ?>">
				<?php esc_html_e( 'Heading tag:', 'scheduled-posts-showcase' ); ?>
			</label>
			<select
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'heading_tag' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'heading_tag' ) ); ?>"
			>
				<?php foreach ( $heading_tags as $tag => $label ) : ?>
					<option value="<?php echo esc_attr( $tag ); ?>" <?php selected( $instance['heading_tag'], $tag ); ?>>
						<?php echo esc_html( $label ); ?>
					</option>
				<?php endforeach; ?>
			</select>
		</p>

		<!-- List Style -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'list_style' ) ); ?>">
				<?php esc_html_e( 'List style:', 'scheduled-posts-showcase' ); ?>
			</label>
			<select
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'list_style' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'list_style' ) ); ?>"
			>
				<option value="theme" <?php selected( $instance['list_style'], 'theme' ); ?>>
					<?php esc_html_e( 'Theme default', 'scheduled-posts-showcase' ); ?>
				</option>
				<option value="dashicon" <?php selected( $instance['list_style'], 'dashicon' ); ?>>
					<?php esc_html_e( 'Dashicon', 'scheduled-posts-showcase' ); ?>
				</option>
				<option value="none" <?php selected( $instance['list_style'], 'none' ); ?>>
					<?php esc_html_e( 'None', 'scheduled-posts-showcase' ); ?>
				</option>
			</select>
		</p>

		<!-- Icon -->
		<p>
			<label><?php esc_html_e( 'Icon (for Dashicon style):', 'scheduled-posts-showcase' ); ?></label>
			<span class="sps-widget-icons">
				<?php foreach ( $icons as $icon ) : ?>
					<label class="sps-widget-icon-option" title="<?php echo esc_attr( $icon ); ?>">
						<input
							type="radio"
							name="<?php echo esc_attr( $this->get_field_name( 'icon' ) ); ?>"
							value="<?php echo esc_attr( $icon ); ?>"
							<?php checked( $instance['icon'], $icon ); ?>
							style="display:none;"
						>
						<span class="dashicons <?php echo esc_attr( $icon ); ?>"></span>
					</label>
				<?php endforeach; ?>
			</span>
		</p>

		<!-- Container Style -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'container_style' ) ); ?>">
				<?php esc_html_e( 'Container style:', 'scheduled-posts-showcase' ); ?>
			</label>
			<select
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'container_style' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'container_style' ) ); ?>"
			>
				<option value="card" <?php selected( $instance['container_style'], 'card' ); ?>>
					<?php esc_html_e( 'Card', 'scheduled-posts-showcase' ); ?>
				</option>
				<option value="minimal" <?php selected( $instance['container_style'], 'minimal' ); ?>>
					<?php esc_html_e( 'Minimal', 'scheduled-posts-showcase' ); ?>
				</option>
			</select>
		</p>

		<hr>
		<p><strong><?php esc_html_e( 'Additional content', 'scheduled-posts-showcase' ); ?></strong></p>

		<!-- Footer -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'footer' ) ); ?>">
				<?php esc_html_e( 'Footer HTML:', 'scheduled-posts-showcase' ); ?>
			</label>
			<textarea
				class="widefat"
				rows="3"
				id="<?php echo esc_attr( $this->get_field_id( 'footer' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'footer' ) ); ?>"
				placeholder="<?php esc_attr_e( 'Add a call to action or additional text (optional). Allowed HTML: <a>, <strong>, <em>, <br>, <span>, <p>', 'scheduled-posts-showcase' ); ?>"
			><?php echo esc_textarea( $instance['footer'] ); ?></textarea>
		</p>

		<!-- No Posts Message -->
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'no_posts_message' ) ); ?>">
				<?php esc_html_e( 'No posts message:', 'scheduled-posts-showcase' ); ?>
			</label>
			<input
				type="text"
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'no_posts_message' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'no_posts_message' ) ); ?>"
				value="<?php echo esc_attr( $instance['no_posts_message'] ); ?>"
			>
		</p>

		<hr>
		<p>
			<a href="<?php echo esc_url( admin_url( 'tools.php?page=scheduled-posts-showcase' ) ); ?>" class="button" target="_blank">
				<?php esc_html_e( 'Configure global defaults', 'scheduled-posts-showcase' ); ?>
			</a>
		</p>

		<?php
	}

	/**
	 * Update widget settings.
	 *
	 * @param array $new_instance New settings.
	 * @param array $old_instance Old settings.
	 * @return array Sanitized settings.
	 */
	public function update( $new_instance, $old_instance ) {
		$instance = array();

		$instance['title']           = sanitize_text_field( $new_instance['title'] ?? '' );
		$instance['count']           = absint( $new_instance['count'] ?? 5 );
		$instance['post_type']       = sanitize_key( $new_instance['post_type'] ?? 'post' );
		$instance['order']           = ( 'DESC' === strtoupper( $new_instance['order'] ?? 'ASC' ) ) ? 'DESC' : 'ASC';
		$instance['show_image']      = ! empty( $new_instance['show_image'] );
		$instance['image_size']      = sanitize_key( $new_instance['image_size'] ?? 'thumbnail' );
		$instance['show_date']       = ! empty( $new_instance['show_date'] );
		$instance['show_excerpt']    = ! empty( $new_instance['show_excerpt'] );
		$instance['show_categories'] = ! empty( $new_instance['show_categories'] );
		$instance['heading']         = sanitize_text_field( $new_instance['heading'] ?? '' );
		$instance['heading_tag']     = sanitize_key( $new_instance['heading_tag'] ?? 'h3' );
		$instance['list_style']      = sanitize_key( $new_instance['list_style'] ?? 'theme' );
		$instance['icon']            = sanitize_text_field( $new_instance['icon'] ?? 'dashicons-calendar-alt' );
		$instance['container_style'] = sanitize_key( $new_instance['container_style'] ?? 'card' );
		$instance['footer']          = wp_kses( $new_instance['footer'] ?? '', spscase_get_allowed_footer_html() );
		$instance['no_posts_message'] = sanitize_text_field( $new_instance['no_posts_message'] ?? '' );

		// Validate count.
		if ( $instance['count'] < 1 ) {
			$instance['count'] = 1;
		}
		if ( $instance['count'] > 20 ) {
			$instance['count'] = 20;
		}

		// Validate heading tag.
		$valid_tags = array_keys( spscase_get_heading_tags() );
		if ( ! in_array( $instance['heading_tag'], $valid_tags, true ) ) {
			$instance['heading_tag'] = 'h3';
		}

		// Validate list style.
		$valid_styles = array( 'theme', 'dashicon', 'none' );
		if ( ! in_array( $instance['list_style'], $valid_styles, true ) ) {
			$instance['list_style'] = 'theme';
		}

		// Validate icon.
		$valid_icons = spscase_get_available_icons();
		if ( ! in_array( $instance['icon'], $valid_icons, true ) ) {
			$instance['icon'] = 'dashicons-calendar-alt';
		}

		// Validate container style.
		$valid_containers = array( 'card', 'minimal' );
		if ( ! in_array( $instance['container_style'], $valid_containers, true ) ) {
			$instance['container_style'] = 'card';
		}

		// Validate image size.
		$valid_sizes = array( 'thumbnail', 'medium' );
		if ( ! in_array( $instance['image_size'], $valid_sizes, true ) ) {
			$instance['image_size'] = 'thumbnail';
		}

		return $instance;
	}

	/**
	 * Get available public post types.
	 *
	 * @return array Post types with slug => label.
	 */
	private function get_post_types() {
		$post_types = get_post_types( array( 'public' => true ), 'objects' );
		$options    = array();

		foreach ( $post_types as $post_type ) {
			if ( 'attachment' === $post_type->name ) {
				continue;
			}
			$options[ $post_type->name ] = $post_type->labels->singular_name;
		}

		return $options;
	}
}
