<?php
/**
 * Shortcode class.
 *
 * Registers and handles the [scheduled-posts-showcase] shortcode.
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPSCASE_Shortcode class.
 *
 * Handles shortcode registration and output.
 */
class SPSCASE_Shortcode {

	/**
	 * Initialize shortcode.
	 *
	 * @return void
	 */
	public static function init() {
		add_shortcode( 'scheduled-posts-showcase', array( __CLASS__, 'render' ) );
	}

	/**
	 * Render shortcode output.
	 *
	 * @param array  $atts    Shortcode attributes.
	 * @param string $content Shortcode content (unused).
	 * @return string Shortcode output.
	 */
	public static function render( $atts, $content = null ) {
		// Check visibility.
		if ( ! spscase_user_can_view() ) {
			return '';
		}

		$defaults = spscase_get_default_settings();

		// Parse attributes.
		$atts = shortcode_atts(
			array(
				'count'            => $defaults['count'],
				'post_type'        => $defaults['post_type'],
				'order'            => $defaults['order'],
				'show_image'       => $defaults['show_image'] ? '1' : '0',
				'image_size'       => $defaults['image_size'],
				'show_date'        => $defaults['show_date'] ? '1' : '0',
				'show_excerpt'     => $defaults['show_excerpt'] ? '1' : '0',
				'show_categories'  => $defaults['show_categories'] ? '1' : '0',
				'heading'          => $defaults['heading'],
				'heading_tag'      => $defaults['heading_tag'],
				'list_style'       => $defaults['list_style'],
				'icon'             => $defaults['icon'],
				'container_style'  => $defaults['container_style'],
				'footer'           => $defaults['footer'],
				'no_posts_message' => $defaults['no_posts_message'],
			),
			$atts,
			'scheduled-posts-showcase'
		);

		// Convert string booleans.
		$settings = array(
			'count'            => absint( $atts['count'] ),
			'post_type'        => sanitize_key( $atts['post_type'] ),
			'order'            => strtoupper( $atts['order'] ) === 'DESC' ? 'DESC' : 'ASC',
			'show_image'       => self::parse_bool( $atts['show_image'] ),
			'image_size'       => sanitize_key( $atts['image_size'] ),
			'show_date'        => self::parse_bool( $atts['show_date'] ),
			'show_excerpt'     => self::parse_bool( $atts['show_excerpt'] ),
			'show_categories'  => self::parse_bool( $atts['show_categories'] ),
			'heading'          => sanitize_text_field( $atts['heading'] ),
			'heading_tag'      => sanitize_key( $atts['heading_tag'] ),
			'list_style'       => sanitize_key( $atts['list_style'] ),
			'icon'             => sanitize_text_field( $atts['icon'] ),
			'container_style'  => sanitize_key( $atts['container_style'] ),
			'footer'           => wp_kses( $atts['footer'], spscase_get_allowed_footer_html() ),
			'no_posts_message' => sanitize_text_field( $atts['no_posts_message'] ),
			'excerpt_length'   => $defaults['excerpt_length'],
			'accent_color'     => $defaults['accent_color'],
		);

		// Get posts.
		$posts = SPSCASE_Query::get_posts( $settings );

		// Return empty if no posts (no wrapper).
		if ( empty( $posts ) ) {
			return '';
		}

		// Render output.
		return SPSCASE_Render::output( $posts, $settings );
	}

	/**
	 * Parse boolean value from string.
	 *
	 * @param mixed $value Value to parse.
	 * @return bool Parsed boolean.
	 */
	private static function parse_bool( $value ) {
		if ( is_bool( $value ) ) {
			return $value;
		}

		if ( is_string( $value ) ) {
			$value = strtolower( trim( $value ) );
			return in_array( $value, array( '1', 'true', 'yes', 'on' ), true );
		}

		return (bool) $value;
	}
}
